package org.lsst.ccs.subsystem.airwatch.main;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.time.Duration;
import java.time.Instant;
import java.util.HashMap;
import java.util.Map;
import org.apache.commons.mail.Email;
import org.apache.commons.mail.SimpleEmail;
import org.lsst.ccs.AlertService;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Sends email messages using some simple scheduling in order to avoid spamming.
 * @author tether
 */
public class EmailSender {
    private static final Logger log = Logger.getLogger(EmailSender.class.getName());

    private final LocalConfigService config;

    private final AlertService alertService;

    private final Map<String, Instant> sendingTimes;

    public EmailSender(final LocalConfigService config, final AlertService alertService) {
        this.config = config;
        this.alertService = alertService;
        sendingTimes = new HashMap<>();
    }

    private static final Duration NO_SPAM = Duration.ofMinutes(60);

    public void send(final String subject, final Throwable exc) {
        sendingTimes.putIfAbsent(subject, Instant.MIN);
        final Duration elapsed = Duration.between(sendingTimes.get(subject), Instant.now());
        try {
            final String[] recipients = config.getEmailRecipients().toArray(new String[]{});
            if ((recipients.length > 0) && (elapsed.compareTo(NO_SPAM) > 0)) {
                sendingTimes.put(subject, Instant.now());
                final Email emsg = new SimpleEmail();
                emsg.setHostName(config.getSMTPServer());
                emsg.setFrom(config.getEmailSender());
                emsg.setBounceAddress(config.getEmailBounceAddress());
                emsg.addReplyTo(config.getEmailBounceAddress());
                emsg.setSubject("[Airwatch] " + subject);
                final StringWriter sw = new StringWriter();
                final PrintWriter pw = new PrintWriter(sw);
                exc.printStackTrace(pw);
                emsg.setMsg(sw.toString());
                emsg.addTo(recipients);
                emsg.send();
            }
        }
        catch (org.apache.commons.mail.EmailException mailExc) {
            final String summary = "Error composing or sending email.";
            log.error(summary, mailExc);
            alertService.raiseAlert(
                    Alerts.emailDeliveryAlert(),
                    AlertState.WARNING,
                    summary);
        }
    }

}
