package org.lsst.ccs.subsystem.airwatch.main;

import org.lsst.ccs.drivers.commons.DriverException;

import org.lsst.ccs.HardwareException;

import org.lsst.ccs.utilities.logging.Logger;

import org.lsst.ccs.drivers.lighthouse.DataChannel;
import org.lsst.ccs.drivers.lighthouse.DataRecord;
import org.lsst.ccs.drivers.lighthouse.LighthouseClient;
import org.lsst.ccs.drivers.lighthouse.RecordFlag;

import java.io.Serializable;

import java.time.Duration;
import java.time.Instant;

import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * Implements readout of a Lighthouse Worldwide Solutions 3016-like instrument with a direct
 * serial or network connection. Each instance communicates with its instrument using
 * the ASCII version of the Modbus protocol. That in turn is built on top of the CCS ASCII driver
 * so the connection can be via serial port or network. It's assumed that the instrument
 * is putting records into its internal buffer. This code never removes records from the buffer,
 * only copies them, and so must keep track of the timestamp of the newest record ever read.
 * @author tether
 */
public class LighthouseModbus implements Instrument {
    private static final Logger log = Logger.getLogger("org.lsst.ccs.subsystem.airwatch");

    // Timeout for I/O operations.
    private static final Duration DEVICE_TIMEOUT = Duration.ofSeconds(10);

    // Let the driver log debug messages?
    private static final boolean DEVICE_DEBUG = false;

    // After we set the device clock from the system clock, when we read it back it
    // must be this close to a new system clock reading.
    private static final Duration DEVICE_CLOCK_MAX_ERROR = Duration.ofSeconds(10);

    private final int index;
    private final boolean enabled;
    private final String location;
    private final Instant lastDataTime;
    private final String connectionInfo;
    private final Optional<HardwareException> lastException;
    private final List<TrendableRecord> trendables;

    /**
     * Constructs a new instance from configuration data. The new instance will
     * be disabled for reading, have no stored data and have no stored exception.
     * @param config The subsystem's configuration information for the instrument.
     */
    public LighthouseModbus(InstrumentConfig config) {
        this.index = config.index;
        this.enabled = false;
        this.location = config.loc;
        this.lastDataTime = config.lastDataTime;
        this.connectionInfo = config.conn;
        this.lastException = Optional.empty();
        this.trendables = Collections.emptyList();
    }

    /**
     * Constructs a new instance using a value specified for each field.
     * @param index The instrument index (place in the list of instruments).
     * @param enabled The enabled-for-readout flag.
     * @param location The name of the location being monitored.
     * @param lastDataTime The next readout will get data newer than this.
     * @param connectionInfo The string of connection info from the configuration.
     * @param lastException The last exception thrown by an operation.
     * @param trendables The trendable items resulting from the last readout.
     */
    private LighthouseModbus(
        int index,
        boolean enabled,
        String location,
        Instant lastDataTime,
        String connectionInfo,
        Optional<HardwareException> lastException,
        List<TrendableRecord> trendables)

    {
        this.index = index;
        this.enabled = enabled;
        this.location = location;
        this.lastDataTime = lastDataTime;
        this.connectionInfo = connectionInfo;
        this.lastException = lastException;
        this.trendables = trendables;
    }

    /** {@inheritDoc } */
    @Override
    public Optional<HardwareException> getLastException() {
        return lastException;
    }

    /** {@inheritDoc } */
    @Override
    public int getIndex() {
        return index;
    }

    /** {@inheritDoc } */
    @Override
    public InstrumentStatus getStatus() {
        return new InstrumentStatus(index, enabled, location, lastDataTime);
    }

    /** {@inheritDoc } */
    @Override
    public Stream<TrendableRecord> getTrendables() {
        return trendables.stream();
    }

    /** {@inheritDoc } */
    @Override
    public Instrument disable() {
        return new LighthouseModbus(
            index,
            false,
            location,
            lastDataTime,
            connectionInfo,
            lastException,
            trendables);
    }

    // This implementation assumes that the subsystem configuration list of
    // known locations is stored in the instrument and in the same order.
    /** {@inheritDoc } */
    @Override
    public Instrument enable(LocationConfig loc) {
        boolean newEnabled = enabled;
        String newLocation = location;
        HardwareException hwexc = null;
        try (final ClientHolder holder = new ClientHolder(parseConnectionInfo(connectionInfo)))
        {
            final LighthouseClient client = holder.getClient();
            client.stopRecording();
            client.setClock(DEVICE_CLOCK_MAX_ERROR);
            client.setLocationIndex(loc.index + 1); // Instrument starts numbering at 1.
            final String locCheck = client.getLocationName();
            if (!locCheck.equalsIgnoreCase(loc.name)) {
                final String msg = String.format(
                    "Location table error for instrument at '%s'.%n" +
                        "Location #%d is '%s', should be '%s'.",
                    location, loc.index, locCheck, loc.name);
                hwexc = new HardwareException(true, msg);
            }
            for (DataChannel dchan: DataChannel.values()) {
                if (dchan.isAnalog()) continue; // Only particle channels have thresholds.
                // We rely on Lighthouse DataChannel names being identical to the
                // abstract InstrumentChannel names. 
                final InstrumentChannel ichan = InstrumentChannel.parse(dchan.getName()).get();
                if (loc.thresholds.containsKey(ichan)) {
                    // A threshold was specified for this channel. Set and enable the alarm.
                    final long thresh = loc.thresholds.get(ichan);
                    client.setAlarmThreshold(dchan, thresh);
                    client.setAlarmEnable(dchan, true);
                }
                else {
                    // No threshold specified for this channel. Disable the alarm.
                    client.setAlarmEnable(dchan, false);
                }
            }
            newEnabled = true;
            newLocation = loc.name;
        }
        catch (DriverException exc) {
            // Hardware exceptions we record.
            hwexc = new HardwareException(true, exc);
        }
        catch (Exception exc) {
            // Any other exception means buggy code.
            throw new Error(exc);
        }

        return new LighthouseModbus(
            index,
            newEnabled,
            newLocation,
            lastDataTime,
            connectionInfo,
            Optional.ofNullable(hwexc),
            Collections.emptyList());
    }

    /** {@inheritDoc } */
    @Override
    public Instrument read() {
        HardwareException hwexc = null;
        List<DataRecord> records = Collections.emptyList();
        List<TrendableRecord> newData = Collections.emptyList();
        Instant newLast = lastDataTime;
        Instrument ins = null;
        try (final ClientHolder holder = new ClientHolder(parseConnectionInfo(connectionInfo)))
        {
            final LighthouseClient client = holder.getClient();
            client.stopRecording();
            client.setClock(DEVICE_CLOCK_MAX_ERROR);
            records = client.getRecentRecords(lastDataTime);
        }
        catch (DriverException exc) {
            hwexc = new HardwareException(true, exc);
        }
        catch (Exception exc) {
            throw new Error(exc);
        }
        // We're using the most-recent-first mode of instrument readout, so the
        // timestamp on the first record is the last-data time we want.
        newData = extractTrendables(records.stream());
        if (newData.size() > 0) {
            newLast = newData.get(0).getMasterTimestamp();
        }
        ins = new LighthouseModbus(
            index,
            enabled,
            location,
            newLast,
            connectionInfo,
            Optional.ofNullable(hwexc),
            Collections.unmodifiableList(newData));
        return ins;
    }
    
    // Client holder for use with try-with-resources in order to ensure that
    // the instrument is left running and that the connection to it is
    // closed.
    private static class ClientHolder implements AutoCloseable {
        private final LighthouseClient client;
        public ClientHolder(LighthouseClient client) {this.client = client;}
        public LighthouseClient getClient() {return client;}
        @Override public void close() throws DriverException {
            client.startRecording();
            client.close();}
    }

    // Interpret the connection info string to yield a LighthouseClient object.
    private static LighthouseClient parseConnectionInfo(String info) throws DriverException {
        // Info will contain two words separated by whitespace.
        String[] words = info.trim().split("\\s+");
        final String method = words[0].toLowerCase();
        final String parameters = words[1];
        LighthouseClient client = null;
        if (method.equals("serial")) {
            // The second word is the name of the serial port.
            client = LighthouseClient.serialOpen(parameters, DEVICE_TIMEOUT, DEVICE_DEBUG);
        }
        else if (method.equals("ftdi")) {
            // The second word is of the form [host:]ftdiSerialNumber.
            client = LighthouseClient.ftdiOpen(parameters, DEVICE_TIMEOUT, DEVICE_DEBUG);
        }
        else if (method.equals("net")) {
            // The second word is of the form host:portNumber.
            // portNumber = unsigned decimal int.
            words = parameters.split(":");
            final String host = words[0];
            final int portNo = Integer.parseUnsignedInt(words[1]);
            client = LighthouseClient.netOpen(host, portNo, DEVICE_TIMEOUT, DEVICE_DEBUG);
        }
        else {
            throw new IllegalArgumentException(
                "Illegal Lighthouse connection type: " + method
            );
        }
        return client;
    }

    /**
     * Make trendable records from the instrument data records. Each instrument data
     * record results in a single {@code TrendableModbusRecord} whose master key
     * and timestamp are respectively the instrument location and timestamp. Each data
     * channel is made into a key-value pair of channel name and channel value. We take
     * advantage of the Lighthouse channel names being the same as the trending keys
     * defined in class {@code InstrumentChannel}. 
     * <p>
     * In addition to the channnel key-value pairs we add two boolean flags to each
     * trendable record with reserved key names of "limitViolation" and "malfunction".
     * The former is set to {@code true} only if the data record contained channels
     * whose values are outside of set limits. The latter is set to {@code true} only if
     * the data record contained instrument malfunction flags.
     * @param records The instrument data records.
     * @return The list of trendable records.
     */
    private static List<TrendableRecord> extractTrendables(
        Stream<DataRecord> records)
    {
        return
            records
            .map(rec -> {
                // Master key and timestamp.
                final String key = rec.getLocation().toLowerCase();
                final Instant stamp = rec.getStartTime();
                // Channel names-value pairs.
                final Map<String, Serializable> items =
                    rec.getChannels()
                    .collect(
                        Collectors.toMap(
                            datum -> datum.getChannel().getName(),
                            datum -> datum.getValue()
                        )
                    );
                // Limit violation flag. N.B.: "retainAll" would have
                // been better named "retainOnly".
                Set<RecordFlag> flags = EnumSet.copyOf(rec.getFlags());
                flags.retainAll(EnumSet.of(RecordFlag.HIGH, RecordFlag.LOW));
                items.put(TrendableModbusRecord.LIMIT_VIOLATION_KEY,
                          flags.isEmpty() ? Integer.valueOf(0) : Integer.valueOf(1));
                // Instrument malfunction flag.
                flags = EnumSet.copyOf(rec.getFlags());
                flags.retainAll(EnumSet.of(RecordFlag.FLOW_RATE, RecordFlag.LASER,
                RecordFlag.MALFUNCTION, RecordFlag.OVERFLOW, RecordFlag.SAMPLER));
                items.put(TrendableModbusRecord.MALFUNCTION_KEY, 
                          flags.isEmpty() ? Integer.valueOf(0) : Integer.valueOf(1));
                return new TrendableModbusRecord(key, stamp, items);
            })
            .collect(Collectors.toList());
    }
}
