package org.lsst.ccs.subsystem.airwatch.main;

import java.io.Serializable;

import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.TreeMap;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.lsst.ccs.HardwareException;

import org.lsst.ccs.drivers.opc.OPCItem;
import org.lsst.ccs.drivers.opc.OPCClient;

import org.lsst.ccs.drivers.commons.DriverException;

import org.lsst.ccs.utilities.logging.Logger;

import static org.lsst.ccs.subsystem.airwatch.main.InstrumentChannel.COUNTER_0_3;
import static org.lsst.ccs.subsystem.airwatch.main.InstrumentChannel.COUNTER_3_0;
import static org.lsst.ccs.subsystem.airwatch.main.InstrumentChannel.HUMIDITY;
import static org.lsst.ccs.subsystem.airwatch.main.InstrumentChannel.TEMPERATURE;

/**
 * Copies instrument readings published by Lighthouse's LMS Express RT program. That program
 * implements an OPC DA server holding only the latest reading from each device connected
 * to LMS Express RT.
 * @author tether
 */
public class LighthouseOPC implements Instrument {
    private static final Logger log = Logger.getLogger("org.lsst.ccs.subsystem.airwatch");

    // Timeout for I/O operations.
    private static final Duration DEVICE_TIMEOUT = Duration.ofSeconds(10);

    // Let the driver log debug messages?
    private static final boolean DEVICE_DEBUG = false;

    private final int index;
    private final boolean enabled;
    private final String location;
    private final Instant lastDataTime;
    private final String connectionInfo;
    private final Optional<HardwareException> lastException;
    private final List<TrendableRecord> trendables;
    private final ConfigurationService config;

    /**
     * Constructs a new instance from configuration data. The new instance will
     * be disabled for reading, have no stored data and have no stored exception.
     * @param iconf The subsystem's configuration information for the instrument.
     */
    public LighthouseOPC(InstrumentConfig iconf) {
        this.index = iconf.index;
        this.enabled = false;
        this.location = iconf.loc;
        this.lastDataTime = iconf.lastDataTime;
        this.connectionInfo = iconf.conn;
        this.lastException = Optional.empty();
        this.trendables = Collections.emptyList();
        this.config = iconf.service;
    }

    /**
     * Constructs a new instance using a value specified for each field.
     * @param index The instrument index (place in the list of instruments).
     * @param enabled The enabled-for-readout flag.
     * @param location The name of the location being monitored.
     * @param lastDataTime The most recent timestamp seen.
     * @param connectionInfo The string of connection info from the configuration.
     * @param lastException The last exception thrown by an operation.
     * @param trendables The trendable items resulting from the last readout.
     * @param config The configuration service ref.
     */
    private LighthouseOPC(
        int index,
        boolean enabled,
        String location,
        Instant lastDataTime,
        String connectionInfo,
        Optional<HardwareException> lastException,
        List<TrendableRecord> trendables,
        ConfigurationService config)

    {
        this.index = index;
        this.enabled = enabled;
        this.location = location;
        this.lastDataTime = lastDataTime;
        this.connectionInfo = connectionInfo;
        this.lastException = lastException;
        this.trendables = trendables;
        this.config = config;
    }

    /** {@inheritDoc } */
    @Override
    public Optional<HardwareException> getLastException() {
        return lastException;
    }

    /** {@inheritDoc } */
    @Override
    public int getIndex() {
        return index;
    }

    /** {@inheritDoc } */
    @Override
    public InstrumentStatus getStatus() {
        return new InstrumentStatus(index, enabled, location, lastDataTime);
    }

    /** {@inheritDoc } */
    @Override
    public Stream<TrendableRecord> getTrendables() {
        return trendables.stream();
    }

    /** {@inheritDoc } */
    @Override
    public Instrument disable() {
        return new LighthouseOPC(
            index,
            false,
            location,
            lastDataTime,
            connectionInfo,
            lastException,
            trendables,
            config);
    }

    /** {@inheritDoc } */
    @Override
    public Instrument enable(LocationConfig loc) {
        final boolean newEnabled = true;
        String newLocation = loc.name;
        return new LighthouseOPC(
            index,
            newEnabled,
            newLocation,
            lastDataTime,
            connectionInfo,
            Optional.empty(),
            Collections.emptyList(),
            config);
    }

    /** {@inheritDoc } */
    @Override
    public Instrument read() {
        HardwareException hwexc = null;
        List<TrendableRecord> newData = Collections.emptyList();
        Instant newLast = lastDataTime;
        Instrument ins;
        OPCClient client = null;
        try {
            client = parseConnectionInfo();
            final ReadResults res = getLatestReadings(client);
            client = res.client;
            newData = res.data;
            if (client.getException().isPresent()) {
               throw client.getException().get();
            }
        }
        catch (DriverException exc) {
            hwexc = new HardwareException(true, exc);
        }
        catch (Throwable exc) {
            throw new Error(exc);
        }
        finally {
            if (client != null) {client.close();}
        }

        // If the read was successful, update read time.
        if (newData.size() > 0) {
            newLast = Instant.now();
        }
        ins = new LighthouseOPC(
            index,
            enabled,
            location,
            newLast,
            connectionInfo,
            Optional.ofNullable(hwexc),
            Collections.unmodifiableList(newData),
            config);
        return ins;
    }

    // Interpret the connection info string to yield an OPCClient object.
    private OPCClient parseConnectionInfo() throws DriverException {
        // The connection info is the name of a CCS configuration item. That item
        // is an array of strings containing this information:
        // (0) The host name or IPv4 address for the server.
        // (1) The Windows domain of the local account used to log in.
        // (2) The class ID (CLSID) of the OPC server.
        // (3) The username of the Windows local account used to log in.
        // (4) The password of the Windows local account used to log in.
        // In the usual case all the "instruments" will actually be on the same server.
        final List<String> srvInf = config.getOPCServerInfo(connectionInfo);
        if (srvInf.size() < 5) {
            throw new DriverException("Login information for "
                                      + connectionInfo + " is missing or incomplete.");
        }
        return new OPCClient(srvInf.get(0), srvInf.get(1), srvInf.get(2),
            srvInf.get(3), srvInf.get(4));
    }

    private ReadResults getLatestReadings(final OPCClient oldClient)
    throws Throwable
    {
        OPCClient client = oldClient;
        // Construct the OPC tag prefix for this location.
        final String prefix = "LMS/Inputs/" + location + "/";
        // Construct the full tags.
        final String loc = prefix + "Short Location Name";

        final String abs30   = prefix + "3.0/Aggregates/Cumulative Counts/Data";
        final String norm30  = prefix + "3.0/Aggregates/Cumulative Normalized Counts/Data";
        final String fault30 = prefix + "3.0/Aggregates/Cumulative Counts/Status/Fault/Triggered";
        final String alarm30 = prefix + "3.0/Aggregates/Cumulative Counts/Status/Alarm/Triggered";

        final String abs03   = prefix + "0.3/Aggregates/Cumulative Counts/Data";
        final String norm03  = prefix + "0.3/Aggregates/Cumulative Normalized Counts/Data";
        final String fault03 = prefix + "0.3/Aggregates/Cumulative Counts/Status/Fault/Triggered";
        final String alarm03 = prefix + "0.3/Aggregates/Cumulative Counts/Status/Alarm/Triggered";

        final String humidity    = prefix + "Relative Humidity/Data";
        final String faultHumid  = prefix + "Relative Humidity/Status/Fault/Triggered";
        final String alarmHumid  = prefix + "Relative Humidity/Status/Alarm/Triggered";

        final String temperature = prefix + "Temperature/Data";
        final String faultTemp   = prefix + "Temperature/Status/Fault/Triggered";
        final String alarmTemp   = prefix + "Temperature/Status/Alarm/Triggered";

        final List<String> tags =
            Arrays.asList(new String[]{loc,
                                       abs30, norm30, fault30, alarm30,
                                       abs03, norm03, fault03, alarm03,
                                       humidity, faultHumid, alarmHumid,
                                       temperature, faultTemp, alarmTemp});
        // Make a group from the tags and read the group. It's unfortunate but
        // each addition to the group sends a validity-check request to
        // the server. The same behavior prevents us from making a group
        // in advance without a server connection. For only a few channels
        // read out at long intervals the overhead is tolerable.
        client =
            client
                .setGroup(tags)
                .readGroup(true);

        // Convert the OPCItems into TrendableRecords. For this we'll need to replace
        // the server path name of each data item with the short name used to
        // make the trending key.
        // There is a separate item produced for each tag in the group. Note that the
        // short location name has its own item, as do the value and the flags for each
        // channel.
        final Map<String, OPCItem> serverItems =
            client.getData()
            .stream()
            .collect(Collectors.toMap(x -> x.getTag(), x -> x));
        // Get the short location name that was read.
        final String shortLocName = (String)serverItems.get(loc).getValue();
        // For each channel assemble one TrendableRecord from the short location name
        // and and the items read for the channel.
        final List<TrendableRecord> result = new ArrayList<>();
        result.add(makeCounterRecord(serverItems, abs30, norm30, fault30, alarm30,
                                     COUNTER_3_0, shortLocName));
        result.add(makeCounterRecord(serverItems, abs03, norm03, fault03, alarm03,
                                     COUNTER_0_3, shortLocName));
        result.add(makeAnalogRecord(serverItems, humidity, faultHumid, alarmHumid,
                                     HUMIDITY, shortLocName));
        result.add(makeAnalogRecord(serverItems, temperature, faultTemp, alarmTemp,
                                     TEMPERATURE, shortLocName));
        return new ReadResults(client, result);
    }

    private static class ReadResults {
        public final OPCClient client;
        public final List<TrendableRecord> data;
        ReadResults(final OPCClient client, final List<TrendableRecord> data) {
            this.client = client;
            this.data = data;
        }
    }

    private TrendableRecord makeCounterRecord(
        final Map<String, OPCItem> serverItems,
        final String absoluteKey,
        final String densityKey,
        final String faultKey,
        final String alarmKey,
        final InstrumentChannel channel,
        final String shortLocName)
    {
        final Map<String, Serializable> trendingItems = new TreeMap<>();
        // Although the count, alarm flag and malfunction flag each have their
        // own timestamp, they should all be about the same so we'll use the
        // timestamp of the count for all of them. In the trending display
        // of the CCS graphical console we want each particle size to show
        // as a subtree item with count and flags as leaf items underneath.
        // To do that we associate the channel-name key with an object
        // containing the items s read for that channel.
        trendingItems.put(channel.getKey(), // E.g., "0.3", "3.0"
            new CounterPoint(
                serverItems.get(absoluteKey), // particles in sample
                serverItems.get(densityKey),  // particles per unit volume
                serverItems.get(alarmKey),    // limitViolation flag
                serverItems.get(faultKey)     // malfunction flag
            )
        );
        return new TrendableOPCRecord(
            shortLocName,
            serverItems.get(absoluteKey).getTimestamp(), // time of channel count.
            trendingItems
        );
    }

    // A "point" is all the tag-values read for one channel at a given time.
    public static class CounterPoint implements Serializable {
        private static final long serialVersionUID = 1_0_3;
        public final Double  absolute;
        public final Double  density;
        public final Integer limitViolation;
        public final Integer malfunction;
        CounterPoint(
            final OPCItem absolute,
            final OPCItem density,
            final OPCItem viol,
            final OPCItem malf)
        {
            this.absolute = (Double)absolute.getValue();
            this.density = (Double)density.getValue();
            // At the time this was written, boolean values could not go into the trending DB.
            final Boolean v = (Boolean)viol.getValue();
            this.limitViolation = v ? 1 : 0;
            final Boolean m = (Boolean)malf.getValue();
            this.malfunction = m ? 1 : 0;
        }
        @Override
        public String toString() {
            return String.format(
                "[absolute: %s, density: %s,limitViolation: %s, malfunction: %s]",
                absolute, density, limitViolation, malfunction);
        }
    }

    // See the similar code for counter channels above.
    private TrendableRecord makeAnalogRecord(
        final Map<String, OPCItem> serverItems,
        final String valueKey,
        final String faultKey,
        final String alarmKey,
        final InstrumentChannel channel,
        final String shortLocName)
    {
        final Map<String, Serializable> trendingItems = new TreeMap<>();
        trendingItems.put(channel.getKey(),
            new AnalogPoint(
                serverItems.get(valueKey),
                serverItems.get(faultKey),
                serverItems.get(alarmKey)
            ));
        return new TrendableOPCRecord(
            shortLocName,
            serverItems.get(valueKey).getTimestamp(),
            trendingItems
        );
    }

    public static class AnalogPoint implements Serializable {
        private static final long serialVersionUID = 1_0_3;
        public final Double  value;
        public final Integer limitViolation;
        public final Integer malfunction;
        AnalogPoint(
            final OPCItem value,
            final OPCItem viol,
            final OPCItem malf)
        {
            this.value = (Double)value.getValue();
            // At the time this was written, boolean values could not go into the trending DB.
            final Boolean v = (Boolean)viol.getValue();
            this.limitViolation = v ? 1 : 0;
            final Boolean m = (Boolean)malf.getValue();
            this.malfunction = m ? 1 : 0;
        }
        @Override
        public String toString() {
            return String.format(
                "[value: %s, limitViolation: %s, malfunction: %s]",
                value, limitViolation, malfunction);
        }
    }
}
