package org.lsst.ccs.subsystem.airwatch.main;

import java.io.Serializable;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Level;
import java.util.stream.Collectors;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import static org.lsst.ccs.command.annotations.Command.CommandType.ACTION;
import static org.lsst.ccs.command.annotations.Command.CommandType.QUERY;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HardwareController;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.framework.Signal;
import org.lsst.ccs.framework.SignalHandler;
import org.lsst.ccs.framework.SignalLevel;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.utilities.logging.Logger;
import org.lsst.ccs.utilities.scheduler.PeriodicTask;

/** The main module for the Airwatch subsystem. This module is responsible for:
 * <ul>
 * <li>Establishing initial contact with the configured instruments.</li>
 * <li>Providing and managing the periodic readout task.</li>
 * <li>Alert management.</li>
 * <li>Instrument management and status reporting.</li>
 * </ul>
 *  * @author tether
 */
public final class AirwatchMain
extends Subsystem
implements ClearAlertHandler, HardwareController, HasLifecycle, SignalHandler
{
/* This module maintains a list of immutable Instrument objects which describe the state
 * of each instrument and specify the allowed operations. Mutating operations therefore follow
 * the read-modify-write pattern. A CopyOnWriteArrayList is enough to synchronize query
 * commands with mutating action commands and with the readout task but it isn't enough to
 * synchronize the action commands with the periodic readout.
 *
 * Take for example the case of disabling readout for a previously enabled instrument with
 * no prior readout data. The disable() command copies the instrument state and
 * produces a new state with the enabled flag set to false. While that's happening
 * the readout task might copy the same old state and produce a new state with new readout data.
 * Depending on who writes back first either the enable flag will remain true or the
 * new readout data will become inaccessible.
 *
 * For this reason there's an extra lock used to synchronize the readout task with mutating
 * action commands. The readout task will hold the lock while it reads instruments and modifies
 * the state list. The mutating action commands must take the lock before doing anything else,
 * delaying the readout task until they finish their execution, which should be brief.
 * They in turn release the lock after completing their changes.
 *
 * The periodic readout task normally works its way down the instrument list, reading any data
 * accumulated since the last-data-time for each instrument and publishing that data in the
 * trending database. As it goes the task modifies the list of Instrument objects to
 * reflect the new state of data collection for each instrument.
 *
 * The readout task checks for interruption after performing read-and-publish for each
 * instrument. This is needed to ensure that the instrument's last-data-time is consistent with
 * what has been posted for trending. An interruption will cause the Instrument
 * objects that have not been processed to be left unchanged in the instrument list.
 *
 */

    // How long in ms callIfNotBusy() will wait for the readout lock.
    private static final int BUSY_TIMEOUT = 120_000;

    ////////// Commands //////////

    /**
     * Gives the status of each of the instruments. Includes instrument number, location,
     * enable flag and the timestamp of the last record read.
     * @return A string with a heading line followed by one line per instrument.
     */
    @Command(type=QUERY, description="Displays a short report on instrument status.", level=Command.NORMAL)
    public String instruments() {return new InstrumentReport(instruments).toString();}

    /**
     * Prevents an instrument from being read until further notice.
     * @param index An instrument number.
     * @return "OK", or "BUSY" if a readout is in progress.
     * @throws IndexOutOfBoundsException if {@code index} is invalid.
     * @throws HardwareException if device operations fail.
     * @throws Exception if {@code ident} is invalid or ambiguous.
     * @see #enable(java.lang.String, java.lang.String)
     */
    @Command(type=ACTION, description="Prevents further readout of an instrument", level=Command.ENGINEERING1,
        timeout=BUSY_TIMEOUT+100)
    public String disable(
        @Argument(description="Instrument index number.")
        final int index
    ) throws Exception
    {
        return callIfNotBusy(() -> {
            final Instrument ins = instruments.get(index);
            instruments.set(ins.getIndex(), ins.disable());
            return "OK";
        });
    }

    /**
     * Enables periodic readout for an instrument.
     * @param index An instrument number.
     * @return "OK", or "BUSY" if a readout is in progress.
     * @throws IndexOutOfBoundsException if {@code index} is invalid.
     * @throws HardwareException if device operations fail.
     * @see #disable(java.lang.String)
     */
    @Command(type = ACTION, description="Enables an instrument for readout.", level=Command.ENGINEERING1,
         timeout=BUSY_TIMEOUT + 1000)
    public String enable(
        @Argument(description="Instrument index number.")
        final int index
    ) throws Exception
    {
        return callIfNotBusy( () -> {
            final Instrument ins = instruments.get(index);
            final Instrument newIns = ins.enable();
            instruments.set(ins.getIndex(), newIns);
            if (newIns.getLastException().isPresent()) {throw newIns.getLastException().get();}
            return "OK";
        });
    }

    /**
     * Signal handler for stop and abort.
     * @param sig The signal.
     * @return TreeWalkerDiag.GO.
     */
    @Override
    public TreeWalkerDiag signal(Signal sig) {
        if (sig.getLevel() == SignalLevel.STOP) {
            readoutTask.cancel(false); // Don't interrupt any current readout.
        }
        else if (sig.getLevel() == SignalLevel.HALT) {
            readoutTask.cancel(true); // Interrupt any current readout.
        }
        return TreeWalkerDiag.GO;
    }


    ////////// References to other subsystem components //////////

    @LookupField(strategy=Strategy.CHILDREN)
    private volatile LocalConfigService config;

    @LookupField(strategy = Strategy.TREE)
    private volatile AlertService alertService;


    ////////// Subsystem state data //////////

    /**
     * The descriptions of all instruments being managed. Updated by ACTION
     * commands and read by the readout task and by QUERY commands.
     */
    private final CopyOnWriteArrayList<Instrument> instruments;

    /**
     * Synchronizes action commands that mutate {@link #instruments} with
     * the periodic readout task.
     */
    private final Lock readoutLock;

    /**
     * Regularly scheduled to read the instruments.
     */
    private PeriodicTask readoutTask;

    /**
     * The subsystem's logger.
     */
    private final Logger log;

    /**
     * Sends emails with simple spam throttling.
     */
    private EmailSender mailSender;


    ////////// Lifecycle methods //////////
    // Arranged in chronological order of calling.

    /**
     * Constructor.
     */
    public AirwatchMain() {
        // This is a worker subsystem.
        super("airwatch", AgentInfo.AgentType.WORKER);

        // A component's proper name need not be unique, just its full path name.
        // Affects the name of configuration parameters and CLI commands.
        this.getAgentInfo().getAgentProperties().setProperty("org.lsst.ccs.use.full.paths", "true");

        config = null;
        alertService = null;
        instruments = new CopyOnWriteArrayList<>();
        log = Logger.getLogger(AirwatchMain.class.getName());
        readoutTask = null;
        readoutLock = new ReentrantLock();
    }

    /**
     * Checks whether all the subsystem components have been constructed.
     * Called just after the subsystem component tree has been built.
     * @throws Error if anything is missing.
     */
    @Override
    public void init() {
        final List<String> missing = new ArrayList<>();
        if (config == null) {
            missing.add("LocalConfigService component");
        }
        if (alertService == null) {
            missing.add("CCS alert service");
        }
        if (!missing.isEmpty()) {
            throw new RuntimeException("Can't find " + String.join(", ", missing));
        }
    }

    /**
     * Attempts to start all configured instruments that are operational.
     * Called in the middle of the initialization phase after {@code start()} has been called
     * for all configurable components. All instruments are stopped and re-started
     * with correct configurations for their assigned locations.
     * @throws HardwareException if any fail to respond or if any report hardware faults.
     * @return TreeWalkerDiag.GO.
     */
    @Override
    public TreeWalkerDiag checkHardware() throws HardwareException {
        config.makeConfigurationObjects();
        readConfiguration();
        scheduleFirstReadout(Duration.ZERO);
        return TreeWalkerDiag.GO;
    }

    /**
     * Creates the email sender. Called at the start of the initialization phase.
     */
    @Override
    public void start() {
        mailSender = new EmailSender(config, alertService);
    }

    private void readConfiguration() {
        final List<InstrumentConfig> configs = config.getInstrumentConfigs();
        final List<Instrument> insts
            = configs
            .stream()
            .map(cfg -> cfg.type.make(cfg))
            .collect(Collectors.toList());
        instruments.addAll(insts);
    }

    private void scheduleFirstReadout(Duration initialDelay) {
        this.readoutTask =
            this
                .getScheduler()
                .scheduleAtFixedRate(this::readoutTaskBody,
                    initialDelay.getSeconds(),
                    config.getReadoutInterval().getSeconds(),
                    TimeUnit.SECONDS);
    }

    private void readoutTaskBody() {
        //  Synchronize with mutating action commands.
        try {
            readoutLock.lockInterruptibly();
        }
        catch (InterruptedException exc) {
            Thread.currentThread().interrupt();
            return;
        }

        // Read, update status, post trendables.
        try {
            instruments
                .stream()
                .filter(ins -> ins.getStatus().enabled)
                .forEach(ins -> {
                    if (Thread.currentThread().isInterrupted()) {
                        throw new TerminationException();
                    }
                    readAndUpdate(ins);
                });
            publishDewPoint();
        } catch (TerminationException exc) {
            log.warning("Instrument readout was interrupted!");
        }
        finally {
            readoutLock.unlock();
        }
    }

    private static class TerminationException extends RuntimeException {}

    private void readAndUpdate(final Instrument ins) {
        final Instrument newIns = ins.read();
        instruments.set(ins.getIndex(), newIns);
        checkForExceptionDuringReadout(newIns);
        // Send trendables to trending.
        newIns
            .getTrendables()
            .forEach(msg -> msg.post(this));
    }

    private boolean checkForExceptionDuringReadout(final Instrument newIns) {
        // Post an alert and send email if readout was terminated by an exception.
        newIns.getLastException().ifPresent(exc -> {
            final String badloc = newIns.getStatus().location;
            final String cause = badloc + ": " + exc.getMessage();
            alertService.raiseAlert(
                Alerts.instrumentIOAlert(badloc),
                AlertState.WARNING,
                cause);
            mailSender.send(cause, exc);
        });
        return newIns.getLastException().isPresent();
    }

    /**
     * Does nothing.  Called at the end of the initialization phase
     * after {@code start()} has been called for all configurable components and after
     * {@code checkHardware()} and {@code checkStarted()} have been called for all
     * hardware controllers.
     */
    @Override
    public void postStart() {}

    /**
     * Cancels periodic readout if this hasn't already been done (part of shutdown).
     * The instruments themselves are not stopped and continue to collect data.
     * Called at the beginning of the shutdown phase.
     * @throws HardwareException if periodic readout is still scheduled.
     * @see #signal(org.lsst.ccs.framework.Signal)
     */
    @Override
    public void checkStopped() throws HardwareException {
        readoutTask.cancel(false);
        if (!readoutTask.isCancelled()) {
            throw new HardwareException(false, "Could not cancel the readout task.");
        }
    }

    /**
     * Handles all alerts by allowing them to be cleared.
     * @param alert The alert to be cleared.
     * @param state The current state (severity) of the alert.
     * @return {@code CLEAR_ALERT}.
     */
    @Override
    public ClearAlertHandler.ClearAlertCode canClearAlert(final Alert alert, final AlertState state) {
        return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
    }

    ////////// Utilities //////////

    /**
     * Gets a list of all instruments with short names in the given collection.
     * @param shorts The collection of short names to look for.
     * @return The list of matching instruments, possibly empty.
     */
    private List<Instrument> findInstrumentsByShortName(Collection<String> shorts) {
    return instruments
        .stream()
        .filter((Instrument ins) -> {
            final InstrumentStatus status = ins.getStatus();
            final Boolean found = shorts.contains(status.shortName);
            return found;
        })
        .collect(Collectors.toList());
    }

    /**
     * Calls a {@code Callable<String>} on behalf of a command that needs to synchronize
     * with the periodic readout. Upon exit the current thread will
     * not hold the readout lock.
     * @param cmd The code to run.
     * @return The result from {@code cmd}.
     * @throws Exception if {@code cmd.call()} throws.
     */
    private String callIfNotBusy(Callable<String> cmd) throws Exception {
        if (!readoutLock.tryLock(BUSY_TIMEOUT, TimeUnit.MILLISECONDS)) {
            // Send rejection of the command.
            return "BUSY";
        }
        try {
            // Execute the command.
            return cmd.call();
        } finally {
            readoutLock.unlock();
        }
    }
    
    /**
     * From the given instrument gets all trendable records that contain
     * either temperature and humidity.
     * @param ins The instrument.
     * @return The list of trendables, which will be empty if the instrument
     * is not currently enabled for readout.
     */
    private List<TrendableRecord> getDewPointData(final Instrument ins) {
        // Instrument must be enabled.
        if (!ins.getStatus().enabled) {
            return Collections.emptyList();
        }
        return ins
            .getTrendables()
            .filter( (TrendableRecord tr) -> {
                final Map<String, Serializable> items = tr.getItems();
                return items.containsKey(InstrumentChannel.TEMPERATURE.getKey())
                    || items.containsKey(InstrumentChannel.HUMIDITY.getKey());
             })
            .collect(Collectors.toList());
    }
    
    public static final String N_DEW_TEMPS_KEY = "calculated/nDewTemps";
    public static final String N_DEW_HUMIDS_KEY = "calculated/nDewHumids";
    public static final String DEW_POINT_KEY = "calculated/dewPoint";

    private void publishDewPoint() {
        // - Collect all the trendable records with temperature and/or humidity data
        //   into a list "readings".
        final List<TrendableRecord> readings = new ArrayList<>();
        for (final Instrument ins: findInstrumentsByShortName(config.getDewShortNames())) {
            readings.addAll(getDewPointData(ins));
        }
        // Calculate the mean temperature T and mean humidity RH.
        final List<Double> temps = readings.stream()
            .map(tr -> (AnalogPoint)tr.getItems().get(InstrumentChannel.TEMPERATURE.getKey()))
            .filter(ap -> ap != null)
            .map(ap -> ap.value)
            .collect(Collectors.toList());
        final List<Double> humids = readings.stream()
            .map(tr -> (AnalogPoint)tr.getItems().get(InstrumentChannel.HUMIDITY.getKey()))
            .filter(ap -> ap != null)
            .map(ap -> ap.value)
            .collect(Collectors.toList());
        if (temps.isEmpty() || humids.isEmpty()) {
            log.log(Level.WARNING, "Cannot calculate dew point: nDewTemps = {0}, nDewHumids = {1}.",
                    new Object[]{temps.size(), humids.size()});
            return;
        }
        final double tempMean = temps.stream().mapToDouble(Double::doubleValue).average().getAsDouble();
        final double humidMean = humids.stream().mapToDouble(Double::doubleValue).average().getAsDouble();
        // - Calculate Tdp from T and RH.
        final double tdp = Utility.dewPoint(tempMean, humidMean);
        // - Create and publish a new KVD list with the current time as the master timestamp
        //   and KV pairs for no. of temps, no.of humids and dew point.
        final KeyValueDataList calc = new KeyValueDataList();
        calc.addData(N_DEW_TEMPS_KEY, temps.size());
        calc.addData(N_DEW_HUMIDS_KEY, humids.size());
        calc.addData(DEW_POINT_KEY, tdp);
        // - Publish K on the status bus.
        this.publishSubsystemDataOnStatusBus(calc);
    }
}
