package org.lsst.ccs.subsystem.airwatch.main;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import static org.lsst.ccs.command.annotations.Command.CommandType.ACTION;
import static org.lsst.ccs.command.annotations.Command.CommandType.QUERY;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HardwareController;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.framework.Signal;
import org.lsst.ccs.framework.SignalHandler;
import org.lsst.ccs.framework.SignalLevel;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.utilities.logging.Logger;
import org.lsst.ccs.utilities.scheduler.PeriodicTask;

/** The main module for the Airwatch subsystem. This module is responsible for:
 * <ul>
 * <li>Establishing initial contact with the configured instruments.</li>
 * <li>Providing and managing the periodic readout task.</li>
 * <li>Alert management.</li>
 * <li>Instrument management and status reporting.</li>
 * </ul>
 *  * @author tether
 */
public final class AirwatchMain
implements ClearAlertHandler, HardwareController, HasLifecycle, SignalHandler
{
/* This module maintains a list of immutable Instrument objects which describe the state
 * of each instrument and specify the allowed operations. Mutating operations therefore follow
 * the read-modify-write pattern. A CopyOnWriteArrayList is enough to synchronize query
 * commands with mutating action commands and with the readout task but it isn't enough to
 * synchronize the action commands with the periodic readout.
 *
 * Take for example the case of disabling readout for a previously enabled instrument with
 * no prior readout data. The disable() command copies the instrument state and
 * produces a new state with the enabled flag set to false. While that's happening
 * the readout task might copy the same old state and produce a new state with new readout data.
 * Depending on who writes back first either the enable flag will remain true or the
 * new readout data will become inaccessible.
 *
 * For this reason there's an extra lock used to synchronize the readout task with mutating
 * action commands. The readout task will hold the lock while it reads instruments and modifies
 * the state list. The mutating action commands must take the lock before doing anything else,
 * delaying the readout task until they finish their execution, which should be brief.
 * They in turn release the lock after completing their changes.
 *
 * The periodic readout task normally works its way down the instrument list, reading any data
 * accumulated since the last-data-time for each instrument and publishing that data in the
 * trending database. As it goes the task modifies the list of Instrument objects to
 * reflect the new state of data collection for each instrument.
 *
 * The readout task checks for interruption after performing read-and-publish for each
 * instrument. This is needed to ensure that the instrument's last-data-time is consistent with
 * what has been posted for trending. An interruption will cause the Instrument
 * objects that have not been processed to be left unchanged in the instrument list.
 *
 */

    ////////// Commands //////////

    /**
     * Gives the status of each of the instruments. Includes instrument number, location,
     * enable flag and the timestamp of the last record read.
     * @return A string with a heading line followed by one line per instrument.
     */
    @Command(type=QUERY, description="Displays a short report on instrument status.")
    public String instruments() {return new InstrumentReport(instruments).toString();}

    /**
     * Prevents an instrument from being read until further notice.
     * @param ident Either an instrument number or a location name.
     * @return "OK", or "BUSY" if a readout is in progress.
     * @throws Exception if {@code ident} is invalid or ambiguous.
     * @see #enable(java.lang.String, java.lang.String)
     */
    @Command(type=ACTION, description="Prevents further readout of an instrument")
    public String disable(
        @Argument(description="Instrument index number or location.")
        final String ident
    ) throws Exception
    {
        return callIfNotBusy(() -> {
            final List<Instrument> insList = findInstruments(ident);
            if (insList.size() != 1) {
                throw new IllegalArgumentException(("There are " + insList.size()) + " instruments at " + ident);
            }
            final Instrument ins = insList.get(0);
            instruments.set(ins.getIndex(), ins.disable());
            return "OK";
        });
    }

    /**
     * Sets a new location for an instrument then enables periodic readout for it.
     * @param ident Either an instrument number or a location name.
     * @param newLocation The name of the new location to be monitored.
     * @return "OK", or "BUSY" if a readout is in progress.
     * @throws IllegalArgumentException if {@code ident} is invalid or ambiguous.
     * @throws IllegalArgumentException if {@code newLocation} is invalid.
     * @throws HardwareException if device operations fail.
     * @see #disable(java.lang.String)
     */
    @Command(type = ACTION, description="Sets new location for and enables instrument.")
    public String enable(
        @Argument(description="Instrument index number or current location.")
        final String ident,
        @Argument(description="New location.")
        final String newLocation
    ) throws Exception
    {
        return callIfNotBusy( () -> {
            final List<Instrument> insList = findInstruments(ident);
            if (insList.size() != 1) {
                throw new IllegalArgumentException(
                    ("There are " + insList.size()) +
                    " instruments at " + ident);
            }
            String loc = null;
            if (locations.containsKey(newLocation)) {
                loc = newLocation;
            }
            else if (locations.containsKey(newLocation.toLowerCase())) {
                loc = newLocation.toLowerCase();
            }
            if (loc == null) {
                throw new IllegalArgumentException("Unknown location " + newLocation);
            }
            final Instrument ins = insList.get(0);
            final Instrument newIns = ins.enable(locations.get(loc));
            instruments.set(ins.getIndex(), newIns);
            if (newIns.getLastException().isPresent()) {throw newIns.getLastException().get();}
            return "OK";
        });
    }

    /**
     * Signal handler for stop and abort.
     * @param sig The signal.
     * @return TreeWalkerDiag.GO.
     */
    @Override
    public TreeWalkerDiag signal(Signal sig) {
        if (sig.getLevel() == SignalLevel.STOP) {
            readoutTask.cancel(false); // Don't interrupt any current readout.
        }
        else if (sig.getLevel() == SignalLevel.HALT) {
            readoutTask.cancel(true); // Interrupt any current readout.
        }
        return TreeWalkerDiag.GO;
    }


    ////////// References to other subsystem components //////////

    @LookupField(strategy=Strategy.CHILDREN)
    private volatile LocalConfigService config;

    @LookupField(strategy = Strategy.TREE)
    private volatile AlertService alertService;

    @LookupField(strategy = Strategy.TREE)
    private volatile Subsystem subsys;


    ////////// Subsystem state data //////////

    /**
     * The descriptions of all known locations, keyed by location name.
     */
    private Map<String, LocationConfig> locations;

    /**
     * The descriptions of all instruments being managed. Updated by ACTION
     * commands and read by the readout task and by QUERY commands.
     */
    private final CopyOnWriteArrayList<Instrument> instruments;

    /**
     * Synchronizes action commands that mutate {@link #instruments} with
     * the periodic readout task.
     */
    private final Lock readoutLock;

    /**
     * Regularly scheduled to read the instruments.
     */
    private PeriodicTask readoutTask;

    /**
     * The subsystem's logger.
     */
    private final Logger log;

    /**
     * Sends emails with simple spam throttling.
     */
    private EmailSender mailSender;


    ////////// Lifecycle methods //////////
    // Arranged in chronological order of calling.

    /**
     * Constructor.
     */
    public AirwatchMain() {
        config = null;
        alertService = null;
        locations = null;
        instruments = new CopyOnWriteArrayList<>();
        log = Logger.getLogger("org.lsst.ccs.subsystem.airwatch");
        readoutTask = null;
        readoutLock = new ReentrantLock();
    }

    /**
     * Checks whether all the subsystem components have been constructed.
     * Called just after the subsystem component tree has been built.
     * @throws Error if anything is missing.
     */
    @Override
    public void init() {
        final List<String> missing = new ArrayList<>();
        if (config == null) {
            missing.add("LocalConfigService component");
        }
        if (alertService == null) {
            missing.add("CCS alert service");
        }
        if (!missing.isEmpty()) {
            throw new RuntimeException("Can't find " + String.join(", ", missing));
        }
    }

    /**
     * Attempts to start all configured instruments.
     * Called in the middle of the initialization phase after {@code start()} has been called
     * for all configurable components. All instruments are stopped and re-started
     * with correct configurations for their assigned locations.
     * @throws HardwareException if any fail to respond or if any report hardware faults.
     * @return TreeWalkerDiag.GO.
     */
    @Override
    public TreeWalkerDiag checkHardware() throws HardwareException {
        config.makeConfigurationObjects();
        readConfiguration();
        enableAllInstruments();
        scheduleFirstReadout(Duration.ZERO);
        return TreeWalkerDiag.GO;
    }

    /**
     * Creates the email sender. Called at the start of the initialization phase.
     */
    @Override
    public void start() {
        mailSender = new EmailSender(config, alertService);
    }

    private void readConfiguration() {
        locations = Collections.unmodifiableMap(config.getLocationConfigs()
            .stream()
            .collect(Collectors.toMap(loc -> loc.name.toLowerCase(), Function.identity()))
        );
        final List<InstrumentConfig> configs = config.getInstrumentConfigs();
        final List<Instrument> insts
            = configs
            .stream()
            .map(cfg -> cfg.type.make(cfg))
            .collect(Collectors.toList());
        instruments.addAll(insts);
    }

    private void enableAllInstruments() throws HardwareException {
        final List<HardwareException> exceptions = new ArrayList<>();
        instruments.forEach(inst -> {
            final LocationConfig loc = locations.get(inst.getStatus().location.toLowerCase());
            final Instrument newIns = inst.enable(loc);
                instruments.set(inst.getIndex(), newIns);
                newIns.getLastException().ifPresent(exc -> exceptions.add(exc));
        });
        // Make a chain of all the hardware exceptions and throw the top of the chain.
        if (exceptions.size() > 0) {
            HardwareException top = exceptions.remove(0);
            for (HardwareException exc : exceptions) {top = new HardwareException(exc);}
            throw top;
        }
    }

    private void scheduleFirstReadout(Duration initialDelay) {
        this.readoutTask =
            subsys
                .getScheduler()
                .scheduleAtFixedRate(this::readoutTaskBody,
                    initialDelay.getSeconds(),
                    config.getReadoutInterval().getSeconds(),
                    TimeUnit.SECONDS);
    }

    private void readoutTaskBody() {
        //  Synchronize with mutating action commands.
        try {
            this.readoutLock.lockInterruptibly();
        }
        catch (InterruptedException exc) {
            Thread.currentThread().interrupt();
            return;
        }

        // Read, update status, post trendables.
        try {
            instruments
                .stream()
                .filter(ins -> ins.getStatus().enabled)
                .forEach(ins -> {
                    if (Thread.currentThread().isInterrupted()) {
                        throw new TerminationException();
                    }
                    readAndUpdate(ins);
                });
        } catch (TerminationException exc) {
            log.warning("Instrument readout was interrupted!");
        }
        finally {
            readoutLock.unlock();
        }
    }

    private static class TerminationException extends RuntimeException {}

    private void readAndUpdate(final Instrument ins) {
        final Instrument newIns = ins.read();
        instruments.set(ins.getIndex(), newIns);
        checkForExceptionDuringReadout(newIns);
        // Send trendables to trending.
        newIns
            .getTrendables()
            .forEach(msg -> msg.post(subsys));
        // Update instrument status in configuration localConfig.
        config.updateInstrument(newIns.getStatus());
    }

    private boolean checkForExceptionDuringReadout(final Instrument newIns) {
        // Post an alert and send email if readout was terminated by an exception.
        newIns.getLastException().ifPresent(exc -> {
            final String badloc = newIns.getStatus().location;
            final String cause = badloc + ": " + exc.getMessage();
            alertService.raiseAlert(
                Alerts.instrumentIOAlert(badloc),
                AlertState.WARNING,
                cause);
            mailSender.send(cause, exc);
        });
        return newIns.getLastException().isPresent();
    }

    /**
     * Does nothing.  Called at the end of the initialization phase
     * after {@code start()} has been called for all configurable components and after
     * {@code checkHardware()} and {@code checkStarted()} have been called for all
     * hardware controllers.
     */
    @Override
    public void postStart() {}

    /**
     * Cancels periodic readout if this hasn't already been done (part of shutdown).
     * The instruments themselves are not stopped and continue to collect data.
     * Called at the beginning of the shutdown phase.
     * @throws HardwareException if periodic readout is still scheduled.
     * @see #signal(org.lsst.ccs.framework.Signal)
     */
    @Override
    public void checkStopped() throws HardwareException {
        readoutTask.cancel(false);
        if (!readoutTask.isCancelled()) {
            throw new HardwareException(false, "Could not cancel the readout task.");
        }
    }

    /**
     * Handles all alerts by allowing them to be cleared.
     * @param alert The alert to be cleared.
     * @param state The current state (severity) of the alert.
     * @return {@code CLEAR_ALERT}.
     */
    @Override
    public ClearAlertHandler.ClearAlertCode canClearAlert(final Alert alert, final AlertState state) {
        return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
    }

    ////////// Utilities //////////

    /**
     * Gets a list of all instruments matching the given identification.
     * @param ident The identification, either the name of the location the instrument
     * is monitoring or a device index number.
     * @return The list of instruments, possibly empty.
     */
    private List<Instrument> findInstruments(String ident) {
        List<Instrument> ins = new ArrayList<>();
        try {
            final int i = Integer.parseInt(ident);
            ins.add(instruments.get(i));
        }
        catch (NumberFormatException | IndexOutOfBoundsException exc) {}
        if (ins.isEmpty()) {
            ins = instruments
                    .stream()
                    .filter(i -> i.getStatus().location.equalsIgnoreCase(ident))
                    .collect(Collectors.toList());
        }
        return ins;
    }

    /**
     * Calls a {@code Callable<String>} on behalf of a command that needs to synchronize
     * with the periodic readout. Upon exit the current thread will
     * not hold the readout lock.
     * @param cmd The code to run.
     * @return The result from {@code cmd}.
     * @throws Exception if {@code cmd.call()} throws.
     */
    private String callIfNotBusy(Callable<String> cmd) throws Exception {
        if (!readoutLock.tryLock()) {
            // Send rejection of the command.
            return "BUSY";
        }
        try {
            // Execute the command.
            return cmd.call();
        } finally {
            readoutLock.unlock();
        }
    }
}
