package org.lsst.ccs.subsystem.airwatch.main;

import org.lsst.ccs.bus.data.KeyValueDataList;

import java.io.Serializable;

import java.time.Instant;
import java.util.ArrayList;
import java.util.List;

import java.util.Map;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * Describes an immutable data record matching a record gotten from a directly-connected
 * Lighthouse instrument. Such a record has a single timestamp, a single location key, a single
 * limit violation flag, a single malfunction flag and a set of channel-value pairs.
 * @author tether
 */
public final class TrendableModbusRecord  implements TrendableRecord {
    /**
     * The trending key to use for the flag that tells whether some channel
     * in the record is outside the valid range.
     */
    public  static String LIMIT_VIOLATION_KEY = "limitViolation";

    /**
     * The trending key to use for the flag that tells whether the instrument
     * reported a malfunction for this record.
     */
    public  static String MALFUNCTION_KEY = "malfunction";


    private final String masterKey;
    private final Instant masterTimestamp;
    private final Map<String, Serializable> items;


    /**
     * Construct an instance.
     * @param masterKey Will become the master key.
     * @param masterTimestamp Will the common timestamp.
     * @param items A {@code Map} from value key to serializable value.
     */
    public TrendableModbusRecord(
        String masterKey,
        Instant masterTimestamp,
        Map<String, Serializable> items)
    {
        this.masterKey = masterKey;
        this.masterTimestamp = masterTimestamp;
        this.items = java.util.Collections.unmodifiableMap(items);
    }

    /** {@inheritDoc } */
    @Override
    public String getMasterKey() {return masterKey;}

    /** {@inheritDoc } */
    @Override
    public Instant getMasterTimestamp() {return masterTimestamp;}

    /** {@inheritDoc } */
    @Override
    public List<Alert> getAlerts() {
        final List<Alert> alerts = new ArrayList<>();
        final Serializable limitFlag = items.get(LIMIT_VIOLATION_KEY);
        if (limitFlag == null && (Integer)limitFlag != 0) {
            alerts.add(Alerts.limitViolationAlert(masterKey, "Handheld"));
        }
        final Serializable malFlag = items.get(MALFUNCTION_KEY);
        if (malFlag == null && (Integer)malFlag != 0) {
            alerts.add(Alerts.instrumentMalfunctionAlert(masterKey, "Handheld"));
        }
        return alerts;
    }

    /** {@inheritDoc } */
    @Override
    public Map<String, Serializable> getItems() {return items;}

    /** {@inheritDoc } */
    @Override
    public void post(org.lsst.ccs.Subsystem subsys) {
        final KeyValueDataList kvdl =
            new KeyValueDataList(masterKey,
                CCSTimeStamp.currentTimeFromMillis(masterTimestamp.toEpochMilli()) );
        items.forEach( (key, value) -> kvdl.addData(key, value));
        subsys.publishSubsystemDataOnStatusBus(kvdl);
    }

    /**
     * Create a printable representation of this object.
     * @return A number of lines, each terminated with a newline, like so:
     * <code>
     * Master key: foo
     * Master time: 2016-05-28T22:50:06.000Z
     *     key1: value1
     *     key2: value2
     *     ...
     * </code>
     */
    @Override
    public String toString() {
        final StringBuilder repr = new StringBuilder();
        repr.append(String.format("Master key: %s%n", masterKey));
        repr.append(String.format("Master time: %s%n", masterTimestamp.toString()));
        items.forEach(
            (key, value) -> repr.append(String.format("    %s: %s%n" ,key, value.toString())));
        return repr.toString();
    }
}
