package org.lsst.ccs.subsystem.airwatch.main;

import static java.lang.String.format;
import org.lsst.ccs.bus.data.Alert;

/**
 * Contains a set of static factory functions for the alerts used by the subsystem.
 * @author tether
 */
public class Alerts {
    
    /**
     * Creates an alert about a failure to build or deliver an email message.
     * @return A new Alert instance with an ID of "EmailDelivery".
     */
    public static Alert emailDeliveryAlert() {
        return new Alert("EmailDelivery", "Problem constructing or sending an email message");
    }
    
    /**
     * Creates an alert about a failure to read an instrument.
     * @param location The long location name of the instrument, e.g., "Location 002".
     * @return A new Alert instance with an ID of {@code format("InstrumentIO-%s", location)}.
     */
    public static Alert instrumentIOAlert(final String location) {
        return new Alert(
                format("InstrumentIO-%s", location),
                "I/O error when attempting to read an instrument.");
    }
    
    /**
     * Creates an alert about a sensor malfunction reported by the instrument.
     * @param location The short location name of the instrument, e.g., AUX, MAIN_SE.
     * @param channelName The name of the channel that had the failure.
     * @return A new Alert instance with an ID of {@code format("Malfunction-%s-%s", location, channelName)}.
     */
    public static Alert instrumentMalfunctionAlert(final String location, final String channelName) {
        return new Alert(
                format("Malfunction-%s-%s", location, channelName),
                "Sensor malfunction reported by an instrument.");
    }

    /**
     * Creates an alert about an a sensor reading that's outside of the desired range.
     * @param shortLocationName The location of the instrument, e.g., ENTRY, MAIN_SE.
     * @param channelName Which quantity has gone out of range, e.g., 0.3, Humidity.
     * @return A new Alert instance with an ID of
     * {@code format("LimitViolation-%s-%s", shortLocationName, channelName)}.
     */
    public static Alert limitViolationAlert(final String shortLocationName, final String channelName) {
        return new Alert(
                format("LimitViolation-%s-%s", shortLocationName, channelName),
                "Value out of bounds.");
    }
    
}
