package org.lsst.ccs.subsystem.airwatch.main;

import java.io.Serializable;
import java.time.Instant;
import java.util.Map;
import java.util.Optional;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.subsystem.airwatch.main.LighthouseOPC.CounterPoint;

/**
 * Implements a trendable data item read from a Lighthouse OPC server. On the server
 * each counter has its own malfunction flag and limitViolation (alarm)
 * as well as a count.
 * @author tether
 */
public class TrendableOPCRecord  implements TrendableRecord {

    private final String masterKey;
    private final Instant masterTimestamp;
    private final Map<String, Serializable> items;

    /**
     * Construct an instance.
     * @param masterKey Will become the master key.
     * @param masterTimestamp Will the common timestamp.
     * @param items A {@code Map} from value key to serializable value.
     */
    public TrendableOPCRecord(
        String masterKey,
        Instant masterTimestamp,
        Map<String, Serializable> items)
    {
        this.masterKey = masterKey;
        this.masterTimestamp = masterTimestamp;
        this.items = java.util.Collections.unmodifiableMap(items);
    }

    /** {@inheritDoc } */
    @Override
    public String getMasterKey() {return masterKey;}

    /** {@inheritDoc } */
    @Override
    public Instant getMasterTimestamp() {return masterTimestamp;}

    /** {@inheritDoc } */
    @Override
    public boolean hasLimitViolation() {
        Optional<Serializable> found =
            items.entrySet().stream()
            .map(Map.Entry::getValue)
            .filter(x -> x instanceof CounterPoint)
            .filter(x -> ((CounterPoint)x).limitViolation != 0)
            .findAny();
        return found.isPresent();
    }

    /** {@inheritDoc } */
    @Override
    public boolean hasMalfunction() {
        Optional<Serializable> found =
            items.entrySet().stream()
            .map(Map.Entry::getValue)
            .filter(x -> x instanceof CounterPoint)
            .filter(x -> ((CounterPoint)x).malfunction != 0)
            .findAny();
        return found.isPresent();
    }

    /** {@inheritDoc } */
    @Override
    public Map<String, Serializable> getItems() {return items;}

    /** {@inheritDoc } */
    @Override
    public void post(org.lsst.ccs.Subsystem subsys) {
        final KeyValueDataList kvdl =
            new KeyValueDataList(masterKey, masterTimestamp.toEpochMilli());
        items.forEach( (key, value) -> kvdl.addData(key, value));
        subsys.publishSubsystemDataOnStatusBus(kvdl);
    }

    /**
     * Create a printable representation of this object.
     * @return A number of lines, each terminated with a newline, like so:
     * <code>
     * Master key: foo
     * Master time: 2016-05-28T22:50:06.000Z
     *     key1: value1
     *     key2: value2
     *     ...
     * </code>
     */
    @Override
    public String toString() {
        final StringBuilder repr = new StringBuilder();
        repr.append(String.format("Master key: %s%n", masterKey));
        repr.append(String.format("Master time: %s%n", masterTimestamp.toString()));
        items.forEach(
            (key, value) -> repr.append(String.format("    %s: %s%n" ,key, value.toString())));
        return repr.toString();
    }

}
