package org.lsst.ccs.subsystem.airwatch.main;

import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;

import java.io.Serializable;

import java.time.Instant;

import java.util.Map;

/**
 * Describes an immutable data record matching a record gotten from a directly-connected
 * Lighthouse instrument. Such a record has a single timestamp, a single location key, a single
 * limit violation flag, a single malfunction flag and a set of channel-value pairs.
 * @author tether
 */
public final class TrendableModbusRecord  implements TrendableRecord {
    /**
     * The trending key to use for the flag that tells whether some channel
     * in the record is outside the valid range.
     */
    public  static String LIMIT_VIOLATION_KEY = "limitViolation";

    /**
     * The trending key to use for the flag that tells whether the instrument
     * reported a malfunction for this record.
     */
    public  static String MALFUNCTION_KEY = "malfunction";


    private final String masterKey;
    private final Instant masterTimestamp;
    private final Map<String, Serializable> items;


    /**
     * Construct an instance.
     * @param masterKey Will become the master key.
     * @param masterTimestamp Will the common timestamp.
     * @param items A {@code Map} from value key to serializable value.
     */
    public TrendableModbusRecord(
        String masterKey,
        Instant masterTimestamp,
        Map<String, Serializable> items)
    {
        this.masterKey = masterKey;
        this.masterTimestamp = masterTimestamp;
        this.items = java.util.Collections.unmodifiableMap(items);
    }

    /** {@inheritDoc } */
    @Override
    public String getMasterKey() {return masterKey;}

    /** {@inheritDoc } */
    @Override
    public Instant getMasterTimestamp() {return masterTimestamp;}

    /** {@inheritDoc } */
    @Override
    public boolean hasLimitViolation() {
        final Serializable flag = items.get(LIMIT_VIOLATION_KEY);
        if (flag == null) return false;
        if (flag instanceof Integer) return (Integer)flag != 0;
        throw new Error("Limit violation flag isn't an Integer: " + flag);
    }

    /** {@inheritDoc } */
    @Override
    public boolean hasMalfunction() {
        final Serializable flag = items.get(MALFUNCTION_KEY);
        if (flag == null) return false;
        if (flag instanceof Integer) return (Integer)flag != 0;
        throw new Error("Malfunction flag isn't an Integer: " + flag);
    }

    /** {@inheritDoc } */
    @Override
    public Map<String, Serializable> getItems() {return items;}

    /** {@inheritDoc } */
    @Override
    public void post(org.lsst.ccs.Subsystem subsys) {
        final KeyValueDataList kvdl =
            new KeyValueDataList(masterKey, masterTimestamp.toEpochMilli());
        items.forEach( (key, value) -> kvdl.addData(key, value));
        subsys.publishSubsystemDataOnStatusBus(kvdl);
    }

    /**
     * Create a printable representation of this object.
     * @return A number of lines, each terminated with a newline, like so:
     * <code>
     * Master key: foo
     * Master time: 2016-05-28T22:50:06.000Z
     *     key1: value1
     *     key2: value2
     *     ...
     * </code>
     */
    @Override
    public String toString() {
        final StringBuilder repr = new StringBuilder();
        repr.append(String.format("Master key: %s%n", masterKey));
        repr.append(String.format("Master time: %s%n", masterTimestamp.toString()));
        items.forEach(
            (key, value) -> repr.append(String.format("    %s: %s%n" ,key, value.toString())));
        return repr.toString();
    }
}
