/*
 * Decompiled with CFR 0.152.
 */
package org.lsst.ccs.subsystem.airwatch.main;

import java.io.Serializable;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.Callable;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.annotation.Resource;
import org.apache.commons.mail.EmailException;
import org.apache.commons.mail.SimpleEmail;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HardwareController;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.framework.Signal;
import org.lsst.ccs.framework.SignalLevel;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.subsystem.airwatch.main.ChannelLimitAlert;
import org.lsst.ccs.subsystem.airwatch.main.ConfigurationService;
import org.lsst.ccs.subsystem.airwatch.main.EmailDeliveryAlert;
import org.lsst.ccs.subsystem.airwatch.main.Instrument;
import org.lsst.ccs.subsystem.airwatch.main.InstrumentChannel;
import org.lsst.ccs.subsystem.airwatch.main.InstrumentConfig;
import org.lsst.ccs.subsystem.airwatch.main.InstrumentIOAlert;
import org.lsst.ccs.subsystem.airwatch.main.InstrumentMalfunctionAlert;
import org.lsst.ccs.subsystem.airwatch.main.InstrumentReport;
import org.lsst.ccs.subsystem.airwatch.main.LocationConfig;
import org.lsst.ccs.subsystem.airwatch.main.TrendableRecord;
import org.lsst.ccs.utilities.logging.Logger;
import org.lsst.ccs.utilities.scheduler.PeriodicTask;

public final class AirwatchMain
extends Module
implements HardwareController {
    @Resource(name="config")
    private final ConfigurationService configService = null;
    private Map<String, LocationConfig> locations = null;
    private final CopyOnWriteArrayList<Instrument> instruments = new CopyOnWriteArrayList();
    private final Lock readoutLock;
    private PeriodicTask readoutTask = null;
    private final Logger log = Logger.getLogger((String)"org.lsst.ccs.subsystem.airwatch");

    @Command(type=Command.CommandType.QUERY, description="Displays a short report on instrument status.")
    public String instruments() {
        return new InstrumentReport(this.instruments).toString();
    }

    @Command(type=Command.CommandType.ACTION, description="Prevents further readout of an instrument")
    public String disable(@Argument(description="Instrument index number or location.") String ident) throws Exception {
        return this.callIfNotBusy(() -> {
            List<Instrument> insList = this.findInstruments(ident);
            if (insList.size() != 1) {
                throw new IllegalArgumentException("There are " + insList.size() + " instruments at " + ident);
            }
            Instrument ins = insList.get(0);
            this.instruments.set(ins.getIndex(), ins.disable());
            return "OK";
        });
    }

    @Command(type=Command.CommandType.ACTION, description="Sets new location for and enables instrument.")
    public String enable(@Argument(description="Instrument index number or current location.") String ident, @Argument(description="New location.") String newLocation) throws Exception {
        return this.callIfNotBusy(() -> {
            List<Instrument> insList = this.findInstruments(ident);
            if (insList.size() != 1) {
                throw new IllegalArgumentException("There are " + insList.size() + " instruments at " + ident);
            }
            if (!this.locations.containsKey(newLocation)) {
                throw new IllegalArgumentException("Unknown location " + newLocation);
            }
            Instrument ins = insList.get(0);
            Instrument newIns = ins.enable(this.locations.get(newLocation));
            this.instruments.set(ins.getIndex(), newIns);
            if (newIns.getLastException().isPresent()) {
                throw newIns.getLastException().get();
            }
            return "OK";
        });
    }

    public TreeWalkerDiag signal(Signal sig) {
        if (sig.getLevel() == SignalLevel.STOP) {
            this.readoutTask.cancel(false);
        } else if (sig.getLevel() == SignalLevel.HALT) {
            this.readoutTask.cancel(true);
        }
        return TreeWalkerDiag.GO;
    }

    public AirwatchMain(String name) {
        super(name);
        this.readoutLock = new ReentrantLock();
    }

    public void initModule() {
        if (this.configService == null) {
            throw new Error("Missing ConfigurationService component.");
        }
        this.getSubsystem().addClearAlertHandler(alert -> ClearAlertHandler.ClearAlertCode.CLEAR_ALERT);
    }

    public void start() {
    }

    public TreeWalkerDiag checkHardware() throws HardwareException {
        this.readConfiguration();
        this.enableAllInstruments();
        this.scheduleFirstReadout(Duration.ZERO);
        return TreeWalkerDiag.GO;
    }

    private void readConfiguration() {
        this.locations = Collections.unmodifiableMap(this.configService.getLocationConfigs().stream().collect(Collectors.toMap(loc -> loc.name, Function.identity())));
        List<InstrumentConfig> configs = this.configService.getInstrumentConfigs();
        List insts = configs.stream().map(cfg -> cfg.type.make((InstrumentConfig)cfg)).collect(Collectors.toList());
        this.instruments.addAll(insts);
    }

    private void enableAllInstruments() throws HardwareException {
        ArrayList exceptions = new ArrayList();
        this.instruments.forEach(inst -> {
            LocationConfig loc = this.locations.get(inst.getStatus().location);
            Instrument newIns = inst.enable(loc);
            this.instruments.set(inst.getIndex(), newIns);
            newIns.getLastException().ifPresent(exc -> exceptions.add(exc));
        });
        if (exceptions.size() > 0) {
            HardwareException top = (HardwareException)exceptions.remove(0);
            for (HardwareException exc : exceptions) {
                top = new HardwareException(exc);
            }
            throw top;
        }
    }

    private void scheduleFirstReadout(Duration initialDelay) {
        this.readoutTask = this.getSubsystem().getScheduler().scheduleAtFixedRate(this::readoutTaskBody, initialDelay.getSeconds(), this.configService.getReadoutInterval().getSeconds(), TimeUnit.SECONDS);
    }

    private void readoutTaskBody() {
        try {
            this.readoutLock.lockInterruptibly();
        }
        catch (InterruptedException exc) {
            Thread.currentThread().interrupt();
            return;
        }
        try {
            this.instruments.stream().filter(ins -> ins.getStatus().enabled).forEach(ins -> {
                if (Thread.currentThread().isInterrupted()) {
                    throw new TerminationException();
                }
                this.readAndUpdate((Instrument)ins);
            });
        }
        catch (TerminationException exc) {
            this.log.warning((Object)"Instrument readout was interrupted!");
        }
        finally {
            this.readoutLock.unlock();
        }
    }

    private void readAndUpdate(Instrument ins) {
        Instrument newIns = ins.read();
        this.instruments.set(ins.getIndex(), newIns);
        if (this.checkForExceptionDuringReadout(newIns)) {
            this.instruments.set(ins.getIndex(), newIns.disable());
        }
        this.checkForLimitViolations(newIns);
        this.checkForMalfunctions(newIns);
        newIns.getTrendables().forEach(msg -> msg.post(this.getSubsystem()));
        this.configService.updateInstrument(newIns.getStatus());
    }

    private boolean checkForExceptionDuringReadout(Instrument newIns) {
        newIns.getLastException().ifPresent(exc -> {
            String badloc = newIns.getStatus().location;
            String cause = "Instrument readout error";
            this.getSubsystem().raiseAlert((Alert)new InstrumentIOAlert(), AlertState.WARNING, badloc + ": " + exc.getMessage());
            this.log.error((Object)(badloc + ": " + "Instrument readout error"), (Throwable)exc);
            this.sendEmail("Instrument readout error", exc.toString());
        });
        return newIns.getLastException().isPresent();
    }

    private void checkForLimitViolations(Instrument newIns) {
        String offenders = newIns.getTrendables().filter(TrendableRecord::hasLimitViolation).flatMap(rec -> {
            Map<String, Serializable> items = rec.getItems();
            String location = rec.getMasterKey().toLowerCase();
            Instant time = rec.getMasterTimestamp();
            Map<InstrumentChannel, Long> limits = this.locations.get((Object)location).thresholds;
            return items.keySet().stream().map(InstrumentChannel::parse).filter(Optional::isPresent).map(Optional::get).filter(InstrumentChannel::isParticleChannel).filter(chan -> limits.containsKey(chan)).filter(chan -> (Double)items.get(chan.getKey()) > (double)((Long)limits.get(chan)).longValue()).map(chan -> String.format("%s %10s %5s %10.0f %10d", time, location, chan.getKey(), (Double)items.get(chan.getKey()), limits.get(chan)));
        }).collect(Collectors.joining("\n"));
        if (offenders.length() > 0) {
            String summary = "Limit violation(s) reported by instruments";
            this.getSubsystem().raiseAlert((Alert)new ChannelLimitAlert(), AlertState.WARNING, newIns.getStatus().location + ": " + "Limit violation(s) reported by instruments");
            String msg = String.format("Time, location, particle size (microns), count, limit:%n%s%n", offenders);
            this.log.warning((Object)("Limit violation(s) reported by instruments\n" + msg));
            this.sendEmail("Limit violation(s) reported by instruments", msg);
        }
    }

    private void checkForMalfunctions(Instrument newIns) {
        String offenders = newIns.getTrendables().filter(TrendableRecord::hasMalfunction).map(rec -> rec.getMasterTimestamp().toString() + " " + rec.getMasterKey()).collect(Collectors.joining("\n"));
        if (offenders.length() > 0) {
            String summary = "Malfunctions(s) reported by instrument";
            this.getSubsystem().raiseAlert((Alert)new InstrumentMalfunctionAlert(), AlertState.WARNING, "Malfunctions(s) reported by instrument");
            String msg = String.format("Times and locations:%n%s%n", offenders);
            this.log.warning((Object)("Malfunctions(s) reported by instrument: " + msg));
            this.sendEmail("Malfunctions(s) reported by instrument", msg);
        }
    }

    public void checkStarted() throws HardwareException {
    }

    public void postStart() throws HardwareException {
    }

    public void checkStopped() throws HardwareException {
        if (!this.readoutTask.isCancelled()) {
            throw new HardwareException(false, "You must first use the stop command.");
        }
    }

    private List<Instrument> findInstruments(String ident) {
        List<Instrument> ins = new ArrayList<Instrument>();
        try {
            int i2 = Integer.parseInt(ident);
            ins.add(this.instruments.get(i2));
        }
        catch (IndexOutOfBoundsException | NumberFormatException runtimeException) {
            // empty catch block
        }
        if (ins.isEmpty()) {
            ins = this.instruments.stream().filter(i -> i.getStatus().location.equalsIgnoreCase(ident)).collect(Collectors.toList());
        }
        return ins;
    }

    private String callIfNotBusy(Callable<String> cmd) throws Exception {
        if (!this.readoutLock.tryLock()) {
            return "BUSY";
        }
        try {
            String string = cmd.call();
            return string;
        }
        finally {
            this.readoutLock.unlock();
        }
    }

    private void sendEmail(String subject, String message) {
        try {
            String[] recipients = this.configService.getEmailRecipients().toArray(new String[0]);
            if (recipients.length > 0) {
                SimpleEmail emsg = new SimpleEmail();
                emsg.setHostName(this.configService.getSMTPServer());
                emsg.setFrom(this.configService.getEmailSender());
                emsg.setBounceAddress(this.configService.getEmailBounceAddress());
                emsg.addReplyTo(this.configService.getEmailBounceAddress());
                emsg.setSubject("[Airwatch] " + subject);
                emsg.setMsg(message);
                emsg.addTo(recipients);
                emsg.send();
            }
        }
        catch (EmailException exc) {
            String summary = "Error composing or sending email.";
            this.log.error((Object)"Error composing or sending email.", (Throwable)exc);
            this.getSubsystem().raiseAlert((Alert)new EmailDeliveryAlert(), AlertState.WARNING, "Error composing or sending email.");
        }
    }

    private static class TerminationException
    extends RuntimeException {
        private TerminationException() {
        }
    }
}

