package org.lsst.ccs.subsystem.airwatch.main;

import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;

import java.io.Serializable;

import java.time.Instant;

import java.util.Map;

/**
 * An immutable data record in a form independent of instrument type. The form is chosen
 * to be easily convertible to a {@code KeyValueDataList}.
 * @author tether
 */
public final class TrendableRecord {
    
    
    /**
     * The trending key to use for the flag that tells whether some channel
     * in the record is outside the valid range.
     */
    public  static String LIMIT_VIOLATION_KEY = "limitViolation";
    
    /**
     * The trending key to use for the flag that tells whether the instrument
     * reported a malfunction for this record.
     */
    public  static String MALFUNCTION_KEY = "malfunction";

    
    private final String masterKey;
    private final Instant masterTimestamp;
    private final Map<String, Serializable> items;

    
    /**
     * Construct an instance.
     * @param masterKey Will become the master key.
     * @param masterTimestamp Will the common timestamp.
     * @param items A {@code Map} from value key to serializable value.
     */
    public TrendableRecord(
        String masterKey,
        Instant masterTimestamp,
        Map<String, Serializable> items)
    {
        this.masterKey = masterKey;
        this.masterTimestamp = masterTimestamp;
        this.items = java.util.Collections.unmodifiableMap(items);
    }
    
    /** Gets the master key given to the constructor.
     * @return The key.
     */
    public String getMasterKey() {return masterKey;}
    
    /** Gets the master timestamp given to the constructor.
     * @return The timestamp.
     */
    public Instant getMasterTimestamp() {return masterTimestamp;}
    
    /**
     * Gets the value of the limit violation flag for this record.
     * @return The flag.
     * @throws Error if the flag isn't present of isn't a Double value.
     */
    public boolean hasLimitViolation() {
        final Serializable flag = items.get(LIMIT_VIOLATION_KEY);
        if (flag instanceof Double) return (Double)flag >= 0.5; // Value is 0.0 or 1.0
        throw new Error("Limit violation flag isn't present or isn't Double: " + flag);
    }
    
    /**
     * Gets the value of the instrument malfunction flag for this record.
     * @return The flag.
     * @throws Error if the flag isn't present of isn't a Double value.
     */
    public boolean hasMalfunction() {
        final Serializable flag = items.get(MALFUNCTION_KEY);
        if (flag instanceof Double) return (Double)flag >= 0.5; // Value is 0.0 or 1.0
        throw new Error("Malfunction flag isn't present or isn't Double: " + flag);
    }
    
    /**
     * Gets the map of trending keys to serializable values.
     * @return The map (unmodifiable).
     */
    public Map<String, Serializable> getItems() {return items;}
    
    /**
     * Post this data to the local CCS trending database. The master key and timestamp
     * become the key and timestamp of a {@code KeyValueDataList} to which is added
     * one {@KeyValueData} item for each data channel.
     */
    public void post(org.lsst.ccs.Subsystem subsys) {
        final KeyValueDataList kvdl =
            new KeyValueDataList(masterKey, masterTimestamp.toEpochMilli());
        items.forEach( (key, value) -> kvdl.addData(key, value));
        subsys.publishSubsystemDataOnStatusBus(kvdl);
    }
    
    /**
     * Create a printable representation of this object.
     * @return A number of lines, each terminated with a newline, like so:
     * <code>
     * Master key: foo
     * Master time: 2016-05-28T22:50:06.000Z
     *     key1: value1
     *     key2: value2
     *     ...
     * </code>
     */
    @Override
    public String toString() {
        final StringBuilder repr = new StringBuilder();
        repr.append(String.format("Master key: %s%n", masterKey));
        repr.append(String.format("Master time: %s%n", masterTimestamp.toString()));
        items.forEach(
            (key, value) -> repr.append(String.format("    %s: %s%n" ,key, value.toString())));
        return repr.toString();
    }
}
