package org.lsst.ccs.subsystem.airwatch.main;

import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;

import java.time.Instant;

import java.util.Map;

/**
 * An immutable data record in a form independent of instrument type. The form is chosen
 * to be easily convertible to a {@code KeyValueDataList}.
 * @author tether
 */
public class TrendableRecord {
    
    private final String masterKey;
    private final Instant masterTimestamp;
    private final Map<String, Double> channels;
    
    /**
     * Construct an instance.
     * @param masterKey Will become the master key.
     * @param masterTimestamp Will the common timestamp.
     * @param channels A {@code Map} from channel key to channel value.
     */
    public TrendableRecord(
        String masterKey,
        Instant masterTimestamp,
        Map<String, Double> channels)
    {
        this.masterKey = masterKey;
        this.masterTimestamp = masterTimestamp;
        this.channels = java.util.Collections.unmodifiableMap(channels);
    }
    
    /** Gets the master key given to the constructor.
     * @return The key.
     */
    public String getMasterKey() {return masterKey;}
    
    /** Gets the master timestamp given to the constructor.
     * @return The timestamp.
     */
    public Instant getMasterTimestamp() {return masterTimestamp;}
    
    /**
     * Post this data to the local CCS trending database. The master key and timestamp
     * become the key and timestamp of a {@code KeyValueDataList} to which is added
     * one {@KeyValueData} item for each data channel.
     */
    public void post(org.lsst.ccs.Subsystem subsys) {
        final KeyValueDataList kvdl =
            new KeyValueDataList(masterKey, masterTimestamp.toEpochMilli());
        channels.forEach( (key, value) -> kvdl.addData(key, value));
        subsys.publishSubsystemDataOnStatusBus(kvdl);
    }
    
    /**
     * Create a printable representation of this object.
     * @return A number of lines, each terminated with a newline, like so:
     * <code>
     * Master key: foo
     * Master time: 2016-05-28T22:50:06.000Z
     *     key1: value1
     *     key2: value2
     *     ...
     * </code>
     */
    @Override
    public String toString() {
        final StringBuilder repr = new StringBuilder();
        repr.append(String.format("Master key: %s%n", masterKey));
        repr.append(String.format("Master time: %s%n", masterTimestamp.toString()));
        channels.forEach(
            (key, value) -> repr.append(String.format("    %s: %s%n" ,key, value.toString())));
        return repr.toString();
    }
}
