package org.lsst.ccs.subsystem.airwatch.main;

import org.lsst.ccs.drivers.commons.DriverException;

import org.lsst.ccs.HardwareException;

import org.lsst.ccs.drivers.lighthouse.DataRecord;
import org.lsst.ccs.drivers.lighthouse.LighthouseClient;

import java.time.Duration;
import java.time.Instant;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * Lighthouse Worldwide Solutions 3016-like instrument.
 * @author tether
 */
public class LighthouseInstrument implements Instrument {
    
    // Timeout for I/O operations.
    private static final Duration DEVICE_TIMEOUT = Duration.ofSeconds(10);
    
    // Let the driver log debug messages?
    private static final boolean DEVICE_DEBUG = false;
    
    // After we set the device clock from the system clock, when we read it back it
    // must be this close to a new system clock reading.
    private static final Duration DEVICE_CLOCK_MAX_ERROR = Duration.ofSeconds(10);

    private final int index;
    private final boolean enabled;
    private final String location;
    private final Instant lastDataTime;
    private final String connectionInfo;
    private final Optional<HardwareException> lastException;
    private final List<TrendableRecord> trendables;

    /**
     * Constructs a new instance from configuration data. The new instance will
     * be disabled for reading, have no stored data and have no stored exception.
     * @param config The subsystem's configuration information for the instrument.
     */
    public LighthouseInstrument(InstrumentConfig config) {
        this.index = config.index;
        this.enabled = false;
        this.location = config.loc;
        this.lastDataTime = config.lastDataTime;
        this.connectionInfo = config.conn;
        this.lastException = Optional.empty();
        this.trendables = Collections.EMPTY_LIST;
    }

    /**
     * Constructs a new instance using a value specified for each field.
     * @param index The instrument index (place in the list of instruments).
     * @param enabled The enabled-for-readout flag.
     * @param location The name of the location being monitored.
     * @param lastDataTime The next readout will get data newer than this.
     * @param connectionInfo The string of connection info from the configuration.
     * @param lastException The last exception thrown by an operation.
     * @param trendables The trendable items resulting from the last readout.
     */
    private LighthouseInstrument(
        int index,
        boolean enabled,
        String location,
        Instant lastDataTime,
        String connectionInfo,
        Optional<HardwareException> lastException,
        List<TrendableRecord> trendables)
    {
        this.index = index;
        this.enabled = enabled;
        this.location = location;
        this.lastDataTime = lastDataTime;
        this.connectionInfo = connectionInfo;
        this.lastException = lastException;
        this.trendables = trendables;
    }

    /** {@inheritDoc } */
    @Override
    public Optional<HardwareException> getLastException() {
        return lastException;
    }

    /** {@inheritDoc } */
    @Override
    public int getIndex() {
        return index;
    }

    /** {@inheritDoc } */
    @Override
    public InstrumentStatus getStatus() {
        return new InstrumentStatus(index, enabled, location, lastDataTime);
    }

    /** {@inheritDoc } */
    @Override
    public Stream<TrendableRecord> getTrendables() {
        return trendables.stream();
    }

    /** {@inheritDoc } */
    @Override
    public Instrument disable() {
        return new LighthouseInstrument(
            index,
            false,
            location,
            lastDataTime,
            connectionInfo,
            lastException,
            trendables
        );
    }

    // This implementation assumes that the subsystem configuration list of
    // known locations is stored in the instrument and in the same order.
    @Override
    public Instrument enable(LocationConfig loc) {
        LighthouseClient client = null;
        boolean newEnabled = enabled;
        String newLocation = location;
        HardwareException hwexc = null;
        try {
            client = parseConnectionInfo(connectionInfo);
            client.stopRecording();
            client.setClock(DEVICE_CLOCK_MAX_ERROR);
            client.setLocationIndex(loc.index + 1); // Instrument starts numbering at 1.
            final String locCheck = client.getLocationName();
            if (!locCheck.equalsIgnoreCase(loc.name)) {
                final String msg = String.format(
                    "Location table error for instrument at '%s'.%n" +
                        "Location #%d is '%s', should be '%s'.",
                    location, loc.index, locCheck, loc.name);
                hwexc = new HardwareException(true, msg);
            }
            client.startRecording();
            newEnabled = true;
            newLocation = loc.name;
        }
        catch (Exception exc) {
            if (exc instanceof DriverException) {
                hwexc = new HardwareException(true, exc);
            } else {
                throw new Error(exc);
            }

        }
        finally {
            if (client != null) try {client.close();} catch(DriverException exc){}
        }
        return new LighthouseInstrument(
            index,
            newEnabled,
            newLocation,
            lastDataTime,
            connectionInfo,
            Optional.ofNullable(hwexc),
            Collections.EMPTY_LIST
        );
    }

        @Override
        public Instrument read() {
            LighthouseClient client = null;
            HardwareException hwexc = null;
            List<DataRecord> records = Collections.EMPTY_LIST;
            List<TrendableRecord> newData = Collections.EMPTY_LIST;
            Instant newLast = lastDataTime;
            Instrument ins = null;
            try {
                client = parseConnectionInfo(connectionInfo);
                client.stopRecording();
                client.setClock(DEVICE_CLOCK_MAX_ERROR);
                records = client.getRecentRecords(lastDataTime);
                client.startRecording();
            }
            catch (DriverException exc) {
                if (exc instanceof DriverException) {
                    hwexc = new HardwareException(true, exc);
                    records = Collections.EMPTY_LIST;
                }
                else {
                    throw new Error(exc);
                }
            }
            catch (Exception exc) {throw new Error(exc);}
            finally {
                if (client != null) try {client.close();} catch(DriverException exc){}
            }
            // We're using the most-recent-first mode of instrument readout, so the
            // timestamp on the first record is the last-data time we want.
            newData = extractTrendables(records.stream());
            if (newData.size() > 0) {
                newLast = newData.get(0).getMasterTimestamp();
            }
            ins = new LighthouseInstrument(
                index,
                enabled,
                location,
                newLast,
                connectionInfo,
                Optional.ofNullable(hwexc),
                Collections.unmodifiableList(newData)
            );
            return ins;
        }

        private static LighthouseClient parseConnectionInfo(String info) throws DriverException {
            // Info will contain two words separated by whitespace.
            String[] words = info.trim().split("\\s+");
            final String method = words[0].toLowerCase();
            final String parameters = words[1];
            LighthouseClient client = null;
            if (method.equals("serial")) {
                // The second word is the name of the serial port.
                client = LighthouseClient.serialOpen(parameters, DEVICE_TIMEOUT, DEVICE_DEBUG);
            }
            else if (method.equals("ftdi")) {
                // The second word is of the form [host:]ftdiSerialNumber.
                client = LighthouseClient.ftdiOpen(parameters, DEVICE_TIMEOUT, DEVICE_DEBUG);
            }
            else if (method.equals("net")) {
                // The second word is of the form host:portNumber.
                // portNumber = unsigned decimal int.
                words = parameters.split(":");
                final String host = words[0];
                final int portNo = Integer.parseUnsignedInt(words[1]);
                client = LighthouseClient.netOpen(host, portNo, DEVICE_TIMEOUT, DEVICE_DEBUG);
            }
            else {
                throw new IllegalArgumentException(
                    "Illegal Lighthouse connection type: " + method
                );
            }
            return client;
        }
        
        private static List<TrendableRecord> extractTrendables(Stream<DataRecord> records) {
            return
                records
                .map(rec -> {
                    final String key = rec.getLocation();
                    final Instant stamp = rec.getStartTime();
                    final Map<String, Double> channels =
                        rec.getChannels()
                        .collect(
                            Collectors.toMap(
                                datum -> datum.getChannel().getName(),
                                datum -> datum.getValue()
                            )
                        );
                    return new TrendableRecord(key, stamp, channels);
                })
                .collect(Collectors.toList());
        }
}
