package org.lsst.ccs.subsystem.airwatch.main;

import org.lsst.ccs.HardwareException;

import org.lsst.ccs.bus.states.AlertState;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import static org.lsst.ccs.command.annotations.Command.CommandType.ACTION;
import static org.lsst.ccs.command.annotations.Command.CommandType.QUERY;

import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HardwareController;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.framework.Signal;
import org.lsst.ccs.framework.SignalLevel;
import org.lsst.ccs.framework.TreeWalkerDiag;

import org.lsst.ccs.utilities.logging.Logger;

import org.lsst.ccs.utilities.scheduler.PeriodicTask;

import java.time.Duration;
import java.time.Instant;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;

import java.util.concurrent.Callable;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.TimeUnit;

import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import java.util.function.Function;

import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import javax.annotation.Resource;

/** The main module for the Airwatch subsystem. This module is responsible for:
 * <ul>
 * <li>Establishing initial contact with the configured instruments.</li>
 * <li>Providing and managing the periodic readout task.</li>
 * <li>Alert management.</li>
 * <li>Instrument management and status reporting.</li>
 * </ul>
 *  * @author tether
 */
public class AirwatchMain extends Module implements HardwareController {
/* The module maintains a list of immutable Instrument objects which describe the state
 * of each instrument and specify the allowed operations. Mutating operations therefore follow
 * the read-modify-write pattern. A CopyOnWriteArrayList is enough to synchronize query
 * commands with mutating action commands and with the readout task but it isn't enough to
 * synchronize the action commands with the periodic readout.
 *
 * Take for example the case of disabling readout for a previously enabled instrument with
 * no prior readout data. The disable() command copies the instrument state and
 * produces a new state with the enabled flag set to false. While that's happening
 * the readout task might copy the same old state and produce a new state with new readout data.
 * Depending on who writes back first either the enable flag will remain true or the
 * new readout data will become inaccessible. 
 *
 * For this reason there's an extra lock used to synchronize the readout task with mutating
 * action commands. The readout task will hold the lock while it reads instruments and modifies
 * the state list. The mutating action commands must take the lock before doing anything else,
 * delaying the readout task until they finish their execution, which should be brief.
 * They in turn release the lock after completing their changes. 
 *
 * The periodic readout task normally works its way down the instrument list, reading any data
 * accumulated since the last-data-time for each instrument and publishing that data in the
 * trending database. As it goes the task modifies the list of Instrument objects to
 * reflect the new state of data collection for each instrument.
 *
 * The readout task checks for interruption after performing read-and-publish for each
 * instrument. This is needed to ensure that the instrument's last-data-time is consistent with
 * what has been posted for trending. An interruption will cause the Instrument
 * objects that have not been processed to be left unchanged in the instrument list.
 *
 */

    ////////// Commands //////////

    /**
     * Gives the status of each of the instruments. Includes instrument number, location,
     * enable flag and the timestamp of the last record read.
     * @return A string with a heading line followed by one line per instrument.
     */
    @Command(type=QUERY, description="Displays a short report on instrument status.")
    public String instruments() {return new InstrumentReport(instruments).toString();}

    /**
     * Prevents an instrument from being read until further notice.
     * @param ident Either an instrument number or a location name.
     * @return "OK", or "BUSY" if a readout is in progress.
     * @throws IllegalArgumentException if {@code ident} is invalid or ambiguous.
     * @see #enable(java.lang.String, java.lang.String)
     */
    @Command(type=ACTION, description="Prevents further readout of an instrument")
    public String disable(
        @Argument(description="Instrument index number or location.")
        final String ident
    ) throws Exception
    {
        return callIfNotBusy(() -> {
            final List<Instrument> insList = findInstruments(ident);
            if (insList.size() != 1) {
                throw new IllegalArgumentException(("There are " + insList.size()) + " instruments at " + ident);
            }
            final Instrument ins = insList.get(0);
            instruments.set(ins.getIndex(), ins.disable());
            return "OK";
        });
    }

    /**
     * Sets a new location for an instrument then enables periodic readout for it.
     * @param ident Either an instrument number or a location name.
     * @param newLocation The name of the new location to be monitored.
     * @return "OK", or "BUSY" if a readout is in progress.
     * @throws IllegalArgumentException if {@code ident} is invalid or ambiguous.
     * @throws IllegalArgumentException if {@code newLocation} is invalid.
     * @throws HardwareException if device operations fail.
     * @see #disable(java.lang.String)
     */
    @Command(type = ACTION, description="Sets new location for and enables instrument.")
    public String enable(
        @Argument(description="Instrument index number or current location.")
        final String ident,
        @Argument(description="New location.")
        final String newLocation
    ) throws Exception
    {
        return callIfNotBusy( () -> {
            final List<Instrument> insList = findInstruments(ident);
            if (insList.size() != 1) {
                throw new IllegalArgumentException(
                    ("There are " + insList.size()) +
                    " instruments at " + ident);
            }
            if (!locations.containsKey(newLocation)) {
                throw new IllegalArgumentException("Unknown location " + newLocation);
            }
            final Instrument ins = insList.get(0);
            final Instrument newIns = ins.enable(locations.get(newLocation));
            instruments.set(ins.getIndex(), newIns);
            if (newIns.getLastException().isPresent()) throw newIns.getLastException().get();
            return "OK";
        });
    }

    /**
     * Signal handler for stop and abort.
     * @param sig The signal.
     * @return TreeWalkerDiag.GO.
     */
    @Override
    public TreeWalkerDiag signal(Signal sig) {
        if (sig.getLevel() == SignalLevel.STOP) {
            readoutTask.cancel(false); // Don't interrupt any current readout.
        }
        else if (sig.getLevel() == SignalLevel.HALT) {
            readoutTask.cancel(true); // Interrupt any current readout.
        }
        return TreeWalkerDiag.GO;
    }

    
    ////////// References to other subsystem components //////////

    @Resource(name="config")
    private final ConfigurationService configService;

    
    ////////// Subsystem state data //////////

    /**
     * The descriptions of all known locations, keyed by location name.
     */
    private Map<String, LocationConfig> locations;

    /**
     * The descriptions of all instruments being managed. Updated by ACTION
     * commands and read by the readout task and by QUERY commands.
     */
    private final CopyOnWriteArrayList<Instrument> instruments;

    /**
     * Synchronizes action commands that mutate {@link #instruments} with
     * the periodic readout task.
     */
    private final Lock readoutLock;

    /**
     * Regularly scheduled to read the instruments.
     */
    private PeriodicTask readoutTask;

    /**
     * The subsystem's logger.
     */
    private final Logger log;


    ////////// Lifecycle methods //////////
    // Arranged in chronological order of calling.

    /**
     * Constructor.
     * @param name The module name.
     */
    public AirwatchMain(String name) {
        super(name);
        configService = null;
        locations = null;
        instruments = new CopyOnWriteArrayList<>();
        log = Logger.getLogger("org.lsst.ccs.subsystem.airwatch");
        readoutTask = null;
        readoutLock = new ReentrantLock();
    }

    /**
     * Checks whether all the subsystem components have been constructed
     * and registers the ClearAlertHandler.
     * Called just after the subsystem component tree has been built.
     * @throws Error if anything is missing.
     */
    @Override
    public void initModule() {
        if (configService == null) {
            throw new Error("Missing ConfigurationService component.");
        }
        getSubsystem().addClearAlertHandler(
            alert -> ClearAlertHandler.ClearAlertCode.CLEAR_ALERT);
    }

    /**
     * Does nothing. Called at the start of the initialization phase.
     */
    @Override
    public void start() {}

    /**
     * Attempts to start all configured instruments.
     * Called in the middle of the initialization phase after {@start()} has been called
     * for all configurable components. All instruments are stopped and re-started
     * with correct configurations for their assigned locations.
     * @throws HardwareException if any fail to respond or if any report hardware faults.
     * @return TreeWalkerDiag.GO.
     */
    @Override
    public TreeWalkerDiag checkHardware() throws HardwareException {
        readConfiguration();
        enableAllInstruments();
        scheduleFirstReadout(Duration.ZERO);
        return TreeWalkerDiag.GO;
    }

    private void readConfiguration() {
        locations = Collections.unmodifiableMap(
            configService.getLocationConfigs()
            .stream()
            .collect(Collectors.toMap(loc -> loc.name, Function.identity()))
        );
        final List<InstrumentConfig> configs = configService.getInstrumentConfigs();
        final List<Instrument> insts
            = configs
            .stream()
            .map(cfg -> cfg.type.make(cfg))
            .collect(Collectors.toList());
        instruments.addAll(insts);
    }

    private void enableAllInstruments() throws HardwareException {
        final List<HardwareException> exceptions = new ArrayList<>();
        instruments.forEach(inst -> {
            final LocationConfig loc = locations.get(inst.getStatus().location);
            final Instrument newIns = inst.enable(loc);
                instruments.set(inst.getIndex(), newIns);
                newIns.getLastException().ifPresent(exc -> exceptions.add(exc));
        });
        // Make a chain of all the hardware exceptions and throw the top of the chain.
        if (exceptions.size() > 0) {
            HardwareException top = exceptions.remove(0);
            for (HardwareException exc : exceptions) top = new HardwareException(exc);
            throw top;
        }
    }

    private void scheduleFirstReadout(Duration initialDelay) {
        this.readoutTask =
            getSubsystem()
                .getScheduler()
                .scheduleAtFixedRate(this::readoutTaskBody,
                    initialDelay.getSeconds(),
                    configService.getReadoutInterval().getSeconds(),
                    TimeUnit.SECONDS);
    }

    private void readoutTaskBody() {
        //  Synchronize with mutating action commands.
        try {
            this.readoutLock.lockInterruptibly();
        }
        catch (InterruptedException exc) {
            Thread.currentThread().interrupt();
            return;
        }
        
        // Read, update status, post trendables.
        try {
            instruments
                .stream()
                .filter(ins -> ins.getStatus().enabled)
                .forEach(ins -> {
                    if (Thread.currentThread().isInterrupted()) {
                        throw new TerminationException();
                    }
                    readAndUpdate(ins);
                });
        } catch (TerminationException exc) {
            log.warning("Instrument readout was interrupted!");
        }
        finally {
            readoutLock.unlock();
        }
    }

    private static class TerminationException extends RuntimeException {}
    
    private void readAndUpdate(final Instrument ins) {
        final Instrument newIns = ins.read();
        instruments.set(ins.getIndex(), newIns);
        newIns.getLastException().ifPresent(exc -> {
            instruments.set(ins.getIndex(), newIns.disable());
            final String cause = "Problem during readout for " + newIns.getStatus().location;
            getSubsystem().raiseAlert(
                new InstrumentIO(),
                AlertState.WARNING,
                newIns.getStatus().location + ": " + exc.getMessage());
            log.error(cause, exc);
        });
        newIns
            .getTrendables()
            .forEach(msg -> msg.post(getSubsystem()));
        configService.updateInstrument(newIns.getStatus());
    }

    /**
     * Checks whether all the instruments are running. Called after checkHardware().
     * @throws HardwareException if any aren't.
     */
    @Override
    public void checkStarted() throws HardwareException {}

    /**
     * Does nothing.  Called at the end of the initialization phase
     * after {@code start()} has been called for all configurable components and after
     * {@code checkHardware()} and {@code checkStarted()} have been called for all
     * hardware controllers.
     * @throws HardwareException never.
     */
    @Override
    public void postStart() throws HardwareException {}

    /**
     * Checks whether future readouts have been canceled.
     * The instruments themselves are not stopped and continue to collect data.
     * Called at the beginning of the shutdown phase.
     * @throws HardwareException if periodic readout is still scheduled.
     * @see #stop()
     */
    @Override
    public void checkStopped() throws HardwareException {
        if (!readoutTask.isCancelled()) {
            throw new HardwareException(false, "You must first use the stop command.");
        }
    }

    
    ////////// Utilities //////////
    
    /**
     * Gets a list of all instruments matching the given identification.
     * @param ident The identification, either the name of the location the instrument
     * is monitoring or a device index number.
     * @return The list of instruments, possibly empty.
     */
    private List<Instrument> findInstruments(String ident) {
        List<Instrument> ins = new ArrayList<>();
        try {
            final int i = Integer.parseInt(ident);
            ins.add(instruments.get(i));
        }
        catch (NumberFormatException | IndexOutOfBoundsException exc) {}
        if (ins.isEmpty()) {
            ins = instruments
                    .stream()
                    .filter(i -> i.getStatus().location.equalsIgnoreCase(ident))
                    .collect(Collectors.toList());
        }
        return ins;
    }
    
    /**
     * Calls a {@code Callable<String>} on behalf of a command that needs to synchronize
     * with the periodic readout. Upon exit the current thread will
     * not hold the readout lock.
     * @param cmd The code to run.
     * @return The result from {@code cmd}.
     * @throws Throwable if {@code cmd.call()} throws.
     */
    private String callIfNotBusy(Callable<String> cmd) throws Exception {
        if (!readoutLock.tryLock()) {
            // Send rejection of the command.
            return "BUSY";
        }
        try {
            // Execute the command.
            return cmd.call();
        } finally {
            readoutLock.unlock();
        }
    }
}
