package org.lsst.ccs.subsystem.airwatch.main;

import org.lsst.ccs.drivers.commons.DriverException;

import org.lsst.ccs.HardwareException;

import org.lsst.ccs.drivers.lighthouse.DataRecord;
import org.lsst.ccs.drivers.lighthouse.LighthouseClient;

import java.time.Duration;
import java.time.Instant;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import java.util.concurrent.Callable;

import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * Lighthouse Worldwide Solutions 3016-like instrument.
 * @author tether
 */
public class LighthouseInstrument implements Instrument {
    
    private static final Duration DEVICE_TIMEOUT = Duration.ofSeconds(10);
    private static final boolean DEVICE_DEBUG = false;

    private final int index;
    private final boolean enabled;
    private final String location;
    private final Instant lastDataTime;
    private final Callable<LighthouseClient> connector;
    private final Optional<HardwareException> lastException;
    private final List<TrendableRecord> trendables;

    public LighthouseInstrument(InstrumentConfig config) {
        this.index = config.index;
        this.enabled = false;
        this.location = config.loc;
        this.lastDataTime = config.lastDataTime;
        this.connector = parseConnectionInfo(config.conn);
        this.lastException = Optional.empty();
        this.trendables = Collections.EMPTY_LIST;
    }

    private LighthouseInstrument(
        int index,
        boolean enabled,
        String location,
        Instant lastDataTime,
        Callable<LighthouseClient> connector,
        Optional<HardwareException> lastException,
        List<TrendableRecord> trendables)
    {
        this.index = index;
        this.enabled = enabled;
        this.location = location;
        this.lastDataTime = lastDataTime;
        this.connector = connector;
        this.lastException = lastException;
        this.trendables = trendables;
    }

    @Override
    public Optional<HardwareException> getLastException() {
        return lastException;
    }

    @Override
    public int getIndex() {
        return index;
    }

    @Override
    public InstrumentStatus getStatus() {
        return new InstrumentStatus(index, enabled, location, lastDataTime);
    }

    @Override
    public Stream<TrendableRecord> getTrendables() {
        return trendables.stream();
    }

    @Override
    public Instrument disable() {
        return new LighthouseInstrument(
            index,
            false,
            location,
            lastDataTime,
            connector,
            lastException,
            trendables
        );
    }

    @Override
    public Instrument enable(LocationConfig loc) {
        return new LighthouseInstrument(
            index,
            false,
            location,
            lastDataTime,
            connector,
            Optional.empty(),
            Collections.EMPTY_LIST
        );
    }

        @Override
        public Instrument read() {
            LighthouseClient client = null;
            HardwareException hwexc = null;
            Stream<DataRecord> records = Stream.empty();
            List<TrendableRecord> newData = Collections.EMPTY_LIST;
            Instant newLast = lastDataTime;
            Instrument ins = null;
            try {
                client = connector.call();
                records = client.getRecentRecords(lastDataTime);
            }
            catch (Exception exc) {
                if (exc instanceof DriverException) {
                    hwexc = new HardwareException(true, exc);
                    records = Stream.empty();
                }
                else {
                    throw new Error(exc);
                }
            }
            finally {
                if (client != null) try {client.close();} catch(DriverException exc){}
            }
            // We're using the most-recent-first mode of instrument readout, so the
            // timestamp on the first record is the last-data time we want.
            newData = extractTrendables(records);
            if (newData.size() > 0) {
                newLast = newData.get(0).getMasterTimestamp();
            }
            ins = new LighthouseInstrument(
                index,
                enabled,
                location,
                newLast,
                connector,
                Optional.ofNullable(hwexc),
                Collections.unmodifiableList(newData)
            );
            return ins;
        }

        private static Callable<LighthouseClient> parseConnectionInfo(String info) {
            // Info will contain two words separated by whitespace.
            String[] words = info.trim().split("\\w");
            final String method = words[0].toLowerCase();
            final String parameters = words[1];
            Callable<LighthouseClient> connector = null;
            if (method.equals("serial")) {
                // The second word is the name of the serial port.
                connector = () ->
                    LighthouseClient
                        .serialOpen(parameters, DEVICE_TIMEOUT, DEVICE_DEBUG);
            }
            else if (method.equals("ftdi")) {
                // The second word is of the form [host:]ftdiSerialNumber.
                connector = () ->
                    LighthouseClient
                        .ftdiOpen(parameters, DEVICE_TIMEOUT, DEVICE_DEBUG);
            }
            else if (method.equals("net")) {
                // The second word is of the form host:portNumber.
                // portNumber = unsigned decimal int.
                words = parameters.split(":");
                final String host = words[0];
                final int portNo = Integer.parseUnsignedInt(words[1]);
                connector = () ->
                    LighthouseClient
                        .netOpen(host, portNo, DEVICE_TIMEOUT, DEVICE_DEBUG);
            }
            else {
                throw new IllegalArgumentException(
                    "Illegal Lighthouse connection type: " + method
                );
            }
            return connector;
        }
        
        private static List<TrendableRecord> extractTrendables(Stream<DataRecord> records) {
            return
                records
                .map(rec -> {
                    final String key = rec.getLocation();
                    final Instant stamp = rec.getStartTime();
                    final Map<String, Double> channels =
                        rec.getChannels()
                        .collect(
                            Collectors.toMap(
                                datum -> datum.getChannel().getName(),
                                datum -> datum.getValue()
                            )
                        );
                    return new TrendableRecord(key, stamp, channels);
                })
                .collect(Collectors.toList());
        }
}
