package org.lsst.ccs.subsystem.airwatch.main;

import org.lsst.ccs.commons.annotations.ConfigurationParameter;

import org.lsst.ccs.framework.ConfigurableComponent;

import java.time.Duration;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;

import java.util.ArrayList;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static java.util.function.Function.identity;

import java.util.Collections;

import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;

/**
 * Implements a configuration service based on the CCS subsystem configuration API.
 * An instance will be the receptacle of all the subsystem configuration data which is declared
 * here using the {@link ConfigurationParameter} annotation. The instance will be a component
 * of the subsystem and hence is a subclass of {@link ConfigurableComponent}.
 * @author tether
 */
public class CCSConfiguration extends ConfigurableComponent implements ConfigurationService {

    ////////// Implementation of ConfigurationService //////////
    private volatile Duration readoutInterval;
    private volatile List<InstrumentConfig> instConfigs;
    private volatile List<LocationConfig> locConfigs;

    /** {@inheritDoc } */
    @Override
    public List<InstrumentConfig> getInstrumentConfigs() {
        return instConfigs;
    }

    /** {@inheritDoc } */
    @Override
    public void updateInstrument(InstrumentStatus stat) {
        instrumentLocations[stat.index] = stat.location;
        instrumentLastDataTimes[stat.index] = stat.lastDataTime.toString();
        change("instrumentLocations", instrumentLocations);
        change("instrumentLastDataTimes", instrumentLastDataTimes);
        getEnvironment().saveAllChanges();
    }

    /** {@inheritDoc } */
    @Override
    public List<LocationConfig> getLocationConfigs() {
        return locConfigs;
    }

    /** {@inheritDoc } */
    @Override
    public Duration getReadoutInterval() {return readoutInterval;}


    ////////// Lifecycle methods //////////
    /** Makes an instance. */
    public CCSConfiguration() {
        super();
    }

    /**
     * Checks configuration data and makes the configuration objects.
     * Called at the start of the initialization phase.
     */
    @Override
    public void start() {
        try {
            readoutInterval = Duration.parse(readoutIntervalString);
        }
        catch (java.time.DateTimeException exc) {
            throw new IllegalArgumentException("Bad configuration. Invalid readout interval.");
        }
        makeLocationConfigs();
        makeInstrumentConfigs();
    }

    private void makeLocationConfigs() {
        // Map instrument channels to the particle threshold configuration members
        // to references to the members themselves.
        final Map<InstrumentChannel, Map<String, Long>> chanToLoc =
            Stream.of(InstrumentChannel.values())
            .filter(InstrumentChannel::isParticleChannel)
            .collect(Collectors.toMap(
                identity(),
                chan -> {
                    try {
                        return
                            (Map<String, Long>)this.getClass()
                            .getDeclaredField(channelThresholdsConfigVarName(chan))
                            .get(this);
                    } catch(NoSuchFieldException exc) {
                        throw new IllegalArgumentException("Missing configuration variable?", exc);
                    } catch(IllegalAccessException exc) {
                        throw new Error("Impossible!", exc);}
                    }));

        // Now perform validity checks on each threshold value map.
        // All locations named must be in knownLocations.
        // All threshold values must be in the range [0, Integer.MAX_VALUE].
        final List<String> knownLocs = Arrays.asList(knownLocations);
        chanToLoc.forEach((chan, configVar) -> {
            checkStringSet(
                configVar.keySet().stream(),
                knownLocs.stream(),
                "Bad locations in " + channelThresholdsConfigVarName(chan));
            checkThresholds(configVar, channelThresholdsConfigVarName(chan));
        });

        // Create the location configurations. Transpose: one map-by-location per channel
        // becomes one map-by-channel per location.
        final List<Map<InstrumentChannel, Long>> locToChan =
            IntStream.range(0, knownLocs.size())
            .mapToObj(i -> new EnumMap<InstrumentChannel, Long>(InstrumentChannel.class))
            .collect(ArrayList::new, ArrayList::add, ArrayList::addAll);

        chanToLoc.forEach((chan, configVar) -> {
            configVar.forEach((locName, threshold) -> {
                final int i = knownLocs.indexOf(locName);
                locToChan.get(i).put(chan, threshold);
            });
        });

        locConfigs = Collections.unmodifiableList(
            IntStream.range(0, knownLocs.size())
            .mapToObj(i -> new LocationConfig(knownLocs.get(i), i, locToChan.get(i)))
            .collect(ArrayList::new, ArrayList::add, ArrayList::addAll)
            );
    }

    private void makeInstrumentConfigs() {
        final List<String> knownLocs = Arrays.asList(knownLocations);
        final String[] types = instrumentTypes;
        final String[] conns = instrumentConnections;
        final String[] locs = instrumentLocations;
        final String[] times = instrumentLastDataTimes;
        // Validity checking.
        if ((types.length != conns.length)
            || (types.length != locs.length)
            || (types.length != times.length)) {
            throw new IllegalArgumentException(
                "Bad configuration. Not all the instrumentXxx[] arrays are the same length."
            );
        }
        if (types.length == 0) {
            throw new IllegalArgumentException("Bad configuration. No instruments defined!");
        }
        checkStringSet(Stream.of(locs), knownLocs.stream(), "Unknown instrument locations");
        checkStringSet(
            Stream.of(types),
            Stream.of(InstrumentType.values()).map(InstrumentType::getConfigName),
            "Unknown instrument types");
        checkStringSet(
            Stream.of(times),
            Stream.of(times).filter(t -> {
                try {
                    Instant.parse(t);
                    return true;
                } catch (java.time.format.DateTimeParseException exc) {
                    return false;
                }
            }),
            "Last data-times rejected by Instant.parse()"
        );
        // Construction of configuration objects.
        instConfigs = Collections.unmodifiableList(
            IntStream
            .range(0, types.length)
            .mapToObj(i -> new InstrumentConfig(
                    i,
                    InstrumentType.parse(types[i]).get(),
                    conns[i],
                    locs[i],
                    Instant.parse(times[i])))
            .collect(Collectors.toList()));
    }

    private static void checkStringSet(
        Stream<String> mentioned,
        Stream<String> legal,
        String errorMsg)
    {
        Set<String> unknown = mentioned.collect(Collectors.toSet());
        unknown.removeAll(legal.collect(Collectors.toSet()));
        if (unknown.size() > 0) {
            throw new IllegalArgumentException(
                "Bad configuration. " + errorMsg + ": " +
                unknown.stream().collect(Collectors.joining(", ")));
        }
    }

    private static void checkThresholds(Map<String, Long> threshVar, String varName) {
        threshVar.forEach((locName, threshold) -> {
            if ((threshold < 0) || (threshold > Integer.MAX_VALUE)) {
                throw new IllegalArgumentException(
                    String.format(
                        "Bad configuration. Bad threshold value %d for location %s in %s.",
                        threshold,
                        locName,
                        varName
                        )
                );
            }
        });
    }

    private String channelThresholdsConfigVarName(InstrumentChannel chan) {
        return "locationThresholds_" + chan.getKey().replaceFirst("\\.", "_");
    }

    ////////// Configuration data //////////

    /**
     * The time between readouts of all the instruments.
     */
    @ConfigurationParameter(name="readoutInterval", description="The time between readouts.")
    private volatile String readoutIntervalString = "PT1H";

    /**
     * The set of all legal names for instrument locations.
     */
    @ConfigurationParameter(description="All valid location names.")
    private volatile String[] knownLocations = new String[] {};

    /**
     * Maps location names to alarm thresholds for the 0.3 micron channel.
     */
    @ConfigurationParameter(description="Alarm thresholds for the 0.3 micron channel.")
    private volatile Map<String, Long> locationThresholds_0_3 = Collections.EMPTY_MAP;

    /**
     * Maps location names to alarm thresholds for the 0.5 micron channel.
     */
    @ConfigurationParameter(description="Alarm thresholds for the 0.5 micron channel.")
    private volatile Map<String, Long> locationThresholds_0_5 = Collections.EMPTY_MAP;

    /**
     * Maps location names to alarm thresholds for the 1.0 micron channel.
     */
    @ConfigurationParameter(description="Alarm thresholds for the 1.0 micron channel.")
    private volatile Map<String, Long> locationThresholds_1_0 = Collections.EMPTY_MAP;

    /**
     * Maps location names to alarm thresholds for the 2.5 micron channel.
     */
    @ConfigurationParameter(description="Alarm thresholds for the 2.5 micron channel.")
    private volatile Map<String, Long> locationThresholds_2_5 = Collections.EMPTY_MAP;

    /**
     * Maps location names to alarm thresholds for the 5.0 micron channel.
     */
    @ConfigurationParameter(description="Alarm thresholds for the 5.0 micron channel.")
    private volatile Map<String, Long> locationThresholds_5_0 = Collections.EMPTY_MAP;

    /**
     * Maps location names to alarm thresholds for the 10.0 micron channel.
     */
    @ConfigurationParameter(description="Alarm thresholds for the 10.0 micron channel.")
    private volatile Map<String, Long> locationThresholds_10_0 = Collections.EMPTY_MAP;

    /**
     * For each instrument, its make and/or model, for example "lighthouse".
     */
    @ConfigurationParameter(description="Instrument make and/or model.")
    private volatile String[] instrumentTypes = new String[] {};

    /**
     * For each instrument, a string containing connection information to be parsed.
     * For example "serial /dev/ttyUSB0" or "net foo.slac.stanford.edu 2250"
     */
    @ConfigurationParameter(description="How to open a connection to the instrument.")
    private volatile String[] instrumentConnections = new String[] {};

    /**
     * For each instrument, its location, drawn from the set of known locations.
     */
    @ConfigurationParameter(description="The last known location of the instrument.")
    private volatile String[] instrumentLocations = new String[] {};

    /**
     * For each instrument, the time of the most recent data read from it. Subsequent
     * reads will obtain data later than this. The string must be acceptable to
     * Instant.parse().
     */
    @ConfigurationParameter(description="The Instant of the lastest data read from the instrument.")
    private volatile String[] instrumentLastDataTimes = new String[] {};
}
