package org.lsst.ccs.startup;

import java.io.Serializable;
import java.lang.reflect.Constructor;
import org.lsst.ccs.framework.annotations.ParameterSetterBuilder;
import java.lang.reflect.Field;
import java.lang.reflect.Type;
import org.lsst.ccs.CCSCst;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.description.EffectiveNode;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.stream.Collectors;
import org.lsst.ccs.bus.states.ConfigurationState;
import org.lsst.ccs.utilities.conv.InputConversionEngine;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.config.AParameterDescription;
import org.lsst.ccs.config.ConfigurableSubsystem;
import org.lsst.ccs.config.ConfigurationProxy;
import org.lsst.ccs.config.Factories;
import org.lsst.ccs.config.LocalConfigurationProxy;
import org.lsst.ccs.config.ParameterBase;
import org.lsst.ccs.config.ParameterConfiguration;
import org.lsst.ccs.config.ParameterDescription;
import org.lsst.ccs.config.ParameterPath;
import org.lsst.ccs.config.SubsystemDescription;
import org.lsst.ccs.config.utilities.ConfigUtils;
import org.lsst.ccs.description.ComponentNodeBuilder;
import org.lsst.ccs.description.DescriptiveNode;
import org.lsst.ccs.framework.annotations.ParameterSetter;
import org.lsst.ccs.utilities.reflect.ConstructorUtils;
import org.lsst.ccs.utilities.constraints.Constraints;
import org.lsst.ccs.utilities.conv.TypeConversionException;
import org.lsst.ccs.utilities.logging.Logger;
import org.lsst.gruth.jutils.ArrayParm;
import org.lsst.gruth.jutils.BoolParm;
import org.lsst.gruth.jutils.DblParm;
import org.lsst.gruth.jutils.HollowParm;
import org.lsst.gruth.jutils.IntParm;
import org.lsst.gruth.jutils.ListParm;
import org.lsst.gruth.jutils.MapParm;
import org.lsst.gruth.jutils.NamedRefParm;
import org.lsst.gruth.jutils.StringParm;
import org.lsst.gruth.jutils.ValParm;

/**
 * A collection of static utilities linked to boot operations. Note that:
 * <UL>
 * <LI>
 * a descriptive node represents a configuration description
 * <LI/>
 * an effective node is the real subsystem and module hierarchy (ready to run)
 * <LI/>
 * a "raw" subsystem is without configuration data
 * </UL>
 *
 * @author bamade
 */
// Date: 05/06/12
public class BootUtils {
    

    
    private BootUtils() {
    }
    
    /**
     * Very important class: when an instance is created you can have :
     * <UL>
     * <LI/> a descriptive Node
     * <LI/> the names of subsystem, configuration, tag
     * <LI/> the corresponding Effective Node
     * <LI/> a Subsystem ready to run (but not started)
     * </UL>
     */
    private static class LocalBootObject {
        
        /** The map of component names to their set of configuration setting methods. */
        private final Map<String, ParameterSetter> parameterSetterDictionary = new HashMap<>();
        /** The initial descriptiveNode: not the one modified by the
         * Configuration. */
        private final DescriptiveNode descriptiveNode;
        private final SubsystemDescription subsystemDescription;
        private final String subsystemName;
        private final String[] taggedCategories;
        private String tagName;
        
        /** For warning purposes.*/
        private boolean usesTypedParms = false;
        
        /**
         * creates a boot Object from a description file name.
         *
         * @param pathName
         * @throws Exception
         */
        LocalBootObject(String pathName) throws Exception {
            this(pathName, "");
        }
        
        /**
         * Creates a boot Object from a description file name and a
         * configuration property file. The values in the configuration file
         * override the parameters set in the description File.
         *
         * @param fullDescription
         * @param config can be a file name OR the config name only
         * @throws Exception
         */
        LocalBootObject(String fullDescription, String config) throws Exception {
            this(fullDescription, config, null);
        }
        
        LocalBootObject(String fullDescription, String config, String subsystemAlias) throws Exception {
            if (config == null) {
                config = "";
            }
            if (config.contains("/")) {
                throw new IllegalArgumentException("configuration files must be located at the root of a resource directory");
            }
            if (config.contains(".")) {
                throw new IllegalArgumentException("configuration input must be the name of the configuration only");
            }
            
            // First build a DescriptiveNode from the provided fullDescription
            descriptiveNode = ComponentNodeBuilder.buildDescriptiveNode(fullDescription);
            
            /*
            * Resolving :
            * - the subsystem name
            * - the configuration name
            * - the tag name
            */
            subsystemName = (subsystemAlias == null) ? descriptiveNode.getSubsystemName() : subsystemAlias;
            taggedCategories = config.split(",");
            
            //TODO: Why do we need this variable? Can it be removed or can we use "fullDescription"?
            tagName = fullDescription;
            if (fullDescription.contains(":")) {
                tagName = fullDescription.substring(fullDescription.indexOf(':') + 1);
            }
            
            CCSCst.LOG_TODO.fine("TODO: Configuration proxy should also be able to interact with remote database (switch between local and remote?)");
            
            /*
            creates a SubsystemDescription Object from the descriptive node
            the parameters that are tagged "name" and that are References to other
            components are not included in the list of Parameters that can be modified
            */
            subsystemDescription = buildSubsystemDescription(subsystemName, tagName,
                    descriptiveNode);
        }
        
        public Subsystem getSubsystem() throws Exception{
            
            /** The configuration proxy that will be associated to the configurable subsystem. */
            ConfigurationProxy proxy = new LocalConfigurationProxy(subsystemDescription);
            
            /** Descriptive Node modification. */
            // First step : get the values for startup configuration
            Map.Entry<ConfigurationState, Set<ParameterConfiguration>> stateAndParmConfigs
                    = proxy.getInitialParameterConfigurations(ConfigUtils.parseConfigurationStringWithDefaults(subsystemDescription.getCategorySet(), taggedCategories));
            // Second step : modify the descriptive node.
            modifyDescriptiveNode(descriptiveNode, stateAndParmConfigs.getValue());
            
            /** Effective Node creation. */
            EffectiveNode effectiveNode = ComponentNodeBuilder.buildEffectiveNode(descriptiveNode);
            
            /** Effective Node modification. */
            modifyEffectiveNode(effectiveNode, stateAndParmConfigs.getValue(), proxy);
            
            /** Subsystem construction from the effective Node. */
            ConfigurableSubsystem subsystem = new NodeModularSubsystem(subsystemName, proxy, effectiveNode, stateAndParmConfigs.getKey(), parameterSetterDictionary);
            return subsystem;
        }
        
        public String getSubsystemName() {
            return subsystemName;
        }
        
        public String getTagName() {
            return tagName;
        }
        
        /**
         * Creates a collection of ParameterDescription objects out of the
         * attributes of {@code node}. There are currently two supported ways of defining
         * configurable parameters : through the constructor or using the ConfigurationParameter
         * annotation.
         *
         * @param node
         * @param maxLevel
         * @return a Collection of ParameterDescription.
         */
        private Collection<ParameterDescription> parameterDescriptionsFromNode(DescriptiveNode node) {
            Collection<ParameterDescription> res = new ArrayList<>();
            // Looking for configurable parameters defined in the description node.
            Map<String, Object> attributes = node.getAttributes();
            String nodeKey = node.getKey();
            Class klass = node.getRealValue();
            
            // build the ParameterSetter corresponding to this node.
            // warning : the target is not set.
            ParameterSetter parmSetterForNode = ParameterSetterBuilder.buildParameterSetterFromClass(klass);
            parameterSetterDictionary.put(nodeKey, parmSetterForNode);
            
            // -- Creation of the ParameterDescription objects from the annotated parameters
            Set<Map.Entry<String, Field>> entrySet = parmSetterForNode.getParameterFields().entrySet();
            for (Map.Entry<String, Field> entry : entrySet) {
                String parmName = entry.getKey();
                ConfigurationParameter a = entry.getValue().getAnnotation(ConfigurationParameter.class);
                
                // Default value non accessible before object creation.
                String defaultValue = "";
                ParameterBase base = new ParameterBase(nodeKey, "", parmName, entry.getValue().getGenericType().getTypeName(), defaultValue);
                String description = a.description();
                String simpleName = "";
                String category = a.category();
                AParameterDescription parmDescription = new AParameterDescription(base, description, simpleName, category, 0);
                res.add(parmDescription);
            }
            
            // -- Creation of the ParameterDescriptions objects from the groovy description
            if (attributes != null) {
                List<String> unannotatedparms = new ArrayList<>();
                Constructor ctorForNode = node.getConstructor();
                int attributeIndex = 0;
                for (Map.Entry<String, Object> entry : attributes.entrySet()) {
                    Object val = entry.getValue();
                    String parmName = entry.getKey();
                    
                    if ("name".equalsIgnoreCase(parmName) || val instanceof NamedRefParm) {
                        attributeIndex++;
                        continue;
                    }
                    
                    if (val instanceof HollowParm) {
                        if (!usesTypedParms && (
                                val instanceof ArrayParm
                                || val instanceof BoolParm
                                || val instanceof IntParm
                                || val instanceof DblParm
                                || val instanceof MapParm
                                || val instanceof ListParm
                                || val instanceof StringParm
                                || val instanceof ValParm
                                )) {
                            usesTypedParms = true;
                        }
                        
                        HollowParm hollow = (HollowParm) val;
                        
                        if (!hollow.isReadOnly()) {
                            if (!parmSetterForNode.getParameterFields().containsKey(parmName)) {
                                unannotatedparms.add(parmName);
                                
                                // Find the type
                                Type type = ConstructorUtils.getParameterType(ctorForNode, attributeIndex);
                                parmSetterForNode.addTypeForParm(parmName, type);
                                ParameterBase base = new ParameterBase(nodeKey, "", parmName, type.getTypeName(), hollow.toString());
                                String description = "";
                                String simpleName = "";
                                String constraints = "";
                                String category = "";
                                boolean notModifiableAtRuntime = false;
                                int level = 0;
                                Properties props = hollow.getProperties();
                                if (props != null) {
                                    // description
                                    description = props.getProperty("description", "");
                                    // simpleName
                                    simpleName = props.getProperty("simpleName", "");
                                    // constraints
                                    constraints = props.getProperty("constraints", "");
                                    // check default value against constraints
                                    try {
                                        Constraints.check(new InputConversionEngine().convertArgToType(base.getDefaultValue(), type), constraints);
                                    } catch(TypeConversionException ex) {
                                        throw new IllegalArgumentException(ex.getMessage(), ex);
                                    }
                                    parmSetterForNode.addConstraint(parmName, constraints);
                                    // category
                                    category = props.getProperty("category", "");
                                    // level
                                    String request = props.getProperty("level");
                                    if (request != null) {
                                        level = Integer.parseInt(request);
                                    }
                                    String isStatic = props.getProperty("static");
                                    if (isStatic != null) {
                                        notModifiableAtRuntime = Boolean.valueOf(isStatic);
                                    }
                                    
                                }
                                parmSetterForNode.addFinalPropForParm(parmName, notModifiableAtRuntime);
                                AParameterDescription parmDescription = new AParameterDescription(base, description, simpleName, category, level);
                                res.add(parmDescription);
                            }
                        }
                    }
                    attributeIndex++;
                }
                if (!unannotatedparms.isEmpty()) {
                    Logger.getLogger("org.lsst.ccs.startup").fine("unannotated parameters for component " + nodeKey + " : " + unannotatedparms);
                }
                
            }
            return res;
        }
        
        /**
         * creates a new {@code SubsystemDescription} populated with "empty"
         * {@code ParameterDescriptions}. The object is not in the database since
         * the list of descriptions should be modified first.
         *
         * @param subsystemName
         * @param tagName
         * @param descriptiveNode
         * @return a {@code SubsystemDescription} object containing the description
         * of the parameters as described in {@code descriptiveNode}.
         */
        private SubsystemDescription buildSubsystemDescription(String subsystemName, String tagName, DescriptiveNode descriptiveNode) {
            SubsystemDescription res = Factories.createRawSubsystemDescription(subsystemName, tagName, descriptiveNode);
            ArrayList<ParameterDescription> list = new ArrayList<>();
            
            descriptiveNode.proceduralNodeWalk(
                    node -> {
                        list.addAll(parameterDescriptionsFromNode(node));
                    },
                    null);
            if (usesTypedParms) {
                Logger.getLogger("org.lsst.ccs.startup").warn(
                        "Subsystem description contains calls such as "
                                + "anArray, aBool, anInt, aDbl, aMap, aList, aString, aVal, "
                                + "migration instructions can be found at https://confluence.slac.stanford.edu/display/LSSTCAM/Configurable+Parameters");
            }
            res.addParameterDescriptions(list);
            return res;
        }
        
        /**
         * Modifies the descriptive node with respect to the startup configuration.
         * It consists in changing the values that are going to be passed to the
         * classe's constructor.
         * TODO : from the ParameterConfiguration objects we
         * only need their name and current value.
         *
         * @param topNode
         * @param parmConfigs the set of parameters that are modified by the startup
         * configuration.
         */
        private void modifyDescriptiveNode(DescriptiveNode topNode, Set<? extends ParameterConfiguration> parmConfigs) {
            Iterator<? extends ParameterConfiguration> iterator = parmConfigs.iterator();
            while (iterator.hasNext()) {
                ParameterConfiguration parameterConfiguration = iterator.next();
                ParameterPath path = parameterConfiguration.getPath();
                String componentName = path.getComponentName();
                String codeName = path.getCodeName();
                if (codeName != null && !"".equals(codeName)) {
                    throw new UnsupportedOperationException(" no change on methods yet --> " + codeName);
                }
                String parameterName = path.getParameterName();
                DescriptiveNode goalComponent = topNode.getNodeByName(componentName);
                if (goalComponent == null) {
                    throw new IllegalArgumentException("no component for name :" + componentName);
                }
                Map mapAttributes = goalComponent.getAttributes();
                // not null
                if (mapAttributes == null) {
                    throw new IllegalArgumentException("incompatible attribute list for component//parameter " + path);
                }
                // get ConfigurationParameter
                Object rawParm = mapAttributes.get(parameterName);
                if (rawParm == null) {
                    // parameterName may be annotated and not passed to the constructor
                    continue;
                }
                if (parameterSetterDictionary.get(componentName).getParameterFields().containsKey(parameterName)) {
                    // The parameter is both annotated and mentionned in the constructor :
                    // We prefer setting the field over modifying its value passed to the constructor
                    continue;
                }
                if (rawParm instanceof HollowParm) {
                    HollowParm hollow = (HollowParm) rawParm;
                    ParameterSetter parmSetter = parameterSetterDictionary.get(componentName);
                    try {
                        hollow.modifyChecked((Serializable)new InputConversionEngine().convertArgToType(parameterConfiguration.getConfiguredValue(), parmSetter.getTypeForParm(parameterName)));
                    } catch (TypeConversionException ex) {
                        throw new IllegalArgumentException(ex.getMessage(), ex);
                    }
                    iterator.remove();
                } else {
                    throw new IllegalArgumentException("parameter not modifiable" + rawParm);
                }
            }
        }
        
        /**
         * The annotated fields of the given effective node are set to the values
         * fetched from the configuration service. In parmConfigs there are only the
         * parameters that have not been passed to the constructors of the node's
         * components.
         *
         * @param node the top node.
         * @param parmConfigs the annotated parameters
         */
        private void modifyEffectiveNode(EffectiveNode node, Set<ParameterConfiguration> parmConfigs, ConfigurationProxy proxy) {
            // Preliminary step : sort the ParameterConfiguration elements by component
            // and computes its value
            // Todo : should be made upstream
            Map<String, Map<String, String>> parmConfigByComponent = parmConfigs.stream()
                    .collect(Collectors.groupingBy(
                            pc -> pc.getPath().getComponentName(),
                            Collectors.toMap(
                                    pc -> pc.getPath().getParameterName(),
                                    ParameterConfiguration::getConfiguredValue
                            )
                    )
                    );
            // Validation + setting steps
            node.proceduralNodeWalk(
                    n -> {
                        Object innerObj = n.getRealValue();
                        String nodeName = n.getKey();
                        
                        ParameterSetter parmSetter = parameterSetterDictionary.get(nodeName);
                        parmSetter.setTarget(innerObj);
                        
                        // to annotated fields : assign the value specified by the startup configuration
                        // or the default value.
                        Map<String, Field> annotatedFields = parmSetter.getParameterFields();
                        
                        // Update default value of annotated fields to the configuration proxy
                        // This is temporary : the configuration proxy should not have knowledge of a default value
                        for(Map.Entry<String, Field> entry : annotatedFields.entrySet()) {
                            String parmName = entry.getKey();
                            Field f = entry.getValue();
                            Object val;
                            try {
                                f.setAccessible(true);
                                val = f.get(innerObj);
                            } catch (IllegalAccessException ex) {
                                throw new RuntimeException("failure setting parameter : " + ex.getMessage(), ex);
                            }
                            String constraint = parmSetter.getConstraintFor(parmName);
                            if (constraint != null) {
                                Constraints.check(val, constraint);
                            }
                            proxy.setDefaultValueForParameter(nodeName, parmName, val);
                        }
                        
                        Map<String, String> toValidate = parmConfigByComponent.get(nodeName);
                        // Validation step
                        if (toValidate == null || toValidate.isEmpty()) return;
                        
                        for (Map.Entry<String, String> entry : toValidate.entrySet()) {
                            try {
                                parmSetter.submitChange(entry.getKey(), entry.getValue());
                            } catch (TypeConversionException ex) {
                                throw new IllegalArgumentException("wrong parameter string representation for : " + entry.getKey(), ex);
                            }
                        }
                        parmSetter.invokeValidateBulkChange(proxy.getCurrentValuesForComponent(nodeName, Collections.emptySet()), false);
                        
                        // Setting step
                        
                        try {
                            parmSetter.invokeSetParameters();
                        } catch (Exception ex) {
                            throw new IllegalArgumentException("failure setting parameter : " + ex.getMessage(), ex);
                        }
                    }, null);
        }
        
       
        
    }
    ////////////////////// END LOCAL BOOT OBJECT
    
    /**
     * Builds a {@code Subsystem} instance out of a description file.
     *
     * @param descriptionName
     * @return a subsystem ready to run.
     * @throws Exception
     */
    public static Subsystem getSubsystemFromFile(String descriptionName) throws Exception {
        return getSubsystemFromFile(descriptionName, null);
    }
    
    /**
     * Builds a {@code Subsystem} given a description name and an initial
     * configuration.
     *
     * @param descriptionName the name of the subsystem description
     * @param configName the name of the configuration the subsystem is started
     * with
     * @return a ready to run subsystem
     * @throws Exception
     */
    public static Subsystem getSubsystemFromFile(String descriptionName, String configName) throws Exception {
        return getSubsystemFromFile(descriptionName, configName, null);
    }
    
    /**
     * Builds a {@code Subsystem} given a description name, an initial
     * configuration and a alias. The subsystem will be named "alias" on the
     * buses.
     *
     * @param pathName
     * @param propertiesFileName
     * @param subsystemAlias
     * @return A ready to run subsystem
     * @throws Exception
     */
    public static Subsystem getSubsystemFromFile(String pathName, String propertiesFileName, String subsystemAlias) throws Exception {
        return new LocalBootObject(pathName, propertiesFileName, subsystemAlias).getSubsystem();
    }
    
    public static SubsystemDescription buildSubsystemDescription(String pathName) throws Exception {
        return new LocalBootObject(pathName).subsystemDescription;
    }
    
    /**
     * saves the latest run configuration in a local cache. This is saved in a
     * file (see naming convention handled by <TT>baseNameFromNames</TT>
     * that will reside in tha cache directory. The name of this file will be
     * reported in another text file in this directory named "latest.txt".
     * <p/>
     * TODO: The path of the cache directory is to be determined.
     *
     * @param node
     * @param subsystemName
     * @param configName
     * @param tag
     */
    public static void saveInCache(DescriptiveNode node, String subsystemName,
            String configName, String tag) {
        //TODO: implement saveInCache
    }
    
    public static DescriptiveNode getLatestInCache() {
        return null;
    }
    
    public static void bootFromCache() {
        throw new UnsupportedOperationException("bootFromCache");
    }
    
}
