package org.lsst.ccs.startup;

import java.lang.reflect.Field;
import org.lsst.ccs.config.ConfigurationProxy;
import org.lsst.ccs.description.ComponentLookupService;
import org.lsst.ccs.config.ConfigurableSubsystem;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.framework.*;
import org.lsst.ccs.description.EffectiveNode;

import java.util.*;
import java.util.function.Consumer;
import javax.annotation.Resource;
import org.lsst.ccs.bus.messages.StatusHeartBeat;
import org.lsst.ccs.bus.states.ConfigurationState;
import org.lsst.ccs.bus.states.PhaseState;
import org.lsst.ccs.command.CommandSet;
import org.lsst.ccs.command.CommandSetBuilder;
import org.lsst.ccs.command.annotations.Command;

import static org.lsst.ccs.command.annotations.Command.CommandType.SIGNAL;
import static org.lsst.ccs.command.annotations.Command.CommandType.ACTION;
import org.lsst.ccs.config.ParameterPath;
import org.lsst.ccs.framework.annotations.ParameterSetter;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * the top modular subsystem that deals with configuration proxy and lookup service
 *
 * @author bamade
 */
public class NodeModularSubsystem extends ConfigurableSubsystem {

    private final ComponentLookupService lookup;
    private static final Logger mod_log = Logger.getLogger("org.lsst.ccs.startup");

    /**
     * Builds a NodeModularSubsystem out of a SubsystemDescription object.
     * Usage of this constructor supposes that a subsystem description has already been built upstream (or fetched from a database)
     *
     * @param subsystemName the name of the subsystem on the buses
     * @param configurationProxy the configuration proxy for this subsystem
     * @param effectiveNode the tree structure of component
     * @param initialState the initial configuration state
     * @throws Exception
     */
    NodeModularSubsystem(String subsystemName, ConfigurationProxy configurationProxy, 
            EffectiveNode effectiveNode, ConfigurationState initialState, Map<String, ParameterSetter> configChangerDictionary) {
        super(subsystemName, configurationProxy, initialState, configChangerDictionary);
        
        //todo : create a Context if needed
        this.lookup = new ComponentLookupService(effectiveNode);

        // populates the componentList and sets the environment of each configurable component
        lookup.proceduralNodeWalk(
                null,
                null,
                node -> {
                    String key = node.getKey();
                    Object value = node.getRealValue();
                    if (value instanceof Configurable) {
                        Configurable configurable = (Configurable) value;
                        configurable.setEnvironment(new ConfigurationEnvironment(key, getConfigurationProxy(), lookup, this));
                    }
                });
        
        // invokes init on all Modules
        doInitComponents();
    }

    //Add the ConfigurationInfo the the StatusHeartBeat
    @Override
    protected void updateHeartBeat(StatusHeartBeat s) {
        //       s.addObject(configurationInfo);
    }

    public ComponentLookupService getLookup() {
        return lookup;
    }
    
    private void doInitComponents() {
        // Provide the Subsystem with the named CommandSets.
        lookup.proceduralNodeWalk(
                null,
                node -> {
                    String nodeName = node.getKey();
                    Object component = node.getRealValue();
                    injectResources(component, nodeName);
                    if (component instanceof Configurable) {
                        ((Configurable)component).init();
                    }
                    CommandSet commandSet = new CommandSetBuilder().buildCommandSet(node.getRealValue());
                    String fullPath = lookup.getFullPathFor(node.getKey()); 
                    addCommandSet(fullPath, commandSet);
                }, 
                null
        );
    }
    
    /**
     * this method enables to inject field values <B>provided the fields are declared in the current class</B>
     * inherited fields are not accessible through this facility.
     * Fields should be other components in the same subsystem: they are queried by name:
     * the name being the name of the field OR the argument of the Resource annotation.
     * <BR/>
     * //todo: annotate Resource for a mutator method (setFieldName)
     */
    private void injectResources(Object obj, String nodeName) {
        // DECLARED FIELDS ONLY!!!!
        for(Class clazz = obj.getClass(); clazz.getCanonicalName().startsWith("org.lsst") ; clazz = clazz.getSuperclass()) {
            Field[] fields = clazz.getDeclaredFields();
            for (Field field : fields) {
                Resource resource = field.getAnnotation(Resource.class);
                if (resource != null) {
                    String name = resource.name();
                    if (name == null || "".equals(name)) {
                        name = field.getName();
                    }
                    Object toInject = lookup.getComponentByName(name);
                    if (toInject == null) {
                        PackCst.CURLOG.warn("In " + nodeName + " : No component found for " + name);
                    } else {
                        try {
                            field.setAccessible(true);
                            field.set(obj, toInject);
                            PackCst.CURLOG.info(toInject + " object injected in " + name + "field  ");
                        } catch (IllegalAccessException e) {
                            PackCst.CURLOG.warn(nodeName + " No possible injection for " + name, e);
                        }
                    }
                }
            }
        }
    }
    
    /**
     * invokes the <TT>start()</TT> methods on all components (starting from the top main Module)
     */
    @Override
    public void doStart() {
        super.doStart();
        lookup.proceduralWalk(null, Configurable.class, Configurable::start, null);
    }

    /**
     * invokes the <TT>postStart()</TT> methods on all components (starting from the top main Module)
     * @throws org.lsst.ccs.HardwareException
     */
    @Override
    public void postStart() throws HardwareException {
        final HardwareException[] exc = new HardwareException[]{null};
        lookup.proceduralWalk(
                null,
                Configurable.class, 
                c -> {
                    try {
                        c.postStart();
                    } catch (HardwareException e) {
                        exc[0] = new HardwareException(c.toString() + " at postStart", e, exc[0]);
                    }
                }
                , null
                );
        if (exc[0] != null) {
            throw exc[0];
        }
    }

    /**
     * invokes the <TT>shutdownNow()</TT> methods on all components (starting from the top main Module)
     */
    @Override
    public void doShutdown() {
        mod_log.fine("Subsystem " + getName() + " shutdown starting");
        lookup.proceduralWalk(null, Configurable.class, Configurable::shutdownNow, null);
    }

    /**
     * forwards a HALT signal to all components (that are signalHandlers)
     */
    @org.lsst.ccs.command.annotations.Command(description = "halt", type = SIGNAL)
    @Override
    public void abort() {
        super.abort();
        sendSignal(new Signal(SignalLevel.HALT));
    }

    /**
     * same as abort but with a time span.
     *
     * @param expectedMaxDelay
     */
    @org.lsst.ccs.command.annotations.Command(description = "halt with expected max delay", type = SIGNAL)
    @Override
    public void abort(long expectedMaxDelay) {
        super.abort();
        sendSignal(new Signal(SignalLevel.HALT, expectedMaxDelay));
    }

    /**
     * forwards a STOP signal to all components (that are signalHandlers)
     */
    @org.lsst.ccs.command.annotations.Command(description = "stops with expected max delay", type = ACTION)
    @Override
    public void stop(long expectedMaxDelay) throws HardwareException {
        super.stop(expectedMaxDelay);
        sendSignal(new Signal(SignalLevel.STOP, expectedMaxDelay));
    }

    /**
     * same as stop(delay) but with an infinite delay
     *
     * @throws HardwareException
     */
    @org.lsst.ccs.command.annotations.Command(description = "stops hardware", type = ACTION)
    public void stop() throws HardwareException {
        //TODO check is this is handled gracefully by stop
        stop(Long.MAX_VALUE);
    }
    
    /**
     * Forwards a Signal to all components starting from the top node.
     * @param signal the signal to propagate.
     */
    private void sendSignal(final Signal signal){
        lookup.treeWalk(null, SignalHandler.class,
                sh -> {
                    return sh.signal(signal);
                },
                null);
    }

    /**
     * forwards a STOP signal to all components and then checks if all hardwares have been stopped within expectedMaxDelay
     *
     * @param expectedMaxDelay
     * @throws HardwareException, InterruptedException
     */
    @org.lsst.ccs.command.annotations.Command(description = "waits until all the hardware devices are actually stopped", type = ACTION)
    public void stopAndWait(long expectedMaxDelay) throws HardwareException, InterruptedException {
        // Stop returns immediately
        stop(expectedMaxDelay);
        // Now check if all hardware are stopped within expectedMaxDelay
        // Todo : discuss step : is there a smarter way to implement this ?
        int step = 10;
        long fracDelay = expectedMaxDelay / step;
        HardwareException lastExc = null;

        for (int i = 0; i < step; i++) {
            try {
                Thread.sleep(fracDelay);
                checkAllHardwareStopped();
                return;
            } catch (HardwareException e) {
                lastExc = e;
            }
        }
        throw lastExc;
    }

    /**
     * invoke a <TT>checkHardware</TT> on all <TT>HardwareController</TT>
     * (except when return diag modifies the path of invocation)
     *
     * @throws HardwareException a list of HardwareException
     */
    @Override
    public void checkHardware() throws HardwareException {
        final HardwareException[] exc = new HardwareException[]{null};
        lookup.treeWalk(
                null,
                HardwareController.class,
                hc -> {
                    try {
                        return hc.checkHardware();
                    } catch (HardwareException e) {
                        exc[0] = new HardwareException(hc.toString() + "not started ", e, exc[0]);
                        //todo: not correct: what if not fatal and we still want to return Handler-children
                        if (e.isFatal()) {
                            return TreeWalkerDiag.STOP;
                        }
                    }
                    return TreeWalkerDiag.GO;
                },
                null
        ); 
        if (exc[0] != null) {
            throw exc[0];
        }
    }

    /**
     * checks if all <TT>HardwareControllers</TT> have been started
     *
     * @throws HardwareException a list of exception fired by hardware that are not started
     */
    @Override
    public void checkAllHardwareStarted() throws HardwareException {
        final HardwareException[] exc = new HardwareException[]{null};
        lookup.proceduralWalk(
                null,
                HardwareController.class,
                hc -> {
                    try {
                        hc.checkStarted();
                    } catch (HardwareException e) {
                        exc[0] = new HardwareException(hc.toString() + "not started ", e, exc[0]);
                    }
                }, null
        );
        if (exc[0] != null) {
            throw exc[0];
        }

    }

    /**
     * checks if all <TT>HardwareControllers</TT> have been stopped
     *
     * @throws HardwareException a list of exception fired by hardware that are not stopped
     */
    // same code as above!
    @Override
    public void checkAllHardwareStopped() throws HardwareException {
        final HardwareException[] exc = new HardwareException[]{null};
        lookup.proceduralWalk(
                null,
                HardwareController.class,
                hc -> {
                    try {
                        hc.checkStopped();
                    } catch (HardwareException e) {
                        exc[0] = new HardwareException(hc.toString() + "not started ", e, exc[0]);
                    }
                }, null
        );
        if (exc[0] != null) {
            throw exc[0];
        }
    }
    
// -- Bulk Change Methods ------------------------------------------------------
     
    /**
     * Changes that have been submitted are processed as a group.
     */
    @Command(description = "processes the bulk change", type = Command.CommandType.CONFIGURATION)
    public void commitBulkChange() {
        try {
            validateBulkChanges();
        } catch (Exception ex) {
            throw ex;
        }
        Set<ParameterPath> processedParms = bulkChange();
        updateStateAndSendStatusConfigurationInfo("", processedParms);
        mod_log.info(processedParms.size() + " successfully set parameters");
    }

    @Override
    protected void validateBulkChanges() {
        // Test against the configuration validation methods
        lookup.proceduralNodeWalk(null, null,
                node -> {
                    String nodeName = node.getKey();
                    ParameterSetter parmSetter = parameterSetterDictionary.get(nodeName);
                    if (!parmSetter.hasSubmittedChanges()) return;
                    
                    Map<String, String> parametersView
                            = configurationProxy.getCurrentValuesForComponent(nodeName, Collections.emptySet());
                    parmSetter.invokeValidateBulkChange(parametersView, isInState(PhaseState.OPERATIONAL));
                });
    }
    
    /**
     * Processes the change of configurable parameters contained in the bulk
     * changes.
     * @return the processed parameters ordered by component name
     */
    @Override
    protected Set<ParameterPath> bulkChange() {
        updateInternalState("", ConfigurationState.CONFIGURING);
        // The set of processed parameters to be returned.
        Set<ParameterPath> processedParms = new HashSet<>();
        // procedural walk starting from the top component
        lookup.proceduralNodeWalk(null, null, new Consumer<EffectiveNode>() {
            
            public void accept(EffectiveNode node) {
                String nodeName = node.getKey();
                ParameterSetter parmSetter = parameterSetterDictionary.get(nodeName);
                try {
                    Map<String, String> processedParmsForComponent = parmSetter.invokeSetParameters();

                    for (Map.Entry<String, String> entry : processedParmsForComponent.entrySet()) {
                        configurationProxy.notifyParameterChange(
                                nodeName,
                                entry.getKey(),
                                entry.getValue()
                        );
                        processedParms.add(new ParameterPath(nodeName, "", entry.getKey()));
                    }
                } catch (BulkSettingException ex) {
                    if (ex.isFatal()) {
                        // cannot determine which parameters have been set and which have not
                        updateInternalState("", ConfigurationState.UNCONFIGURED);
                        throw ex;
                    } else {
                        Map<String, String> processedParmsForComponent = ex.getProcessedParms();
                        // notification to the configuration service
                        for (Map.Entry<String, String> entry : processedParmsForComponent.entrySet()) {
                            configurationProxy.notifyParameterChange(
                                    nodeName,
                                    entry.getKey(),
                                    entry.getValue()
                            );
                            processedParms.add(new ParameterPath(nodeName, "", entry.getKey()));
                        }
                        updateStateAndSendStatusConfigurationInfo(ex.getMessage(), processedParms);
                        // stop the tree walking by throwing a runtime exception
                        throw ex;
                    }
                }
            }
        });
        return processedParms;
    }
    
}
