package org.lsst.ccs.startup;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.config.ConfigurableSubsystem;
import org.lsst.ccs.config.ConfigurationProxy;
import org.lsst.ccs.description.ComponentLookupService;
import org.lsst.ccs.framework.Configurable;
import org.lsst.ccs.framework.ConfigurationServiceException;
import org.lsst.ccs.framework.Signal;
import org.lsst.ccs.framework.SignalHandler;
import org.lsst.ccs.utilities.conv.TypeConversionException;
import org.lsst.ccs.utilities.structs.TreeBranch;
import org.lsst.ccs.utilities.structs.ViewValue;

/**
 * instances of this class are created by the subsystem startup.
 * They operate delegations on behalf of the <TT>Configurable</TT> Object.
 * The <TT>Configurable</TT> Object can thus access its <TT>ConfigurationProxy</TT>
 * and its <TT>ComponentLookupService</TT>
 */
public class ConfigurationEnvironment implements Configurable.Environment {
    /**
     * name of the object as known to the description. (for Modules it is a duplicate
     * of their name.)
     */
    private final String name;

    /**
     * a reference to the Configuration Proxy
     */
    private final ConfigurationProxy configurationProxy;
    /**
     * a reference to the <TT>ComponentLookupService</TT>
     */
    private final ComponentLookupService lookupService;
    
    /**
     * a reference to the subsystem
     */
    private final ConfigurableSubsystem subsystem;

    /**
     * Creates an Environment for a Configurable object.
     * It builds a CommandDictionary for the object and populates a map of ConfigChanger methods.
     * <P/>
     * The configuration proxy provides an access to the configuration database but interactions
     * are of delicate nature :
     * <UL>
     * <LI/> the base is checked as wether the new parameter value is considered possible
     * <LI/> then the parameter is changed locally
     * <LI/> the base is notified about the change
     * </UL>
     *
     * @param name
     * @param configurationProxy
     * @param lookupService
     * @param s
     */
    public ConfigurationEnvironment(String name, ConfigurationProxy configurationProxy,
            ComponentLookupService lookupService, ConfigurableSubsystem s) {
        this.name = name;
            // REVIEW : Seems pointless : the attempt is to access @configChanger annotated
        // methods in the delegate but @ConfigChanger annotation is not accessible
        // at the driver level : there cannot be @ConfigChanger annotated methods in 
        // the delegate
        this.configurationProxy = Objects.requireNonNull(configurationProxy);
        this.lookupService = Objects.requireNonNull(lookupService);
        this.subsystem = Objects.requireNonNull(s);
    }
            //TODO: create a command Method map (same idea as for configMethod) so you can invoke a method with a simpleName (and publish)

    /**
     * performs a parameter change during an engineering session
     *
     * @param parameterName name of parameter for the current object.
     * @param value any object that is fit for the configuration system (see documentation)
     * @throws IllegalArgumentException if parameter has a name unknown to the configuration
     * @throws Exception that can be thrown by the <TT>ConfigurationProxy</TT>
     * (no connection, illegal format, constraints failure) or by the user's code
     * (preconditions)
     *
     */
    @Override
    public void change(String parameterName, Object value) throws Exception {
        if (!isInEngineeringMode()) {
            throw new IllegalStateException("Configuration actions are accepted only in engineering mode");
        }
        getConfigurableSubsystem().change(this.name, parameterName, value);
    }

    @Override
    public void submitChange(String parameterName, Object value) {
        if (!isInEngineeringMode()) {
            throw new IllegalStateException("Configuration actions are accepted only in engineering mode");
        }
        try {
            getConfigurableSubsystem().submitChange(this.name, parameterName, value);
        } catch (TypeConversionException ex) {
            throw new IllegalArgumentException(ex.getMessage(), ex);
        }
    }
    
    /**
     * Saves all changes in their current configuration.
     *
     * @throws org.lsst.ccs.framework.ConfigurationServiceException
     */
    @Override
    public void saveAllChanges() throws ConfigurationServiceException {
        if (!isInEngineeringMode()) {
            throw new IllegalStateException("Configuration actions are accepted only in engineering mode");
        }
        getConfigurableSubsystem().saveAllChanges();
    }

    /**
     * Changes made in the specified categories are saved under the current configuration
     * for this category, changes on parameters that belong to other categories
     * are left unchanged.
     *
     * @param categories a list of categories
     * @throws org.lsst.ccs.framework.ConfigurationServiceException
     */
    @Override
    public void saveChangesForCategories(String... categories) throws ConfigurationServiceException {
        if (!isInEngineeringMode()) {
            throw new IllegalStateException("Configuration actions are accepted only in engineering mode");
        }
        getConfigurableSubsystem().saveChangesForCategories(categories);
    }

    /**
     * Changes made in the specified categories are saved under the newly specified
     * name for this category, changes on parameters that belong to other categories
     * are left unchanged.
     *
     * @param taggedCategories a list of pairs categoryName:configurationName
     * @throws org.lsst.ccs.framework.ConfigurationServiceException
     */
    @Override
    public void saveChangesForCategoriesAs(String... taggedCategories) throws ConfigurationServiceException {
        if (!isInEngineeringMode()) {
            throw new IllegalStateException("Configuration actions are accepted only in engineering mode");
        }
        getConfigurableSubsystem().saveChangesForCategoriesAs(taggedCategories);
    }

    /**
     * All unsaved changes are dropped, each parameter goes back to the value specified
     * by the current configuration for this category.
     * The configuration context is no longer active.
     *
     * @throws Exception
     */
    @Override
    public void dropAllChanges() throws Exception {
        if (!isInEngineeringMode()) {
            throw new IllegalStateException("Configuration actions are accepted only in engineering mode");
        }
        getConfigurableSubsystem().dropAllChanges();
    }

    /**
     * Changes occurring for one of the mentioned categories in categories are dropped,
     * other categories are left untouched.
     * The configuration context remains active.
     *
     * @param categories A list of categories
     * @throws Exception
     */
    @Override
    public void dropChangesForCategories(String... categories) throws Exception {
        if (!isInEngineeringMode()) {
            throw new IllegalStateException("Configuration actions are accepted only in engineering mode");
        }
        getConfigurableSubsystem().dropChangesForCategories(categories);
    }

    private boolean isInEngineeringMode() {
        return getSubsystem().get().isInEngineeringMode();
    }

    /**
     * Checks if a value can be accepted by the Configuration system.
     * to be used by "facade" commands that may set a value.
     * <P/>
     * The returned value is an Object that represent the real value as
     * generated by the Configuration: it is not necessarily the value passed
     * as argument because the argument can both be a real object or its String
     * representation (as extracted from a properties file or from a GUI interaction).
     * This returned object contains both this real value AND its String representation.
     *
     * @param parameterName
     * @param value can be a real Object or its String representation
     * @return An object <TT>ViewValue</TT> that returns both an Object value and
     * its String representation as used by the Configuration.
     * @throws Exception
     * @deprecated use isParameterConfigurable(String parameterName) instead, the return
     * value is irrelevant.
     */
    @Override
    @Deprecated
    public ViewValue getCheckedValueFromConfiguration(String parameterName, Object value)
            throws Exception {
        if (!isParameterConfigurable(parameterName))
            throw new IllegalArgumentException("incoherent parameter name for " + parameterName + "-> " + getNameOfComponent());
        return new ViewValue(parameterName,parameterName);
    }
    
    @Override
    public Boolean isParameterConfigurable(String parameterName) {
        return configurationProxy.isParameterConfigurable(this.name, parameterName);
    }
    
    /**
     * to be used by "facade" commands that may set a value
     * then notify the configuration without going through the preliminary check
     * of the configuration service.
     * <BR/>
     * <B>use at your own risk!</B>
     *
     * @param parameterName
     * @param value
     * @throws Exception
     */
    @Override
    public void notifyChangeWithoutPreliminaryChecks(String parameterName, Object value) throws Exception {
        configurationProxy.notifyUncheckedParameterChange(this.name, parameterName, value);
    }

    /**
     * to be used by "facade" commands that may set a value
     * <BR>
     * the calling must have made a preliminary check with the configuration service.
     * <BR/>
     * <B>indiscriminate use should be discouraged</B>
     *
     * @param parameterName
     * @param value
     * @throws Exception
     */
    @Override
    public void notifyChange(String parameterName, String value) throws Exception {
        configurationProxy.notifyParameterChange(this.name, parameterName, value);
    }

    /**
     * request a Component from a Description by its name.
     *
     * @param name
     * @return A Component. <B>beware</B> unless you are dead sure do not assume this component is
     * a <TT>Configurable</TT> object (any object can be aliased and so put in the dictionary)
     */
    @Override
    public Object getComponentByName(String name) {
        return lookupService.getComponentByName(name);
    }

    /**
     * get the children of the {@code Configurable} object that are assignable
     * to an instance of the class argument.
     *
     * @param classFilter
     * @param <T>
     * @return a map with name of the component as key and its class as value.
     */
    @Override
    public <T> LinkedHashMap<String, T> getChildren(Class<T> classFilter) {
        Objects.requireNonNull(classFilter);
        return lookupService.getChildren(this.name, classFilter);
    }

    /**
     * returns the parent of the current component
     *
     * @return the Component (value) and its name (key)
     * null if there is no parent.
     */
    @Override
    public Map.Entry<String, Object> getParent() {
        return lookupService.getParent(this.name);
    }

    /**
     *
     * @return the name associated with this component
     */
    @Override
    public String getNameOfComponent() {
        return name;
    }

    @Override
    public Optional<Subsystem> getSubsystem() {
        return Optional.of(subsystem);
    }
    
    private ConfigurableSubsystem getConfigurableSubsystem() {
        return (ConfigurableSubsystem) getSubsystem().get();
    }
    
    
    @Override
    public void dropBulkChange() {
        getConfigurableSubsystem().dropSubmittedChangesForComponent(name);
    }

    @Override
    public TreeBranch<String> getComponentTree() {
        final List<TreeBranch<String>> otree = new ArrayList<>(1);
        otree.add(null);
        lookupService.proceduralNodeWalk(
                name,
                // precode
                node -> {
                    //you become the parent of your children
                    // but if you are a leaf the postcode reinstates the previous parent
                    // so your "brothers" (and sisters) can use it
                    otree.add(0, new TreeBranch<>(otree.remove(0), node.getName()));
                },
                // post code
                node -> {
                    TreeBranch<String> parent = otree.get(0).getRealParent();
                    if (parent != null) {
                        otree.remove(0);
                        otree.add(parent);
                    }
                }
        );
        return otree.get(0);
    }

    @Override
    public void sendSignal(Signal signal) {
        lookupService.treeWalk(name, SignalHandler.class,
                sh -> {
                    return sh.signal(signal);
                },
                null);
    }

    @Override
    public Map printConfigurableParameters(String[] categories) {
        return configurationProxy.getCurrentValuesForComponent(name, new HashSet<>(Arrays.asList(categories)));
    }

    @Override
    public Map<String, String> getSubmittedChanges() {
        return getConfigurableSubsystem().getSubmittedChangesForComponent(name);
    }
}
