package org.lsst.ccs.startup;

import org.lsst.ccs.config.ConfigurationProxy;
import org.lsst.ccs.description.ComponentLookupService;
import org.lsst.ccs.config.ConfigurableSubsystem;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.framework.*;
import org.lsst.ccs.utilities.structs.ViewValue;
import org.lsst.ccs.description.EffectiveNode;

import java.util.*;
import org.lsst.ccs.bus.messages.StatusHeartBeat;
import org.lsst.ccs.bus.states.ConfigurationState;
import org.lsst.ccs.command.CommandSet;
import org.lsst.ccs.command.CommandSetBuilder;

import static org.lsst.ccs.command.annotations.Command.CommandType.SIGNAL;
import static org.lsst.ccs.command.annotations.Command.CommandType.ACTION;
import org.lsst.ccs.config.ConfigurationEnvironment;

/**
 * the top modular subsystem that deals with configuration proxy and lookup service
 *
 * @author bamade
 */
public class NodeModularSubsystem extends ConfigurableSubsystem {

    private final ComponentLookupService lookup;

    /**
     * Builds a NodeModularSubsystem out of a SubsystemDescription object.
     * Usage of this constructor supposes that a subsystem description has already been built upstream (or fetched from a database)
     *
     * @param subsystemName the name of the subsystem on the buses
     * @param configurationProxy the configuration proxy for this subsystem
     * @param effectiveNode the tree structure of component
     * @param initialState the initial configuration state
     * @throws Exception
     */
    NodeModularSubsystem(String subsystemName, ConfigurationProxy configurationProxy, EffectiveNode effectiveNode, ConfigurationState initialState) throws Exception {
        super(subsystemName, configurationProxy, initialState);

        //todo : create a Context if needed
        this.lookup = new ComponentLookupService(effectiveNode);
        // populates the componentList and sets the environment of each component
//        realRegisterNodes(effectiveNode);
        lookup.proceduralNodeWalk(
                null,
                null,
                node -> {
                    String key = node.getKey();
                    Object value = node.getRealValue();
                    if (value instanceof Configurable) {
                        Configurable configurable = (Configurable) value;
                        configurable.setEnvironment(new ConfigurationEnvironment(key, configurable, getConfigurationProxy(), lookup, this));
                    }
                });
        
        // invokes init on all Modules
        doInitComponents();
        //We have to provide here the full configuration name
//        try {
//            configurationInfo = new ConfigurationInfo(profile.getName());
//        } catch (Exception e) {
//            e.printStackTrace();
//        }

    }

    //Add the ConfigurationInfo the the StatusHeartBeat
    @Override
    protected void updateHeartBeat(StatusHeartBeat s) {
        //       s.addObject(configurationInfo);
    }

    @Override
    public ComponentLookupService getLookup() {
        return lookup;
    }

    /**
     * *
     *
     * @return a List of [name-component] in PreOrder.
     * @deprecated unused.
     */
    @Deprecated
    public List<ViewValue> getPreOrderComponentList() {
        return Collections.EMPTY_LIST;
    }
    
    private void doInitComponents() {
        lookup.proceduralWalk(null, Configurable.class, Configurable::init, null);
        
        // Provide the Subsystem with the named CommandSets.
        lookup.proceduralNodeWalk(
                null,
                node -> {
                    CommandSet commandSet = new CommandSetBuilder().buildCommandSet(node.getRealValue());
                    String fullPath = lookup.getFullPathFor(node.getKey()); 
                    addCommandSet(fullPath, commandSet);
                }, 
                null
        );
    }
    
    /**
     * invokes the <TT>start()</TT> methods on all components (starting from the top main Module)
     */
    @Override
    public void doStart() {
        super.doStart();
        lookup.proceduralWalk(null, Configurable.class, Configurable::start, null);
    }

    /**
     * invokes the <TT>postStart()</TT> methods on all components (starting from the top main Module)
     */
    @Override
    public void postStart() throws HardwareException {
        final HardwareException[] exc = new HardwareException[]{null};
        lookup.proceduralWalk(
                null,
                Configurable.class, 
                c -> {
                    try {
                        c.postStart();
                    } catch (HardwareException e) {
                        exc[0] = new HardwareException(c.toString() + " at postStart", e, exc[0]);
                    }
                }
                , null
                );
        if (exc[0] != null) {
            throw exc[0];
        }
    }

    /**
     * invokes the <TT>shutdownNow()</TT> methods on all components (starting from the top main Module)
     */
    @Override
    public void doShutdown() {
        log.fine("Subsystem " + getName() + " shutdown starting");
        lookup.proceduralWalk(null, Configurable.class, Configurable::shutdownNow, null);
    }

    /**
     * utility method to find a Component out of a Command destination
     *
     * @param commandDestination
     * @return a {@code Configurable}
     * @deprecated unused
     */
    protected Configurable getCommandDestination(String commandDestination) {
        Configurable configurable = null;
        String configurableName = "main";

        if (commandDestination.contains("/")) {
            configurableName = commandDestination
                    .substring(commandDestination.indexOf("/") + 1);
        }
        configurable = (Configurable) lookup.getComponentByName(configurableName);
        if (configurable == null) {
            throw new IllegalArgumentException(" no such configurable " + configurableName + " for command destination");
        }
        return configurable;
    }

    /**
     * forwards a HALT signal to all components (that are signalHandlers)
     */
    @org.lsst.ccs.command.annotations.Command(description = "halt", type = SIGNAL)
    @Override
    public void abort() {
        super.abort();
        sendSignal(new Signal(SignalLevel.HALT));
    }

    /**
     * same as abort but with a time span.
     *
     * @param expectedMaxDelay
     */
    @org.lsst.ccs.command.annotations.Command(description = "halt with expected max delay", type = SIGNAL)
    @Override
    public void abort(long expectedMaxDelay) {
        super.abort();
        sendSignal(new Signal(SignalLevel.HALT, expectedMaxDelay));
    }

    /**
     * forwards a STOP signal to all components (that are signalHandlers)
     */
    @org.lsst.ccs.command.annotations.Command(description = "stops with expected max delay", type = ACTION)
    @Override
    public void stop(long expectedMaxDelay) throws HardwareException {
        super.stop(expectedMaxDelay);
        sendSignal(new Signal(SignalLevel.STOP, expectedMaxDelay));
    }

    /**
     * same as stop(delay) but with an infinite delay
     *
     * @throws HardwareException
     */
    @org.lsst.ccs.command.annotations.Command(description = "stops hardware", type = ACTION)
    public void stop() throws HardwareException {
        //TODO check is this is handled gracefully by stop
        stop(Long.MAX_VALUE);
    }
    
    /**
     * Forwards a Signal to all components starting from the top node.
     * @param signal the signal to propagate.
     */
    private void sendSignal(final Signal signal){
        lookup.treeWalk(null, SignalHandler.class,
                sh -> {
                    return sh.signal(signal);
                },
                null);
    }

    /**
     * forwards a STOP signal to all components and then checks if all hardwares have been stopped within expectedMaxDelay
     *
     * @param expectedMaxDelay
     * @throws HardwareException, InterruptedException
     */
    @org.lsst.ccs.command.annotations.Command(description = "waits until all the hardware devices are actually stopped", type = ACTION)
    public void stopAndWait(long expectedMaxDelay) throws HardwareException, InterruptedException {
        // Stop returns immediately
        stop(expectedMaxDelay);
        // Now check if all hardware are stopped within expectedMaxDelay
        // Todo : discuss step : is there a smarter way to implement this ?
        int step = 10;
        long fracDelay = expectedMaxDelay / step;
        HardwareException lastExc = null;

        for (int i = 0; i < step; i++) {
            try {
                Thread.sleep(fracDelay);
                checkAllHardwareStopped();
                return;
            } catch (HardwareException e) {
                lastExc = e;
            }
        }
        throw lastExc;
    }

    /**
     * invoke a <TT>checkHardware</TT> on all <TT>HardwareController</TT>
     * (except when return diag modifies the path of invocation)
     *
     * @throws HardwareException a list of HardwareException
     */
    @Override
    public void checkHardware() throws HardwareException {
        final HardwareException[] exc = new HardwareException[]{null};
        lookup.treeWalk(
                null,
                HardwareController.class,
                hc -> {
                    try {
                        return hc.checkHardware();
                    } catch (HardwareException e) {
                        exc[0] = new HardwareException(hc.toString() + "not started ", e, exc[0]);
                        //todo: not correct: what if not fatal and we still want to return Handler-children
                        if (e.isFatal()) {
                            return TreeWalkerDiag.STOP;
                        }
                    }
                    return TreeWalkerDiag.GO;
                },
                null
        ); 
        if (exc[0] != null) {
            throw exc[0];
        }
    }

    /**
     * checks if all <TT>HardwareControllers</TT> have been started
     *
     * @throws HardwareException a list of exception fired by hardware that are not started
     */
    @Override
    public void checkAllHardwareStarted() throws HardwareException {
        final HardwareException[] exc = new HardwareException[]{null};
        lookup.proceduralWalk(
                null,
                Configurable.class,
                configurable -> {
                    if (configurable instanceof HardwareController) {
                        try {
                            ((HardwareController) configurable).checkStarted();
                        } catch (HardwareException e) {
                            exc[0] = new HardwareException(configurable.toString() + "not started ", e, exc[0]);
                        }
                    }
                }, null
        );
        if (exc[0] != null) {
            throw exc[0];
        }

    }

    /**
     * checks if all <TT>HardwareControllers</TT> have been stopped
     *
     * @throws HardwareException a list of exception fired by hardware that are not stopped
     */
    // same code as above!
    @Override
    public void checkAllHardwareStopped() throws HardwareException {
        final HardwareException[] exc = new HardwareException[]{null};
        lookup.proceduralWalk(
                null,
                Configurable.class,
                configurable -> {
                    if (configurable instanceof HardwareController) {
                        try {
                            ((HardwareController) configurable).checkStopped();
                        } catch (HardwareException e) {
                            exc[0] = new HardwareException(configurable.toString() + "not started ", e, exc[0]);
                        }
                    }
                }, null
        );
        if (exc[0] != null) {
            throw exc[0];
        }
    }

}
