package org.lsst.ccs.startup;

import org.lsst.ccs.CCSCst;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.config.*;
import org.lsst.ccs.description.EffectiveNode;
import org.lsst.gruth.nodes.ComponentFactory;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import org.lsst.ccs.bus.states.ConfigurationState;
import org.lsst.ccs.config.ConfigurableSubsystem;
import org.lsst.ccs.config.ConfigurationProxy;
import org.lsst.ccs.config.utilities.ConfigUtils;
import org.lsst.ccs.description.ComponentNodeBuilder;
import org.lsst.ccs.description.DescriptiveNode;
import org.lsst.gruth.jutils.Constraints;
import org.lsst.gruth.jutils.HollowParm;
import org.lsst.gruth.jutils.NamedRefParm;

/**
 * A collection of static utilities linked to boot operations.
 * Note that:
 * <UL>
 * <LI>
 * a descriptive node represents a configuration description
 * <LI/>
 * an effective node is the real subsystem and module hierarchy (ready to run)
 * <LI/>
 * a "raw" subsystem is without configuration data
 * </UL>
 *
 * @author bamade
 */
// Date: 05/06/12
public class BootUtils {

    /** *
     * Very important class: when an instance is created you can have :
     * <UL>
     * <LI/> a descriptive Node
     * <LI/> the names of subsystem, configuration, tag
     * <LI/> the corresponding Effective Node
     * <LI/> a Subsystem ready to run (but not started)
     * </UL>
     */
    public static class LocalBootObject {

        /** *
         * this is the initial descriptiveNode: not the one
         * modified by the Configuration
         */
        DescriptiveNode descriptiveNode;
        DescriptiveNode modifiedDescriptiveNode;
        EffectiveNode effectiveNode;
        ConfigurableSubsystem subsystem;
        private final String subsystemName;
        private final String[] taggedCategories;
        private String tagName;

        /** *
         * creates a boot Object from a description file name
         *
         * @param pathName
         * @throws Exception
         */
        public LocalBootObject(String pathName) throws Exception {
            this(pathName, "");
        }

        /** *
         * creates a boot Object from a description file name and
         * a configuration property file.
         * The values in the configuration file override the parameters set in
         * the description File.
         *
         * @param fullDescription
         * @param config can be a file name OR the config name only
         * @throws Exception
         */
        public LocalBootObject(String fullDescription, String config) throws Exception {
            this(fullDescription, config, null);
        }

        public LocalBootObject(String fullDescription, String config, String subsystemAlias) throws Exception {
            if (config == null) {
                config = "";
            }
            if (config.contains("/")) {
                throw new IllegalArgumentException("configuration files must be located at the root of a resource directory");
            }
            if (config.contains(".")) {
                throw new IllegalArgumentException("configuration input must be the name of the configuration only");
            }

            // First build a DescriptiveNode from the provided fullDescription
            descriptiveNode = ComponentNodeBuilder.buildDescriptiveNode(fullDescription);

            /*
             * Resolving : 
             * - the subsystem name
             * - the configuration name
             * - the tag name
             */
            subsystemName = (subsystemAlias == null) ? descriptiveNode.getSubsystemName() : subsystemAlias;
            taggedCategories = config.split(",");

            //TODO: Why do we need this variable? Can it be removed or can we use "fullDescription"?
            tagName = fullDescription;
            if (fullDescription.contains(":")) {
                tagName = fullDescription.substring(fullDescription.indexOf(':') + 1);
            }

            CCSCst.LOG_TODO.fine("TODO: Configuration proxy should also be able to interact with remote database (switch between local and remote?)");

            /*
             creates a SubsystemDescription Object from the descriptive node
             the parameters that are tagged "name" and that are References to other
             components are not included in the list of Parameters that can be modified
             */
            SubsystemDescription subsystemDescription = buildSubsystemDescription(subsystemName, tagName,
                    descriptiveNode, DataFlavour.PURE_OBJECT_TREE);

            // The configuration proxy that will be associated to the configurable subsystem
            ConfigurationProxy proxy = new LocalConfigurationProxy(subsystemDescription);

            /** * Create the ModifiedDescriptiveNode ** */
            // First step : get the values for startup configuration
            Map.Entry<ConfigurationState, Set<ParameterConfiguration>> stateAndParmConfigs
                    = proxy.getInitialParameterConfigurations(ConfigUtils.parseConfigurationStringWithDefaults(subsystemDescription.getCategorySet(), taggedCategories));
            // Second step : build the modified descriptive node            
            modifiedDescriptiveNode = modifyDescriptiveNode(descriptiveNode, stateAndParmConfigs.getValue());

            /** * Create the effectiveNode ** */
            effectiveNode = BootUtils.buildEffectiveNode(modifiedDescriptiveNode);

            //Build NodeModularSubsystem from effectiveNode
            subsystem = new NodeModularSubsystem(subsystemName, proxy, effectiveNode, stateAndParmConfigs.getKey());
        }

        /** *
         * Gets the initial descriptive node, ie the DescriptiveNode that is
         * built out of the description file only.
         *
         * @return a {@code DescriptiveNode} object
         */
        public DescriptiveNode getDescriptiveNode() {
            return descriptiveNode;
        }

        /** *
         * returns the descriptive node as modified by configuration.
         * (could be identical to initial DescriptiveNode)
         *
         * @return a {@code DescriptiveNode} object
         */
        public DescriptiveNode getModifiedDescriptiveNode() {
            return modifiedDescriptiveNode;
        }

        public Subsystem getSubsystem() {
            return subsystem;
        }

        public String getSubsystemName() {
            return subsystemName;
        }

        public String getTagName() {
            return tagName;
        }

        /** *
         * Gets a reference to a component built by the description file.
         * <B>beware</B> if the subsystem is not started the object may not
         * have been completely initialized (the start() method of the subsystem
         * invokes various initialization)
         *
         * @param key the name of the component.
         * @return a reference to a component that fits the name "key"
         */
        // REVIEW : this methods seems unnecessary
        public Object getRunningObject(String key) {
            return effectiveNode.getIndirect(key);
        }

    }
    ////////////////////// END LOCAL BOOT OBJECT

    /**
     * creates a new {@code SubsystemDescription} populated with "empty" {@code ParameterDescriptions}.
     * The object is not in the database since the list of descriptions should be modified first.
     *
     * @param subsystemName
     * @param tagName
     * @param descriptiveNode
     * @param dataFlavour
     * @return a {@code SubsystemDescription} object containing the description of the parameters
     * as described in {@code descriptiveNode}.
     */
    public static SubsystemDescription buildSubsystemDescription(String subsystemName, String tagName, DescriptiveNode descriptiveNode, DataFlavour dataFlavour) {
        SubsystemDescription res = Factories.createRawSubsystemDescription(subsystemName, tagName, descriptiveNode, dataFlavour);
        ArrayList<ParameterDescription> list = new ArrayList<>();
        populateParameterDescriptionsFromTop(list, descriptiveNode);
        res.addParameterDescriptions(list);
        return res;
    }

    private static void populateParameterDescriptionsFromTop(Collection<ParameterDescription> list,
            DescriptiveNode topNode) {
        list.addAll(parameterDescriptionsFromNode(topNode));
        ArrayList<DescriptiveNode> children = topNode.getChildren();
        for (DescriptiveNode childNode : children) {
            populateParameterDescriptionsFromTop(list, childNode);
        }
    }

    /**
     * Creates a collection of ParameterDescription objects out of the attributes
     * of {@code node}.
     *
     * @param node
     * @param maxLevel
     * @return a Collection of ParameterDescription.
     */
    private static Collection<ParameterDescription> parameterDescriptionsFromNode(DescriptiveNode node) {
        Collection<ParameterDescription> res = new ArrayList<>();
        // Looking for configurable parameters defined in the description node.
        Map attributes = node.getAttributes();
        String nodeKey = node.getKey();
        //TODO: handle positional parameters!!
        //TODO: handle other methods (not only constructors)
        if (attributes != null) {
            Set<Map.Entry> keyVals = attributes.entrySet();
            for (Map.Entry entry : keyVals) {
                Object val = entry.getValue();
                String parmName = (String) entry.getKey();

                if ("name".equalsIgnoreCase(parmName) || val instanceof NamedRefParm) {
                    continue;
                }

                //TODO: handle structParm
                if (val instanceof HollowParm) {
                    HollowParm hollow = (HollowParm) val;
                    if (!hollow.isReadOnly()) {
                        // first get the type
                        String typeName = hollow.getValueClass().getName();
                        ParameterBase base = new ParameterBase(nodeKey, "", parmName, typeName, hollow.toString());
                        String description = "";
                        String simpleName = "";
                        String constraints = "";
                        String category = "";
                        boolean notModifiableAtRuntime = false;
                        int level = 0;
                        Properties props = hollow.getProperties();
                        if (props != null) {
                            // description
                            description = props.getProperty("description", "");
                            // simpleName
                            simpleName = props.getProperty("simpleName", "");
                            // constraints
                            constraints = props.getProperty("constraints", "");
                            // check default value against constraints
                            Constraints.check(typeName, base.getDefaultValue(), constraints);
                            // category
                            category = props.getProperty("category", "");
                            // level
                            String request = props.getProperty("level");
                            if (request != null) {
                                level = Integer.parseInt(request);
                            }
                            String isStatic = props.getProperty("static");
                            if (isStatic != null) {
                                notModifiableAtRuntime = Boolean.valueOf(isStatic);
                            }
                        }
                            AParameterDescription parmDescription = new AParameterDescription(base, description, simpleName, constraints, category, level);
                            parmDescription.setNotModifiableAtRuntime(notModifiableAtRuntime);
                            res.add(parmDescription);
                    }
                }
            }
        }
        return res;
    }

    /**
     * Builds an effective node from a description node
     *
     * @param descriptionNode
     * @return an {@code EffectiveNode} object
     * @throws InvocationTargetException
     * @throws NoSuchMethodException
     * @throws IllegalAccessException
     */
    // argument is of type DescriptiveNode result of type EffectiveNode
    static EffectiveNode buildEffectiveNode(DescriptiveNode descriptionNode) throws InvocationTargetException, NoSuchMethodException, IllegalAccessException {
        ComponentFactory fact = new ComponentFactory(descriptionNode);
        EffectiveNode top = fact.build();
        return top;
    }

    /**
     * Builds a {@code Subsystem} instance out of a description file.
     *
     * @param descriptionName
     * @return a subsystem ready to run.
     * @throws Exception
     */
    public static Subsystem getSubsystemFromFile(String descriptionName) throws Exception {
        return getSubsystemFromFile(descriptionName, null);
    }

    /**
     * Builds a {@code Subsystem} given a description name and an initial configuration.
     *
     * @param descriptionName the name of the subsystem description
     * @param configName the name of the configuration the subsystem is started with
     * @return a ready to run subsystem
     * @throws Exception
     */
    public static Subsystem getSubsystemFromFile(String descriptionName, String configName) throws Exception {
        return getSubsystemFromFile(descriptionName, configName, null);
    }

    /**
     * Builds a {@code Subsystem} given a description name, an initial configuration
     * and a alias. The subsystem will be named "alias" on the buses.
     *
     * @param pathName
     * @param propertiesFileName
     * @param subsystemAlias
     * @return A ready to run subsystem
     * @throws Exception
     */
    public static Subsystem getSubsystemFromFile(String pathName, String propertiesFileName, String subsystemAlias) throws Exception {
        return new LocalBootObject(pathName, propertiesFileName, subsystemAlias).getSubsystem();
    }

    /**
     * saves the latest run configuration in a local cache.
     * This is saved in a file (see naming convention handled by <TT>baseNameFromNames</TT>
     * that will reside in tha cache directory.
     * The name of this file will be reported in another text file in this directory named "latest.txt".
     * <p/>
     * TODO: The path of the cache directory is to be determined.
     *
     * @param node
     * @param subsystemName
     * @param configName
     * @param tag
     */
    public static void saveInCache(DescriptiveNode node, String subsystemName,
            String configName, String tag) {
        //TODO: implement saveInCache
    }

    public static DescriptiveNode getLatestInCache() {
        return null;
    }

    public static void bootFromCache() {
        throw new UnsupportedOperationException("bootFromCache");
    }

    /**
     * Modifies the node in argument with respect to the set of ParameterConfiguration
     * TODO : from the ParameterConfiguration objects we only need their name and current value.
     *
     * @param node
     * @param parmConfigs
     * @return a modified descriptive node
     */
    public static DescriptiveNode modifyDescriptiveNode(DescriptiveNode node, Set<? extends ParameterConfiguration> parmConfigs) {
        DescriptiveNode res = node.clone();

        for (ParameterConfiguration parameterConfiguration : parmConfigs) {
            ParameterPath path = parameterConfiguration.getPath();
            String componentName = path.getComponentName();
            String codeName = path.getCodeName();
            if (codeName != null && !"".equals(codeName)) {
                throw new UnsupportedOperationException(" no change on methods yet --> " + codeName);
            }
            String parameterName = path.getParameterName();
            DescriptiveNode goalComponent = (DescriptiveNode) res.getNodeByName(componentName);
            if (goalComponent == null) {
                throw new IllegalArgumentException("no component for name :" + componentName);
            }
            Map mapAttributes = goalComponent.getAttributes();
            // not null
            if (mapAttributes == null) {
                throw new IllegalArgumentException("incompatible attribute list for component//parameter " + path);
            }
            // get Parameter
            Object rawParm = mapAttributes.get(parameterName);
            if (rawParm instanceof HollowParm) {
                HollowParm hollow = (HollowParm) rawParm;
                hollow.modifyChecked(parameterConfiguration.getConfiguredValue());
            } else {
                throw new IllegalArgumentException("parameter not modifiable" + rawParm);
            }
        }
        return res;
    }

    /**
     * To be invoked before registering a <TT>ConfigProfile</TT>
     * This method creates a real dummy <TT>EffectiveNode</TT> to check for
     * parameters that have preconditions.
     * This means that objects created this way should not affect the hardwares.
     *
     * @param profile
     */
    public static void checkStaticCompatibleConfiguration(ConfigProfile profile) throws Exception {
        DescriptiveNode node = (DescriptiveNode) profile.getSubsystemDescription().getDescriptionData();
        Set<? extends ParameterConfiguration> parmConfigs = profile.getModifiedParameters();

        DescriptiveNode modifiedDescriptiveNode = modifyDescriptiveNode(node, parmConfigs);
        buildEffectiveNode(modifiedDescriptiveNode);
    }

}
