package org.lsst.ccs.startup;

import org.lsst.ccs.HardwareException;
import org.lsst.ccs.Mode;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.Command;
import org.lsst.ccs.command.CommandInvocationException;
import org.lsst.ccs.command.LocalCommandDictionary;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.config.*;
import org.lsst.ccs.config.PackCst;
import org.lsst.ccs.framework.*;
import org.lsst.ccs.utilities.functions.Cocoon;
import org.lsst.ccs.utilities.functions.MutableReference;
import org.lsst.ccs.utilities.structs.ViewValue;
import org.lsst.gruth.jutils.EffectiveNode;

import java.io.IOException;
import java.util.*;

import static org.lsst.ccs.command.annotations.Command.CommandType.ABORT;
import static org.lsst.ccs.command.annotations.Command.CommandType.ACTION;
import org.lsst.ccs.config.utilities.ConfigUtils;

/**
 * the top modular subsystem that deals with configuration proxy and lookup service
 *
 * @author bamade
 */
public class NodeModularSubsystem extends Subsystem {
    //public class NodeModularSubsystem extends BasicModularSubSystem {

    private ComponentLookupService lookup;
    private Configurable mainObject;

    //todo: suppress
    private Context context;

    //TODO: remove configurationproxy from subsystem and move it to this class

    /**
     * this list is built to get an iterator that traverses the tree of components
     * in preOrder: it can be uses when we want to call methods that do not rely
     * on TreeWalkerDiag results.
     * Beware: if any method fails then the tree traversal fails!
     */
    private ArrayList<ViewValue> preOrderComponentList = new ArrayList<>();
    private List<ViewValue> preOrderUnmodifiableView = Collections.unmodifiableList(preOrderComponentList);

    /**
     * @param name    should not be empty
     * @param proxy   the configuration proxy (local, remote or test)
     * @param topNode node describing the top component (main)
     */
    public NodeModularSubsystem(String name, ConfigurationProxy proxy, EffectiveNode topNode) {
        //todo: have a Ctor with name for subsystem
        super();
        Objects.requireNonNull(name);
        setName(name);
        //todo: supress this context things
        context = new Context();
        //TODO: do we need to init() the context?
        context.setSubsystem(this);
        context.setSubsystemName(name);
        this.configurationProxy = proxy;
        proxy.setSubsystem(this);
        //todo : create a Context if needed
        this.lookup = new NodeLookup(topNode);
        // populates the componentList and sets the environment of each component
        realRegisterNodes(topNode);
        mainObject = (Configurable) topNode.getRealValue();
        // invokes init on all Modules
        doInitComponents();
    }

    @Override
    public boolean isSlave() {
        return true;
    }

    public ComponentLookupService getLookup() {
        return lookup;
    }

    /**
     * *
     *
     * @return a List of [name-component] in PreOrder.
     */
    public List<ViewValue> getPreOrderComponentList() {
        return preOrderUnmodifiableView;
    }


    /* suppressed
    public void registerNodes(EffectiveNode node) {
        this.lookup = new NodeLookup(node);
        // HACK because of incoherent trees!
        // remove this hack!
        Iterator<String> iter = node.getAllKeys();
        if (iter != null) {
            while (iter.hasNext()) {
                String key = iter.next();
                Object value = node.getIndirect(key);
                if (value instanceof Module) {
                    // was :  this.registerModule((Module) value);
                    // block left for maintenance purposes
                    this.registerConfigurable(key, (Configurable) value);
                } else if (value instanceof Configurable) {
                    this.registerConfigurable(key, (Configurable) value);
                }
            }
        }
        realRegisterNodes(node);
        // post contruct operations here
    }
    */

    private void realRegisterNodes(EffectiveNode node) {
        String key = node.getKey();
        Object value = node.getRealValue();
        //TODO which order for component
        preOrderComponentList.add(new ViewValue(key, value));
        /*
        /// done : suppress that when we get rid of BasicModularSubsystem
        if (value instanceof Module) {
            // was :  this.registerModule((Module) value);
            // block left for maintenance purposes
            // inherited form BasicModularSubSystem this.registerConfigurable(key, (Configurable) value);
        } else if (value instanceof Configurable) {
            //this.registerConfigurable(key, (Configurable) value);
        }
        */
        // end stupid things
        // problem about order of evaluation should proceed from the ground up
        ArrayList<EffectiveNode> children = node.getChildren();
        if (children != null) {
            for (EffectiveNode child : children) {
                realRegisterNodes(child);
                ;
            }
        }
        if (value instanceof Configurable) {
            Configurable configurable = (Configurable) value;
            configurable.setEnvironment(new Configurable.Environment(key, configurable, configurationProxy, lookup));
            /*
            } */
        }
        /*
         } */
    }

    private void doInitComponents() {
        mainObject.proceduralWalk(Configurable::init, null);
        /*
      */
    }

    /**
     * invokes the <TT>start()</TT> methods on all components (starting from the top main Module)
     */
    @Override
    public void doStart() {
        mainObject.proceduralWalk(Configurable::start, null);
    }

    /**
     * invokes the <TT>postStart()</TT> methods on all components (starting from the top main Module)
     */
    @Override
    public void postStart() throws HardwareException {
        // for HACKERS ONLY: mainObject.proceduralWalk(Cocoon.consumer(Configurable::postStart), null);
        callPostStart(mainObject);
    }

    private void callPostStart(Configurable goal) throws HardwareException {
        goal.postStart();
        for (Configurable configurable : goal.listChildren()) {
            callPostStart(configurable);
        }
    }


    /**
     * invokes the <TT>shutdownNow()</TT> methods on all components (starting from the top main Module)
     */
    @Override
    public void doShutdown() {
        log.info("Subsystem " + getName() + " shutdown starting");
        mainObject.proceduralWalk(Configurable::shutdownNow, null);
    }

    /**
     * utility method to find a Component out of a Command destination
     *
     * @param commandDestination
     * @return
     */
    protected Configurable getCommandDestination(String commandDestination) {
        Configurable configurable = null;
        String configurableName = "main";

        if (commandDestination.contains("/")) {
            configurableName = commandDestination
                    .substring(commandDestination.indexOf("/") + 1);
        }
        configurable = (Configurable) lookup.getComponentByName(configurableName);
        if (configurable == null) {
            throw new IllegalArgumentException(" no such configurable " + configurableName + " for command destination");
        }
        return configurable;
    }

    /**
     * to deprecate: invoke configurable.getCommandBuilder instead.
     *
     * @param object
     * @return
     */
    @Override
    protected LocalCommandDictionary getCommandBuilder(Object object) {
        if (object instanceof Configurable) {
            Configurable configurable = (Configurable) object;
            Configurable.Environment env = configurable.getEnvironment();
            LocalCommandDictionary res = env.getCommandBuilder();
            //String name = env.getNameOfComponent();
            return res;
        }

        return super.getCommandBuilder(object);
    }

    /**
     * *
     * searches for details of an incoming command
     * </P>
     * This method is overriden to deal with Modular subsystem
     *
     * @param command
     * @return
     */
    //TODO: rewrite the use of boolean in searchdictionary is a mess
    @Override
    protected DictionarySearchResult searchForDictionary(Command command) throws CommandInvocationException {
        Configurable goal = getCommandDestination(command.getDestination());
        LocalCommandDictionary localDict = getCommandBuilder(goal);
        if (goal.getEnvironment().getNameOfComponent().equals("main")) {
            // we search first for main then for subsystem
            DictionarySearchResult searchDict = searchForDictionary(false, command, goal, localDict);
            if (searchDict == null) {
                return super.searchForDictionary(command);
            }
            // we use method since we may want to execute a raw method
            if (searchDict.method != null) {
                return searchDict;
            } else {
                return super.searchForDictionary(command);

            }

        } else {
            return searchForDictionary(true, command, goal, localDict);

        }

    }

    /**
     * forwards a HALT signal to all components (that are signalHandlers)
     */
    @org.lsst.ccs.command.annotations.Command(description = "halt", type = ABORT)
    @Override
    public void abort() {
        super.abort();
        Module mainModule = (Module) lookup.getComponentByName("main");
        if (mainModule == null) {
            throw new IllegalArgumentException("no component named main!");
            //TODO log ? or error
        }
        mainModule.percolateSignal(new Signal(SignalLevel.HALT));
    }

    /**
     * same as abort but with a time span.
     *
     * @param expectedMaxDelay
     */
    @org.lsst.ccs.command.annotations.Command(description = "halt with expected max delay", type = ABORT)
    @Override
    public void abort(long expectedMaxDelay) {
        super.abort();
        Module mainModule = (Module) lookup.getComponentByName("main");
        if (mainModule == null) {
            throw new IllegalArgumentException("no component named main!");
            //TODO log ? or error
        }
        mainModule.percolateSignal(new Signal(SignalLevel.HALT, expectedMaxDelay));
    }

    /**
     * forwards a STOP signal to all components (that are signalHandlers)
     */
    @org.lsst.ccs.command.annotations.Command(description = "stops with expected max delay", type = ACTION)
    @Override
    public void stop(long expectedMaxDelay) throws HardwareException {
        super.stop(expectedMaxDelay);
        Module mainModule = (Module) lookup.getComponentByName("main");
        if (mainModule == null) {
            throw new IllegalArgumentException("no component named main!");
            //TODO log ? or error
        }
        mainModule.percolateSignal(new Signal(SignalLevel.STOP, expectedMaxDelay));
    }

    /**
     * same as stop(delay) but with an infinite delay
     *
     * @throws HardwareException
     */
    @org.lsst.ccs.command.annotations.Command(description = "stops hardware", type = ACTION)
    public void stop() throws HardwareException {
        //TODO check is this is handled gracefully by stop
        stop(Long.MAX_VALUE);
    }

    /**
     * forwards a STOP signal to all components and then checks if all hardwares have been stopped within expectedMaxDelay
     *
     * @param expectedMaxDelay
     * @throws HardwareException, InterruptedException
     */
    @org.lsst.ccs.command.annotations.Command(description = "waits until all the hardware devices are actually stopped", type = ACTION)
    public void stopAndWait(long expectedMaxDelay) throws HardwareException, InterruptedException {
        // Stop returns immediately
        stop(expectedMaxDelay);
        // Now check if all hardware are stopped within expectedMaxDelay
        // Todo : discuss step : is there a smarter way to implement this ?
        int step = 10;
        long fracDelay = expectedMaxDelay / step;
        HardwareException lastExc = null;

        for (int i = 0; i < step; i++) {
            try {
                Thread.sleep(fracDelay);
                checkAllHardwareStopped();
                return;
            } catch (HardwareException e) {
                lastExc = e;
            }
        }
        throw lastExc;
    }

    /**
     * invoke a <TT>checkHardware</TT> on all <TT>HardwareController</TT>
     * (except when return diag modifies the path of invocation)
     *
     * @throws HardwareException a list of HardwareException
     */
    @Override
    public void checkHardware() throws HardwareException {
        Module mainModule = (Module) lookup.getComponentByName("main");
        if (mainModule == null) {
            throw new IllegalArgumentException("no component named main!");
            //TODO log ? or error
        }
        final MutableReference<HardwareException> excList = new MutableReference<>();
        mainModule.treeWalk(
                configurable -> {
                    if (configurable instanceof HardwareController) {
                        try {
                            return ((HardwareController) configurable).checkHardware();
                        } catch (HardwareException e) {
                            excList.reference = new HardwareException(configurable.toString() + " fails check", e, excList.reference);
                            //todo: not correct: what if not fatal and we still want to return Handler-children
                            if (e.isFatal()) {
                                return TreeWalkerDiag.STOP;
                            }
                        }
                    }
                    return TreeWalkerDiag.GO;
                }
                , null
        );
        /*

        */
        if (excList.reference != null) {
            throw excList.reference;
        }

    }

    /**
     * checks if all <TT>HardwareControllers</TT> have been started
     *
     * @throws HardwareException a list of exception fired by hardware that are not started
     */
    @Override
    public void checkAllHardwareStarted() throws HardwareException {
        Module mainModule = (Module) lookup.getComponentByName("main");
        if (mainModule == null) {
            throw new IllegalArgumentException("no component named main!");
            //TODO log ? or error
        }
        final MutableReference<HardwareException> excList = new MutableReference<>();
        mainModule.proceduralWalk(
                configurable -> {
                    if (configurable instanceof HardwareController) {
                        try {
                            ((HardwareController) configurable).checkStarted();
                        } catch (HardwareException e) {
                            excList.reference = new HardwareException(configurable.toString() + "not started ", e, excList.reference);
                        }
                    }
                }
                , null
        );
        /*

        */
        if (excList.reference != null) {
            throw excList.reference;
        }

    }

    /**
     * checks if all <TT>HardwareControllers</TT> have been stopped
     *
     * @throws HardwareException a list of exception fired by hardware that are not stopped
     */
    // same code as above!
    @Override
    public void checkAllHardwareStopped() throws HardwareException {
        Module mainModule = (Module) lookup.getComponentByName("main");
        if (mainModule == null) {
            throw new IllegalArgumentException("no component named main!");
            //TODO log ? or error
        }
        final MutableReference<HardwareException> excList = new MutableReference<>();
        mainModule.proceduralWalk(
                configurable -> {
                    if (configurable instanceof HardwareController) {
                        try {
                            ((HardwareController) configurable).checkStopped();
                        } catch (HardwareException e) {
                            excList.reference = new HardwareException(configurable.toString() + "not stopped ", e, excList.reference);
                        }
                    }
                }
                , null
        );
        /* */
        if (excList.reference != null) {
            throw excList.reference;
        }
    }

    /**
     * starts a new Configuration modification context, switches state to Engineering mode.
     * the new context will end when it will be saved or dropped.
     * @deprecated will be neutralized in 2.0 and 2.1 branches.
     */
    @Deprecated
    @org.lsst.ccs.command.annotations.Command(description = "will create a new context for modifying parameters (engineering mode)", type = org.lsst.ccs.command.annotations.Command.CommandType.ACTION)
    public synchronized final void newConfigurationContext() {
        if (innerState.getMode().equals(Mode.NORMAL)) {
            this.switchToEngineeringMode();
        }
        this.configurationProxy.startNewConfigurationContext();
    }

    /**
     * register a new Configuration . the state remains in  Engineering mode (though the current context is dropped)
     *
     * @param configurationName
     * @param tag
     * @throws IOException
     */
    @org.lsst.ccs.command.annotations.Command(description = "@Deprecated: registers a new configuration with name and tag", type = org.lsst.ccs.command.annotations.Command.CommandType.CONFIGURATION)
    @Deprecated
    public final void register(
            @Argument(name = "configurationName", description = "The Name of the Configuration") String configurationName,
            @Argument(name = "tag", description = "The new tag name", defaultValue = "") String tag) throws IOException {
        this.configurationProxy.registerConfiguration(configurationName, tag);
    }

    /**
     * 
     * @param configurationName
     * @throws IOException 
     * @deprecated saveConfiguration should be used instead
     */
    @Deprecated
    @org.lsst.ccs.command.annotations.Command(description = "registers a new configuration with a name", type = org.lsst.ccs.command.annotations.Command.CommandType.CONFIGURATION)
    public final void registerConfiguration(
            @Argument(name = "configurationName", description = "The Name of the Configuration") String configurationName) throws IOException {
        this.configurationProxy.registerConfiguration(configurationName);
    }
    
    /**
     * Saves a configuration with a new name.
     * If a configuration with the same name already exists it is overwritten.
     * The state remains in engineering mode.
     * @param configurationName the new configuration name
     * @throws IOException
     */
    @org.lsst.ccs.command.annotations.Command(description = "Saves a new configuration with a name", type = org.lsst.ccs.command.annotations.Command.CommandType.CONFIGURATION)
    public final void saveConfiguration(
            @Argument(name = "configurationName", description = "The Name of the Configuration") String configurationName) throws IOException {
        this.configurationProxy.registerConfiguration(configurationName);
    }

    /**
     * Saves the current Configuration.
     * The state remains in engineering mode
     * @throws IOException
     */
    @org.lsst.ccs.command.annotations.Command(description = "Overwrites the current configuration", type = org.lsst.ccs.command.annotations.Command.CommandType.CONFIGURATION)
    public final void saveConfiguration() throws IOException {
        this.configurationProxy.registerConfiguration(this.configurationProxy.getConfigurationName());
    }

    /**
     * drops the current modification of parameters: parameters that were active before
     * the configuration context set up are resurected.
     * Note that the state is still ENGINEERING
     * @deprecated Does not perform an actual rollback
     */
    @org.lsst.ccs.command.annotations.Command(description = "will drop context for modifying parameters (engineering mode)", type = org.lsst.ccs.command.annotations.Command.CommandType.CONFIGURATION)
    public void dropConfigurationContext() {
        this.configurationProxy.dropModifications();
        //TODO rollback !!
    }
    
    /**
     * Loads the specified configuration.
     * Parameters listed in the properties file are changed to their new value.
     * Parameters that are not listed are set back to their value defined by the
     * groovy description file
     * Changes that are unsaved are dropped.
     * @param config The name of the configuration
     * @throws IllegalArgumentException if the argument is a path or if it does 
     * not match to an existing configuration file (except when loading the default
     * configuration).
     */
    @org.lsst.ccs.command.annotations.Command(description = "loads a new configuration with a name", type = org.lsst.ccs.command.annotations.Command.CommandType.CONFIGURATION)
    public void loadConfiguration (
            @Argument(name = "configurationName", description = "The Name of the Configuration") String config) throws Exception {
        if (ConfigUtils.isAPath(config)){
            throw new IllegalArgumentException("loadConfiguration requires a configuration name, not a path");
        }
        configurationProxy.startNewConfigurationContext();
        Properties configProps = null;
        ConfigProfile oldProfile = ((LocalConfigurationProxy)configurationProxy).getCurrentProfile();
        String configName = ConfigUtils.baseNameFromNames(oldProfile.getSubsystemName(), config, oldProfile.getTag());
        try {
            configProps =  ((LocalConfigurationProxy)configurationProxy).getWriterProvider().getConfigurationProperties(configName); 
        } catch (IllegalArgumentException ex){
            if(!config.isEmpty()) throw ex;
        }
        ConfigProfile newProfile = Factories.createConfigProfile(oldProfile.getSubsystemDescription(), oldProfile.getName(),oldProfile.getTag(),oldProfile.getUserName(),oldProfile.getLevel());
        if (configProps!=null && !configProps.isEmpty()){
            // Incorporating properties file to the new profile
            newProfile.mergeProperties(configProps);
        }
        switchConfiguration(oldProfile, newProfile);
        // configFileName should not be a path
        configurationProxy.saveModifications(config);
    }
    
    /**
     * Loads the default configuration if it exists. 
     * If not, all parameters are set back to the value defined in the groovy
     * description file.
     * Equivalent to loadConfiguration("")
     * @throws Exception 
     */
    @org.lsst.ccs.command.annotations.Command(description = "loads the default configuration", type = org.lsst.ccs.command.annotations.Command.CommandType.CONFIGURATION)
    public void loadConfiguration () throws Exception {
       loadConfiguration("");
    }


    /**
     * TODO: modify this code: the quesiton is what is changed between profiles
     * 2 different cases: after engineering modifications and other switches.
     *
     * @param oldProfile profile still active in database
     * @param newProfile profile to be registered afterwards
     * @throws Exception could be that a parameter change is refused by code or that rollback failed afterwards
     */
    public void switchConfiguration(ConfigProfile oldProfile, ConfigProfile newProfile) throws Exception {
        // this is wrong  in engineering mode switch if both are static and have the same static data see code below
        /* wrong test
        if(newProfile.isChangingStaticData()) {
           throw new IllegalArgumentException(" Profile " + newProfile.getName() +
           " deals with static parameters and so cannot be loaded at runtime")  ;
        } */
        Module mainModule = (Module) lookup.getComponentByName("main");
        if (mainModule == null) {
            throw new IllegalArgumentException("no component named main!");
        }
        Map<String, List<ParameterConfiguration>> mapChanges = newProfile.getConfigurationMap();
        // problem: we may have ParameterConfiguration objects that are no more in the new Profile
        Map<String, List<ParameterConfiguration>> oldMap = oldProfile.getConfigurationMap();

        List<String[]> changedValues = new ArrayList<>();
        // to understand how this code works see utilities.Cocoon
        // the lambda throws an Exception so is "cocooned"
        mainModule.proceduralWalk(null, Cocoon.consumer((Configurable configurable) ->
        { //begin lambda
            String name = configurable.getName();
            List<ParameterConfiguration> list = mapChanges.get(name);
            List<ParameterConfiguration> oldList = oldMap.get(name);
            // default confguration has an empty list
            // if parameters are suppressed from OldList and not in list
            if(oldList == null) {
                oldList = Collections.emptyList() ;
            }
            try {
                if (list != null) {
                    for (ParameterConfiguration parmConfig : list) {
                        // the equals method of parameterConfiguration allows this !
                        if(oldList.remove(parmConfig)) {
                            // trace
                        };
                        ParameterPath path = parmConfig.getPath();
                        String parameterName = path.getParameterName();
                        String value = parmConfig.getValue();
                        //TODO: beware of time
                        String oldValue = oldProfile.getValueAt(path, PackCst.STILL_VALID);
                        //TODO: some problems if strings are not exactly the same! (though the values are equals! examples : maps , unordered lists, floating point)
                        // change the test
                        if (oldValue.equals(value)) {
                            continue;
                        }
                        // now we REALLY want to change but ....
                        if (parmConfig.getDescription().isNotModifiableAtRuntime()) {
                            throw new IllegalArgumentException(path + "not modifiable at runtime");
                        }
                        String[] histElement = new String[]{name, parameterName, oldValue};

                        configurable.change(parameterName, value);
                        //last pushed in front
                        changedValues.add(0, histElement);
                    }
                }
                // now we deal with parameters that have been deleted
                for (ParameterConfiguration parmConfig : oldList) {
                    ParameterPath path = parmConfig.getPath();
                    String parameterName = path.getParameterName();
                    // we get the default value
                    String value = parmConfig.getDescription().getDefaultValue() ;
                    String oldValue = oldProfile.getValueAt(path, PackCst.STILL_VALID);
                    //TODO: code copy : modify
                    if (parmConfig.getDescription().isNotModifiableAtRuntime()) {
                        //TODO: do we throw exception or just  do nothing?
                        throw new IllegalArgumentException(path + "not modifiable at runtime");
                    }
                    String[] histElement = new String[]{name, parameterName, oldValue};

                    configurable.change(parameterName, value);
                    //last pushed in front
                    changedValues.add(0, histElement);
                }

            } catch (Exception exc) {
                //rollback is tried in reverse order
                for (String[] historyElement : changedValues) {
                    String cpName = historyElement[0];
                    String parmName = historyElement[1];
                    String histValue = historyElement[2];
                    try {
                        Configurable cp = (Configurable) lookup.getComponentByName(cpName);
                        cp.change(parmName, histValue);
                    } catch (Exception rollbackExc) {
                        throw new RollBackException(rollbackExc, cpName, parmName, histValue);
                    }
                }
                throw exc;
            }


        } /*end lambda */) // end Cocoon
        );

    }


}
