package org.lsst.ccs.startup;

import org.lsst.ccs.CCSCst;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.config.*;
import org.lsst.ccs.config.utilities.ConfigUtils;
import org.lsst.ccs.framework.ConfigurationProxy;
import org.lsst.ccs.utilities.jars.CommonResources;
import org.lsst.ccs.utilities.logging.Logger;
import org.lsst.ccs.utilities.tracers.Tracer;
import org.lsst.gruth.jutils.DescriptiveNode;
import org.lsst.gruth.jutils.EffectiveNode;
import org.lsst.gruth.nodes.ComponentFactory;
import org.lsst.gruth.nodes.Utils;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.lang.reflect.InvocationTargetException;
import java.util.List;
import java.util.Properties;
import org.lsst.ccs.config.LocalConfigurationProxy.WriterProvider;

/**
 * A collection of static utilities linked to boot operations.
 * Note that:
 * <UL>
 * <LI>
 * a descriptive node represents a configuration description
 * <LI/>
 * an effective node is the real subsystem and module hierarchy (ready to run)
 * <LI/>
 * a "raw" subsystem is without configuration data
 * </UL>
 *
 * @author bamade
 */
// Date: 05/06/12

public class BootUtils {
    static {
        assert Tracer.version("$Rev$", BootUtils.class, "org-lsst-ccs-startup");
    }

    static {
        //set up logging and gets a logging.properties as resource
        Logger.configure();
        //Class.forName("org.lsst.ccs.utilities.logging.LogManagement");
    }

    /***
     * Very important class: when an instance is created you can have :
     * <UL>
     *     <LI/> a descriptive Node
     *     <LI/> the names of subsystem, configuration, tag
     *     <LI/> the corresponding Effective Node
     *     <LI/> a Subsystem ready to run (but not started)
     * </UL>
     */
    public static class LocalBootObject {
        /***
         * this is the initial descriptiveNode: not the one
         * modified by the Configuration
         */
        DescriptiveNode descriptiveNode;
        DescriptiveNode modifiedDescriptiveNode ;
        EffectiveNode effectiveNode;
        Subsystem subsystem;
        private final String subsystemName;
        private final String configurationName;
        private final String tagName;
        private WriterProvider writerProvider;

       /***
         * creates a boot Object from a description file name
         * @param pathName
         * @throws Exception
         */
        public LocalBootObject(String pathName) throws Exception {
            this(pathName, "");
        }

        /***
         * creates a boot Object from a description file name and
         * a configuration property file.
         * The values in the configuration file override the parameters set in
         * the description File.
         * @param pathName
         * @param config can be a file name OR the config name only
         * @throws Exception
         */
        public LocalBootObject(String pathName, String config) throws Exception {
            this(pathName, config, null, null);
        }
        
        public LocalBootObject(String pathName, String config, String subsystemAlias, WriterProvider writerProvider) throws Exception{
            this.writerProvider = writerProvider;
            System.setProperty("org.lsst.ccs.namefromgroovy", String.valueOf(!pathName.contains(".")).toLowerCase());
            if (config == null)config="";
            descriptiveNode = BootUtils.getDescriptiveNode(pathName);
            NamesAndTag namesAndTag = ConfigUtils.namesFromPaths(pathName, config);
            if (System.getProperty("org.lsst.ccs.namefromgroovy", "false").equals("false")){
                subsystemName = namesAndTag.getSubsystemName();
            } else {
                if (descriptiveNode instanceof MainDescriptiveNode){
                    subsystemName = (subsystemAlias==null)?((MainDescriptiveNode)descriptiveNode).getSubsystemName():subsystemAlias;
                } else throw new RuntimeException("decriptive node is not an instance of MainDescriptiveNode");
            }
            configurationName = namesAndTag.getConfigName();
            tagName = namesAndTag.getTag();
            
            subsystemFromDescriptionNode(descriptiveNode, subsystemName, configurationName, tagName, namesAndTag.getConfigFileName());
        }
        
        /***
         * returns the initial descriptive Node
         * (not the one modified by the Configuration Data)
         * @return
         */
        public DescriptiveNode getDescriptiveNode() {
            return descriptiveNode;
        }

        /***
         * returns the descriptive node as modified by configuration
         * (could be identical to initial DescriptiveNode)
         * @return
         */
        public DescriptiveNode getModifiedDescriptiveNode(){
            return modifiedDescriptiveNode ;
        }

        public EffectiveNode getEffectiveNode() {
            return effectiveNode;
        }

        public Subsystem getSubsystem() {
            return subsystem;
        }

        public String getSubsystemName() {
            return subsystemName;
        }

        public String getConfigurationName() {
            return configurationName;
        }

        public String getTagName() {
            return tagName;
        }


        /***
         * Gets a reference to an object built by the configuration file.
         * <B>beware</B> if the subsystem is not started the object may not
         * have been completely initialized (the start() method of the subsystem
         * invokes various initialization)
         * @param key
         * @return
         */
        public Object getRunningObject(String key) {
            return effectiveNode.getIndirect(key);
        }

        /***
         * creates a Subsystem instance out of a Descriptive Node and Properties
         * (that may override parameters of the descriptive node)
         * @ImplNote
         * <UL>
         *     <LI/> creates a <TT>SubsystemDescription </TT> Object from the descriptive node
         *     the parameters that are tagged "name" and that are References to other
         *     components are not included in the list of Parameters that can be modified (DEFAULT_TREE_PARAMETER_FILTER)
         *     <LI/>
         *     since No SubsystemDescription can be used unless it is under the control
         *     of a DAO this description is passed to a dummy "MemoryDAO"
         *     <LI/>
         *     we create a <TT>ConfigProfile</TT> (where parameters  have values)
         *     out of the subsystem description
         *     <LI/>
         *     This ConfigProfile is modified with the configuration Properties
         *     and saved to the MemoryDAO
         *     <LI/>
         *     The modified configuration is saved to a ModifiedDescriptiveNode
         *     <LI/>
         *     An EffectiveNode is built out of this modified DescriptiveNode
         *
         *
         * </UL>
         *
         * @param componentNode a descriptive node
         * @param subsystemName
         * @param configName
         * @param tag
         * @param configProps
         * @throws Exception
         */
        private void subsystemFromDescriptionNode(DescriptiveNode componentNode, String subsystemName, String configName,
                                                  String tag, String configFileName) throws Exception {

            modifiedDescriptiveNode = componentNode ;
            SubsystemDescription description = Factories.createSubsystemDescription(subsystemName, tag, userName, version,
                    componentNode, DataFlavour.PURE_OBJECT_TREE, SubsystemDescription.DEFAULT_TREE_PARAMETER_FILTER);
            //these Configuration objects: SubsystemDescription and ConfigProfile should be handled by a DAO
            // right now we operate locally so it is a special dummy DAO: MemoryDAO
            CCSCst.LOG_TODO.fine("TODO: DAO at creation of SubsystemDescription should also be able to interact with remote database (switch between local and remote?)") ;
            MemoryDAO dao = new MemoryDAO();
            dao.saveAbstractDescription(description);
            // if no configuration create an empty configuration
            ConfigProfile configProfile = Factories.createConfigProfile(description, configName, tag, userName, level);
             
            LocalConfigurationProxy configurationAccess = new LocalConfigurationProxy(description);
            if(writerProvider != null){
                configurationAccess.setWriterProvider(writerProvider);
            }
            
            Properties configProps = null;
            InputStream propsIs = null;
            try {
                configProps = configurationAccess.getWriterProvider().getConfigurationProperties(configFileName);
            } catch (IllegalArgumentException ex){
                if (!configName.isEmpty()){ //We don't throw an exception when the default configuration is not found
                    throw ex;
                }
            }
            if (configProps != null) {
                // else merge with properties
                try {
                    configProfile.mergeProperties(configProps);
                    modifiedDescriptiveNode = configProfile.getModifiedConfigurationData();
                } catch (IllegalArgumentException exc) {
                    List<String> messages = configProfile.reportFailures();
                    throw exc;
                    //throw new IllegalArgumentException(" configuration errors :"+ messages) ;
                }
            }
            dao.saveAbstractConfig(configProfile);
            configurationAccess.setBaseProfile(configProfile);
            effectiveNode = BootUtils.getEffectiveNode(modifiedDescriptiveNode);
            //system.start()
            subsystem = getRawSubsystemFromEffectiveNode(effectiveNode, configProfile, subsystemName, configName, tag, configurationAccess);
            // set up configuration access
            /* duplicated in getRawSubsystemFromEffectiveNode
            LocalConfigurationProxy configurationAccess = new LocalConfigurationProxy(configProfile);
            subsystem.setConfigurationProxy(configurationAccess);
            */

        }
    }
    ////////////////////// END LOCAL BOOT OBJECT

    //TODO .... set?
    // defult values
    static String userName = "";
    static String version = "1.0";
    static int level = PackCst.DESIGNER_LEVEL;

    /**
     * gets a descriptive node from a file
     *
     * @param pathName
     * @return
     * @throws Exception
     */
    public static DescriptiveNode getDescriptiveNode(String pathName) throws Exception {
        DescriptiveNode componentNode = null;
        InputStream is;
        if (!pathName.endsWith(".ser") && System.getProperty("org.lsst.ccs.namefromgroovy","false").equals("true")){
            pathName += ".groo";
            try {
                is = CommonResources.getInput(BootUtils.class, pathName);
            } catch (IOException ex){
                pathName += "vy";
                is = CommonResources.getInput(BootUtils.class, pathName);
        }
        }
        else {
            is = CommonResources.getInput(BootUtils.class, pathName);
        }

        if (is == null) {
            throw new IOException(pathName + " not found through CommonResources");
        }
        if (pathName.endsWith(".ser")) {
            ObjectInputStream ois = new ObjectInputStream(new BufferedInputStream(is));
            componentNode = (DescriptiveNode) ois.readObject();

        } else if (pathName.endsWith(".groovy") || pathName.endsWith(".groo")) {
            //TODO: handle the encoding!
            componentNode = Utils.getComponentNodeFromGroovy(is, "ISO-8859-1");

        }
        return componentNode;
    }

    /**
     * gets an effective node from a description node
     *
     * @param descriptionNode
     * @return
     * @throws InvocationTargetException
     * @throws NoSuchMethodException
     * @throws IllegalAccessException
     */
    // argument is of type DescriptiveNode result of type EffectiveNode
    public static EffectiveNode getEffectiveNode(DescriptiveNode descriptionNode) throws InvocationTargetException, NoSuchMethodException, IllegalAccessException {
        ComponentFactory fact = new ComponentFactory(descriptionNode);
        EffectiveNode top = fact.build();
        top.doCallsFromTop();
        return top;
    }


    /**
     * gets an effective node from a file
     *
     * @param pathName
     * @return
     * @throws Exception
     */
    public static EffectiveNode getEffectiveNodeFromFile(String pathName) throws Exception {
        DescriptiveNode descriptiveNode = getDescriptiveNode(pathName);
        return getEffectiveNode(descriptiveNode);
    }


    public static Subsystem getRawSubsystemFromEffectiveNode(EffectiveNode effectiveNode, ConfigProfile profile, String subsystemName,
                                                             String configName, String tag,
                                                             ConfigurationProxy confProxy) {
        //todo:
        CCSCst.LOG_TODO.fine("TODO: Configuration proxy should also be able to interact with remote database (switch between local and remote?)") ;
        NodeModularSubsystem system = new NodeModularSubsystem(subsystemName, confProxy, effectiveNode);
        //configurationAccess.setSubsystem(system);
        //system.registerNodes(effectiveNode);
        //system.setConfiguration(configName);
        //system.setTag(tag);
        return system;
    }



    public static Subsystem rawSubsystemFromDescriptionNode(DescriptiveNode descriptionNode,
                                                            ConfigProfile profile,
                                                            String subsystemName, String configName,
                                                            String tag,
                                                            ConfigurationProxy confProxy) throws Exception {
        EffectiveNode top = getEffectiveNode(descriptionNode);
        //system.start()
        return getRawSubsystemFromEffectiveNode(top, profile, subsystemName, configName, tag, confProxy);
    }




    public static Subsystem getSubsystemFromFile(String pathName) throws Exception {
        /** WAS
         ComponentNode descriptiveNode = getDescriptiveNode(pathName) ;
         NamesAndTag namesAndTag = BootUtils.namesFromPath(pathName) ;
         //split name
         String subsystemName = namesAndTag.getSubsystemName() ;
         String configName = namesAndTag.getConfigName() ;
         String tag = namesAndTag.getTag() ;
         Subsystem res = rawSubsystemFromDescriptionNode(descriptiveNode, subsystemName, configName, tag) ;
         return res;
         */
        return getSubsystemFromFile(pathName, null);
    }

    public static Subsystem getSubsystemFromFile(String pathName, String propertiesFileName) throws Exception {
        return getSubsystemFromFile(pathName, propertiesFileName, null);
        }

    public static Subsystem getSubsystemFromFile(String pathName, String propertiesFileName, String subsystemAlias) throws Exception {
        return new LocalBootObject(pathName, propertiesFileName, subsystemAlias, null).getSubsystem();
    }




    /**
     * saves the latest run configuration in a local cache.
     * This is saved in a file (see naming  convention handled by <TT>baseNameFromNames</TT>
     * that will reside in tha cache directory.
     * The name of this file will be reported in another text file in this directory named "latest.txt".
     * <p/>
     * TODO: The path of the cache directory is to be determined.
     *
     * @param node
     * @param subsystemName
     * @param configName
     * @param tag
     */

    public static void saveInCache(DescriptiveNode node, String subsystemName,
                                   String configName, String tag) {
        //TODO: implement saveInCache
    }

    public static DescriptiveNode getLatestInCache() {
        return null;
    }

    public static void bootFromCache() {
        throw new UnsupportedOperationException("bootFromCache");
    }

    /**
     * to be invoked before registering a <TT>ConfigProfile</TT> which is not meant
     * to be used for changing values at runtime.
     * This method creates a real dummy <TT>EffectiveNode</TT> to check for properties
     * that have preconditions. This means that objects created this way should not affect the hardwares.
     * @param profile
     */
    public static void checkStaticCompatibleConfiguration(ConfigProfile profile) throws Exception {
        DescriptiveNode node = profile.getModifiedConfigurationData();
        getEffectiveNode(node);
    }

}
