package org.lsst.ccs.software.versions.puglish.plugin;

import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.logging.Level;
import javax.ws.rs.client.Client;
import javax.ws.rs.client.ClientBuilder;
import javax.ws.rs.client.Entity;
import javax.ws.rs.client.WebTarget;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import org.apache.maven.artifact.Artifact;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.plugins.annotations.ResolutionScope;
import org.apache.maven.project.MavenProject;

/**
 * A Plugin to publish the list of dependencies with their versions.
 *
 */
@Mojo(name = "publish", defaultPhase = LifecyclePhase.PACKAGE, requiresDependencyResolution = ResolutionScope.COMPILE_PLUS_RUNTIME)
public class PublishPlugin extends AbstractMojo {

    private final Client client = ClientBuilder.newClient();
    private WebTarget serv;
    private static final ObjectMapper mapper = new ObjectMapper();
    
    /**
     * Maven project on which the plugin is operating
     */
    @Parameter(defaultValue = "${project}", readonly = true, required = true)
    private MavenProject project;

    @Override
    public void execute() throws MojoExecutionException, MojoFailureException {

        if ( ! this.project.getPackaging().equals("jar") ) {
            getLog().info("Skipping publishing artifacts for non \"jar\" packaging project");
            return;
        }
        
        serv = client.target("https://srs.slac.stanford.edu/releaseManagement/");        

        List<String> allArtifacts = new ArrayList<>();
        
        for ( Artifact a : project.getArtifacts() ) {
            if ( ! a.getType().equals("jar") ) 
                continue;
            allArtifacts.add(a.toString());
        }
        Collections.sort(allArtifacts);

        String thisArtifact = project.getArtifact().toString();
        
        ArtifactsPublicationPayload payload = new ArtifactsPublicationPayload(thisArtifact, allArtifacts);
        

        try (Response response = serv.path("publishArtifacts").request(MediaType.APPLICATION_JSON).post(Entity.entity(payload, MediaType.APPLICATION_JSON))) {
            if ( response.getStatus() == 200 ) {
                getLog().info("Successfully published artifacts.");
            } else {
                RuntimeException re = new RuntimeException("Failed to publish artifacts: "+response.toString());                
                getLog().warn(re.getMessage());                
//                throw re;
            }
        } catch (Exception e) {
            getLog().warn("Exception while publishing artifacts.",e);
//            throw e;
        }
        
        
    }
    
    
    
    public static class ArtifactsPublicationPayload {

        public final String thisArtifact;
        public final List<String> allArtifacts;
        
        ArtifactsPublicationPayload(String thisArtifact, List<String> allArtifacts) {
            this.thisArtifact = thisArtifact;
            this.allArtifacts = allArtifacts;
        }

        @Override
        public String toString() {
            try {
                return "{ \"artifact\" : "+thisArtifact+", \"dependencies\":"+mapper.writeValueAsString(allArtifacts)+"}";
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
        }

    }
    
    

}
