package org.lsst.ccs.release.management;

import java.io.IOException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.json.JSONException;
import org.srs.web.base.db.ConnectionManager;

/**
 *
 * @author turri
 */
public class ReleaseManagementServlet extends HttpServlet {

    static String[] subsystemModules = {"buses", "main", "gui"};

    @Override
    protected void doGet(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {

        //Parameters passed to the servlet
        String projectName = req.getParameter("projectName");
        String releaseVersion = req.getParameter("releaseVersion");
        String snapshotVersion = req.getParameter("snapshotVersion");
        String projectType = req.getParameter("projectType");
        String projectDescription = req.getParameter("projectDescription");
        String projectJiraProject = req.getParameter("projectJiraProject");
        String projectJiraPrefix = req.getParameter("projectJiraPrefix");

        if (projectName != null) {

            // Create connection to ccs database
            Connection conn = null;
            try {
                conn = ConnectionManager.getConnection("jdbc/ccs");
                conn.setAutoCommit(true);

                // Create or update project information
                String projectCheckQuery = "select * from ccs_Projects where name = ?";
                PreparedStatement projectCheckStat = conn.prepareStatement(projectCheckQuery);
                projectCheckStat.setString(1, projectName);
                ResultSet projectCheckResult = projectCheckStat.executeQuery();
                // Variable to tell if the project already exists in the tables
                boolean projectExists = projectCheckResult.next();
                if (!projectExists) {
                    PreparedStatement insertProjectStat = conn.prepareStatement("insert into ccs_Projects (name) values (?)");
                    insertProjectStat.setString(1, projectName);
                    insertProjectStat.executeUpdate();

                    PreparedStatement insertProjectDisplay = conn.prepareStatement("insert into ccs_Project_Display (projectName, projectSiteLocation, projectOrder, links) values (?,?,?,?)");

                    int order = 100;

                    if (projectType.equals("subsystem")) {

                        insertProjectDisplay.setString(1, projectName);
                        insertProjectDisplay.setString(2, "subsystems");
                        insertProjectDisplay.setInt(3, order);
                        insertProjectDisplay.setString(4, "apidoc,sourcecode,testcode,dependencies,tests,graph");
                        insertProjectDisplay.executeUpdate();

                        for (String mod : subsystemModules) {
                            String moduleName = projectName + "-" + mod;
                            PreparedStatement createModule = conn.prepareStatement("insert into ccs_Project_Modules (moduleName,projectName) values (?,?)");
                            createModule.setString(1, moduleName);
                            createModule.setString(2, projectName);
                            createModule.executeUpdate();

                            order++;
                            insertProjectDisplay.setString(1, moduleName);
                            insertProjectDisplay.setString(2, moduleName);
                            insertProjectDisplay.setInt(3, order);
                            insertProjectDisplay.setString(4, "apidoc,sourcecode,testcode,dependencies,tests,jdepend,graph,findbugs,dist");
                            insertProjectDisplay.executeUpdate();
                        }

                    } else if (projectType.equals("driver")) {
                        insertProjectDisplay.setString(1, projectName);
                        insertProjectDisplay.setString(2, "drivers");
                        insertProjectDisplay.setInt(3, order);
                        insertProjectDisplay.setString(4, "apidoc,sourcecode,testcode,dependencies,tests,jdepend,graph,findbugs,dist");
                        insertProjectDisplay.executeUpdate();
                    }

                }

                resp.getWriter().print("Updating information for project " + projectName + "\n");
                resp.getWriter().print("\tDescription: " + emptyStringIfNull(projectDescription) + "\n");
                resp.getWriter().print("\tType: " + emptyStringIfNull(projectType) + "\n");
                resp.getWriter().print("\tJira Project: " + emptyStringIfNull(projectJiraProject) + "\n");
                resp.getWriter().print("\tJira Prefix: " + emptyStringIfNull(projectJiraPrefix) + "\n");

                // Update the project's information
                PreparedStatement projectInfoUpdateStat = conn.prepareStatement("update ccs_Projects set description=?, jiraProject=?, jiraVersionPrefix=?, type=? where name=?");
                projectInfoUpdateStat.setString(1, emptyStringIfNull(projectDescription));
                projectInfoUpdateStat.setString(2, emptyStringIfNull(projectJiraProject));
                projectInfoUpdateStat.setString(3, emptyStringIfNull(projectJiraPrefix));
                projectInfoUpdateStat.setString(4, emptyStringIfNull(projectType));
                projectInfoUpdateStat.setString(5, projectName);
                projectInfoUpdateStat.executeUpdate();

                // If it's a RELEASE
                if (releaseVersion != null && !"".equals(releaseVersion)) {
                    // Select the corresponding release cycle, either the active one or create a default one
                    PreparedStatement activeRelCycleStat = conn.prepareStatement("select * from ccs_Release_Cycle where released = 0 and active = 1");
                    ResultSet activeRelCycleResult = activeRelCycleStat.executeQuery();
                    String relCycle = null;
                    if (activeRelCycleResult.next()) {
                        relCycle = activeRelCycleResult.getString("name");
                    } else {
                        java.text.DateFormat df = new java.text.SimpleDateFormat("yyyy-MM-dd");
                        relCycle = df.format(new java.util.Date());
                        //Now add the release Cycle
                        try {
                            PreparedStatement createRelCycleStat = conn.prepareStatement("insert into ccs_Release_Cycle (name,date) values (?,(select current_date from dual))");
                            createRelCycleStat.setString(1, relCycle);
                            createRelCycleStat.executeUpdate();
                            resp.getWriter().print("\nCreated Release Cycle " + relCycle + "https://srs.slac.stanford.edu/releaseManagement/releaseCycles.jsp\n");
                        } catch (Exception e) {
                            //Silently ignore failure in case the rel cycle already exists.
                        }
                    }

                    // Now add the version to the project versions table
                    PreparedStatement addReleaseStat = conn.prepareStatement("insert into ccs_Project_Version (projectName,version,relCycle,created,isSnapshot) values (?,?,?,(select current_timestamp from dual),0)");
                    addReleaseStat.setString(1, projectName);
                    addReleaseStat.setString(2, releaseVersion);
                    addReleaseStat.setString(3, relCycle);
                    addReleaseStat.executeUpdate();

                    resp.getWriter().print("Added Release " + releaseVersion + " to project " + projectName + " https://srs.slac.stanford.edu/releaseManagement/projectVersions.jsp?projectName=" + projectName + "for Release Cycle " + relCycle + "https://srs.slac.stanford.edu/releaseManagement/releaseNotes.jsp?relCycle=" + relCycle + "\n");

                    // Remove SNAPSHOT version if it exists.
                    PreparedStatement removeSnapshotStat = conn.prepareStatement("delete from ccs_Project_Version where projectName = ? and version = ?");
                    removeSnapshotStat.setString(1, projectName);
                    removeSnapshotStat.setString(2, releaseVersion + "-SNAPSHOT");
                    removeSnapshotStat.executeUpdate();

                    // Finally update the version in the Release Cycle Versions
                    if (relCycle != null) {
                        PreparedStatement updateResCycleVersions = conn.prepareStatement("update ccs_Release_Cycle_Versions set version = ? where projectName = ? and relCycle = ? ");
                        updateResCycleVersions.setString(1, releaseVersion);
                        updateResCycleVersions.setString(2, projectName);
                        updateResCycleVersions.setString(3, relCycle);
                        updateResCycleVersions.executeUpdate();
                    }

                }

                // If it's a SNAPSHOT
                if (snapshotVersion != null && !"".equals(snapshotVersion)) {
                    try {
                        //Add the SNAPSHOT version to the project version table
                        PreparedStatement addSnapshotToProject = conn.prepareStatement("insert into ccs_Project_Version (projectName,version,created) values (?,?,(select current_timestamp from dual))");
                        addSnapshotToProject.setString(1, projectName);
                        addSnapshotToProject.setString(2, snapshotVersion);
                        addSnapshotToProject.executeUpdate();
                        resp.getWriter().print("Added Snapshot " + snapshotVersion + " to project " + projectName + "https://srs.slac.stanford.edu/releaseManagement/projectVersions.jsp?projectName=" + projectName + "\n");
                    } catch (IOException | SQLException e) {

                    }
                }

                if (projectJiraProject != null && !"".equals(projectJiraProject)) {
                    String jiraRelVersion = releaseVersion;
                    String jiraNextVersion = snapshotVersion;
                    if (projectJiraPrefix != null && !"".equals(projectJiraPrefix)) {
                        if (jiraRelVersion != null && !"".equals(jiraRelVersion)) {
                            jiraRelVersion = projectJiraPrefix + "-" + jiraRelVersion;
                        }
                        if (jiraNextVersion != null && !"".equals(jiraNextVersion)) {
                            jiraNextVersion = projectJiraPrefix + "-" + jiraNextVersion;
                        }
                    }
                    try {
                        JiraVersionManagement.updateJiraVersions(projectJiraProject, jiraRelVersion, jiraNextVersion);
                    } catch (JSONException ex) {
                        resp.getWriter().print("Problem updating Jira projects\n");
                        resp.getWriter().print(ex.getMessage() + "\n");
                    }

                }

            } catch (Exception ex) {
                resp.getWriter().print(ex.getMessage());

            } finally {
                try {
                    conn.close();
                } catch (SQLException ex) {
                    resp.getWriter().print(ex.getMessage());
                }
            }
        } else {
            resp.getWriter().print("ReleaseManagementServlet: no projectName was provided.");
        }

    }

    private static String emptyStringIfNull(String value) {
        return value == null ? "" : value;
    }

}
