/*
 * This file is part of the Cliche project, licensed under MIT License.
 * See LICENSE.txt file in root folder of Cliche sources.
 */

package org.lsst.ccs.shell;


import java.lang.annotation.Annotation;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;


/**
 * Command table entry
 */
public class LocalShellCommand extends ShellCommand {

    private Method method;
    private Object handler;

    public LocalShellCommand(Object handler, Method method, String prefix, String name) {
        super(prefix,name);
        assert method != null;
        setParamSpecs( forMethod(method) );
        assert getParamSpecs().length == method.getParameterTypes().length;
        this.method = method;
        this.handler = handler;
        setDescription(makeCommandDescription(method, getParamSpecs()));
    }

    private static ShellCommandParamSpec[] forMethod(Method theMethod) {
        Class[] paramTypes = theMethod.getParameterTypes();
        ShellCommandParamSpec[] result = new ShellCommandParamSpec[theMethod.getParameterTypes().length];
        Annotation[][] annotations = theMethod.getParameterAnnotations();
        assert annotations.length == result.length;
        for (int i = 0; i < result.length; i++) {
            Param paramAnnotation = null;
            for (Annotation a : annotations[i]) {
                if (a instanceof Param) {
                    paramAnnotation = (Param)a;
                    break;
                }
            }
            if (paramAnnotation != null) {
                assert !paramAnnotation.name().isEmpty() : "@Param.name mustn\'t be empty";
                result[i] = new ShellCommandParamSpec(paramAnnotation.name(), paramTypes[i],
                        paramAnnotation.description(), i);
            } else {
                result[i] = new ShellCommandParamSpec(String.format("p%d", i + 1), paramTypes[i], "", i);
            }
        }
        return result;
    }
    
    
    private static String makeCommandDescription(Method method, ShellCommandParamSpec[] paramSpecs) {
        StringBuilder result = new StringBuilder();
        result.append(method.getName());
        result.append('(');
        Class[] paramTypes = method.getParameterTypes();
        assert paramTypes.length == paramSpecs.length;
        boolean first = true;
        for (int i = 0; i < paramTypes.length; i++) {
            if (!first) {
                result.append(", ");
            }
            first = false;
            if (paramSpecs[i] != null) {
                result.append(paramSpecs[i].getName());
                result.append(":");
                result.append(paramTypes[i].getSimpleName());
            } else {
                result.append(paramTypes[i].getSimpleName());
            }
        }
        result.append(") : ");
        result.append(method.getReturnType().getSimpleName());
        return result.toString();
    }

    public Object invoke(Object[] parameters)
            throws CLIException {
        assert method != null;
        try {
            Object result = method.invoke(handler, parameters);
            return result;
        } catch (InvocationTargetException ite) {
            return ite.getCause();
        } catch (Exception ex) {
            throw new CLIException(ex);
        }
    }


}
