/*
 * This file is part of the Cliche project, licensed under MIT License.
 * See LICENSE.txt file in root folder of Cliche sources.
 */
package org.lsst.ccs.shell;

import org.lsst.ccs.shell.util.Strings;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintStream;
import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import jline.console.ConsoleReader;
import jline.console.completer.Completer;
import jline.console.completer.StringsCompleter;
import jline.console.history.FileHistory;
import jline.console.history.MemoryHistory;
import jline.internal.Configuration;

/**
 * Console IO subsystem. This is also one of special command handlers and is
 * responsible for logging (duplicating output) and execution of scripts.
 *
 * @author ASG
 */
public class ConsoleIO implements Input, Output, ShellManageable {

    ConsoleReader consoleReader;
    boolean enableFileHistory = false;

//    public ConsoleIO(BufferedReader in, PrintStream out, PrintStream err) {
//        this.in = in;
//        this.out = out;
//        this.err = err;
//    }
    public ConsoleIO() {
        try {
            consoleReader = new ConsoleReader(System.in, System.out);
            String fileHistoryOutputName = System.getProperty("org.lsst.ccs.cliche.file.history", "");
            String history = enableFileHistory(fileHistoryOutputName);
            System.out.println(history);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
        this.err = System.err;
//        this(new BufferedReader(new InputStreamReader(System.in)),
//                System.out, System.err);
    }
    private BufferedReader in;
    private PrintStream out;
    private PrintStream err;
    private int lastCommandOffset = 0;

    public String readCommand(List<String> path) {
        try {
            String prompt = Strings.joinStrings(path, false, '/');
            switch (inputState) {
                case USER:
                    return readUsersCommand(prompt);
                case SCRIPT:
                    String command = readCommandFromScript(prompt);
                    if (command != null) {
                        return command;
                    } else {
                        closeScript();
                        return readUsersCommand(prompt);
                    }
            }
            return readUsersCommand(prompt);
        } catch (IOException ex) {
            throw new Error(ex);
        }
    }
    private static final String USER_PROMPT_SUFFIX = "> ";
    private static final String FILE_PROMPT_SUFFIX = "$ ";

    private static enum InputState {

        USER, SCRIPT
    }
    private InputState inputState = InputState.USER;

    public void setCommandTable(CommandTable commandTable) {
        List<ShellCommand> commands = commandTable.getCommandTable();
        ArrayList<String> commandNames = new ArrayList<String>();
        for (ShellCommand s : commands) {
            commandNames.add(s.getName());
        }
        StringsCompleter completer = new StringsCompleter(commandNames);
        consoleReader.addCompleter(completer);
    }
    
    public void addCompleter(Completer completer) {
        consoleReader.addCompleter(completer);
    }

    private String readUsersCommand(String prompt) throws IOException {

        String completePrompt = prompt + USER_PROMPT_SUFFIX;
        consoleReader.setPrompt(completePrompt);

//        print(completePrompt);
        lastCommandOffset = completePrompt.length();

        String command = "";
        try {
            command = consoleReader.readLine();
        } catch (Exception e) {};

        if ( consoleReader.getHistory() instanceof FileHistory ) {
            ((FileHistory) consoleReader.getHistory()).flush();
        }

//        String command = in.readLine();
        if (log != null) {
            log.println(command);
        }
        return command;
    }
    private BufferedReader scriptReader = null;

    private String readCommandFromScript(String prompt) throws IOException {
        String command = scriptReader.readLine();
        if (command != null) {
            String completePrompt = prompt + FILE_PROMPT_SUFFIX;
            print(completePrompt);
            lastCommandOffset = completePrompt.length();
        }
        return command;
    }

    private void closeScript() throws IOException {
        if (scriptReader != null) {
            scriptReader.close();
            scriptReader = null;
        }
        inputState = InputState.USER;
    }

    @Command(description = "Reads commands from file")
    public void runScript(
            @Param(name = "filename", description = "Full file name of the script") String filename) throws FileNotFoundException {

        scriptReader = new BufferedReader(new InputStreamReader(new FileInputStream(filename)));
        inputState = InputState.SCRIPT;
    }

    public void outputHeader(String text) {
        if (text != null) {
            println(text);
        }
    }

    public void output(Object obj, OutputConversionEngine oce) {
        if (obj == null) {
            return;
        } else {
            obj = oce.convertOutput(obj);
        }

        if (obj.getClass().isArray()) {
            int length = Array.getLength(obj);
            for (int i = 0; i < length; i++) {
                output(Array.get(obj, i), 0, oce);
            }
        } else if (obj instanceof Collection) {
            for (Object elem : (Collection) obj) {
                output(elem, 0, oce);
            }
        } else {
            output(obj, 0, oce);
        }
    }

    private void output(Object obj, int indent, OutputConversionEngine oce) {
        if (obj == null) {
            return;
        }

        if (obj != null) {
            obj = oce.convertOutput(obj);
        }

        for (int i = 0; i < indent; i++) {
            print("\t");
        }

        if (obj == null) {
            println("(null)");
        } else if (obj.getClass().isPrimitive() || obj instanceof String) {
            println(obj);
        } else if (obj.getClass().isArray()) {
            println("Array");
            int length = Array.getLength(obj);
            for (int i = 0; i < length; i++) {
                output(Array.get(obj, i), indent + 1, oce);
            }
        } else if (obj instanceof Collection) {
            println("Collection");
            for (Object elem : (Collection) obj) {
                output(elem, indent + 1, oce);
            }
        } else if (obj instanceof Throwable) {
            println(obj); // class and its message
            ((Throwable) obj).printStackTrace(out);
        } else {
            println(obj);
        }
    }

    private void print(Object x) {
        try {
            consoleReader.print(x.toString());
        } catch (Exception e) {
            e.printStackTrace();
        }

//        out.print(x);
        if (log != null) {
            log.print(x);
        }
    }

    private void println(Object x) {
        try {
            consoleReader.print(x.toString());
            consoleReader.println();
        } catch (Exception e) {
            e.printStackTrace();
        }
//        out.println(x);
        if (log != null) {
            log.println(x);
        }
    }

    private void printErr(Object x) {
        err.print(x);
        if (log != null) {
            log.print(x);
        }
    }

    private void printlnErr(Object x) {
        err.println(x);
        if (log != null) {
            log.println(x);
        }
    }

    public void outputException(String input, TokenException error) {
        int errIndex = error.getToken().getIndex() + lastCommandOffset;
        while (errIndex-- > 0) {
            printErr("-");
        }
        for (int i = 0; i < error.getToken().getString().length(); i++) {
            printErr("^");
        }
        printlnErr("");
        printlnErr(error);
    }

    public void outputException(Throwable e) {
        printlnErr(e);
        if (e.getCause() != null) {
            printlnErr(e.getCause());
        }
    }
    private PrintStream log = null;

    private boolean isLoggingEnabled() {
        return log != null;
    }
    private int loopCounter = 0;

    public void cliEnterLoop() {
        if (isLoggingEnabled()) {
            loopCounter++;
        }
    }

    public void cliLeaveLoop() {
        if (isLoggingEnabled()) {
            loopCounter--;
        }
        if (loopCounter < 0) {
            disableLogging();
        }
    }

    @Command(description = "Sets up logging, which duplicates all subsequent output in a file")
    public void enableLogging(
            @Param(name = "fileName", description = "Name of the logfile") String filename) throws FileNotFoundException {

        log = new PrintStream(filename);
        loopCounter = 0;
    }

    @Command(description = "Turns off logging")
    public String disableLogging() {
        if (log != null) {
            log.close();
            log = null;
            return "Logging disabled";
        } else {
            return "Logging is already disabled";
        }
    }

    @Command(description = "Turn On File History")
    public String enableFileHistory(
            @Param(name = "file-name", description = "Path to the file to save the table to.") String filename) {
        if (!"".equals(filename)) {
            File out = null;
            try {
                out = new File(filename);
                if ( out.getParent() == null ) {
                    out = new File(Configuration.getUserHome(),filename);
                }
                consoleReader.setHistory(new FileHistory(out));
                enableFileHistory = true;
            } catch (IOException ioe) {
                return "Could not open file history " + filename + "\n" + ioe.getMessage();
            }
            return "Writing command history to file "+out.getAbsolutePath();
        } else {
            enableFileHistory = false;
            consoleReader.setHistory(new MemoryHistory());
            return "Using in memory history";
        }
    }

    @Command(description = "Turn Off File History")
    public String disableFileHistory() {
        return enableFileHistory("");
    }
    
    @Command(description = "Displays current command history")
    public String history() {
        int histSize = consoleReader.getHistory().size();
        if ( histSize == 0 )
            return "No commands are currently in the history";
        StringBuffer output = new StringBuffer();
        for ( int i = 0; i < histSize; i++ ) {
            output.append(" "+i+" "+consoleReader.getHistory().get(i)+"\n");            
        }
        return output.toString();
    }
    
    
}
