package org.lsst.ccs.rest.file.server.client.implementation;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InterruptedIOException;
import java.io.OutputStream;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.lang.reflect.Constructor;
import java.net.URI;
import java.nio.file.AccessMode;
import java.nio.file.CopyOption;
import java.nio.file.DirectoryStream;
import java.nio.file.FileSystem;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.nio.file.attribute.BasicFileAttributeView;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.stream.Collectors;
import javax.ws.rs.client.Client;
import javax.ws.rs.client.ClientBuilder;
import javax.ws.rs.client.Entity;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.UriBuilder;
import org.lsst.ccs.rest.file.server.client.VersionOpenOption;
import org.lsst.ccs.rest.file.server.client.VersionOption;
import org.lsst.ccs.rest.file.server.client.VersionedFileAttributeView;
import org.lsst.ccs.rest.file.server.client.VersionedFileAttributes;
import org.lsst.ccs.web.rest.file.server.data.IOExceptionResponse;
import org.lsst.ccs.web.rest.file.server.data.VersionInfo;
import org.lsst.ccs.web.rest.file.server.data.RestFileInfo;

/**
 *
 * @author tonyj
 */
class RestPath extends AbstractPath {

    private final RestFileSystem fileSystem;
    private final LinkedList<String> path;
    private final boolean isReadOnly;
    private final boolean isAbsolute;

    RestPath(RestFileSystem fileSystem, String path, boolean isReadOnly) {
        this.fileSystem = fileSystem;
        this.path = new LinkedList<>(Arrays.asList(path.split("/")));
        this.isReadOnly = isReadOnly;
        this.isAbsolute = path.startsWith("/");
    }

    RestPath(RestFileSystem fileSystem, List<String> path, boolean isReadOnly, boolean isAbsolute) {
        this.fileSystem = fileSystem;
        this.path = new LinkedList<>(path);
        this.isReadOnly = isReadOnly;
        this.isAbsolute = isAbsolute;
    }

    @Override
    public FileSystem getFileSystem() {
        return fileSystem;
    }

    @Override
    public boolean isAbsolute() {
        return isAbsolute;
    }

    @Override
    public Path getRoot() {
        return fileSystem.getRootDirectories().iterator().next();
    }

    @Override
    public Path getFileName() {
        return new RestPath(fileSystem, path.getLast(), isReadOnly);
    }

    @Override
    public Path getParent() {
        return path.isEmpty() ? null : new RestPath(fileSystem, path.subList(0, path.size() - 1), isReadOnly, isAbsolute);
    }

    @Override
    public int getNameCount() {
        return path.size();
    }

    @Override
    public Path getName(int index) {
        return new RestPath(fileSystem, path.get(index), isReadOnly);
    }

    @Override
    public Path subpath(int beginIndex, int endIndex) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    @Override
    public Path normalize() {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    @Override
    public Path relativize(Path other) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    @Override
    public URI toUri() {
        return fileSystem.getURI(path);
    }

    @Override
    public Path toAbsolutePath() {
        if (isAbsolute) {
            return this;
        } else {
            return new RestPath(fileSystem, path, isReadOnly, true);
        }
    }

    @Override
    public Path toRealPath(LinkOption... options) throws IOException {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    @Override
    public WatchKey register(WatchService watcher, WatchEvent.Kind<?>[] events, WatchEvent.Modifier... modifiers) throws IOException {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    @Override
    public int compareTo(Path other) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    @Override
    public String toString() {
        return (isAbsolute ? "/" : "") + String.join("/", path);
    }

    InputStream newInputStream(OpenOption[] options) throws IOException {
        if (isVersionedFile()) {
            VersionOption vo = getOptions(options, VersionOption.class);
            if (vo == null) {
                vo = VersionOption.DEFAULT;
            }
            URI uri = fileSystem.getRestURI("rest/version/download/", path);
            uri = UriBuilder.fromUri(uri).queryParam("version", vo.value()).build();
            return uri.toURL().openStream();
        } else {
            URI uri = fileSystem.getRestURI("rest/download/", path);
            return uri.toURL().openStream();
        }
    }

    OutputStream newOutputStream(OpenOption[] options) throws IOException {

        // TODO: Deal with options
        VersionOpenOption voo = getOptions(options, VersionOpenOption.class);
        String restPath = isVersionedFile() || voo != null ? "rest/version/upload/" : "rest/upload/" ;
        URI uri = fileSystem.getRestURI(restPath, path);
        BlockingQueue<Future<Response>> queue = new ArrayBlockingQueue<>(1);
        PipedOutputStream out = new PipedOutputStream() {
            @Override
            public void close() throws IOException {
                super.close();
                try {
                    Response response = queue.take().get();
                    checkResponse(response);
                } catch (InterruptedException x) {
                    throw new InterruptedIOException("Interrupt during file close");
                } catch (ExecutionException x) {
                    throw new IOException("Error during file close", x.getCause());
                }
            }
            
        };
        Client client = ClientBuilder.newClient();
        PipedInputStream in = new PipedInputStream(out);
        Future<Response> futureResponse = client.target(uri).request(MediaType.APPLICATION_JSON).async().post(Entity.entity(in, MediaType.APPLICATION_OCTET_STREAM));
        queue.add(futureResponse);
        return out;
    }

    void move(RestPath target, CopyOption[] options) throws IOException {
        Client client = ClientBuilder.newClient();
        URI uri = UriBuilder.fromUri(fileSystem.getRestURI("rest/move/", path)).queryParam("target", String.join("/", target.path)).build();
        Response response = client.target(uri).request(MediaType.APPLICATION_JSON).get();
        checkResponse(response);
    }

    private <T extends OpenOption> T getOptions(OpenOption[] options, Class<T> optionClass) {
        for (OpenOption option : options) {
            if (optionClass.isInstance(option)) {
                return optionClass.cast(option);
            }
        }
        return null;
    }

    BasicFileAttributes getAttributes() throws IOException {
        if (isVersionedFile()) {
            VersionInfo info = getVersionedRestFileInfo();
            return new RestFileAttributes(info.getVersions().get(info.getDefault()-1));
        } else {
            RestFileInfo info = getRestFileInfo();
            return new RestFileAttributes(info);
        }
    }

    VersionedFileAttributes getVersionedAttributes() throws IOException {
        if (!isVersionedFile()) {
            throw new IOException("Cannot read versioned attributes for non-versioned file");
        }
        Client client = ClientBuilder.newClient();
        Response response = client.target(fileSystem.getRestURI("rest/version/info/", path)).request(MediaType.APPLICATION_JSON).get();
        checkResponse(response);
        VersionInfo info = response.readEntity(VersionInfo.class);
        return new RestVersionedFileAttributes(info);
    }

    private VersionInfo getVersionedRestFileInfo() throws IOException {
        Client client = ClientBuilder.newClient();
        Response response = client.target(fileSystem.getRestURI("rest/version/info/", path)).request(MediaType.APPLICATION_JSON).get();
        checkResponse(response);
        VersionInfo info = response.readEntity(VersionInfo.class);
        return info;
    }

    private RestFileInfo getRestFileInfo() throws IOException {
        Client client = ClientBuilder.newClient();
        Response response = client.target(fileSystem.getRestURI("rest/info/", path)).request(MediaType.APPLICATION_JSON).get();
        checkResponse(response);
        RestFileInfo info = response.readEntity(RestFileInfo.class);
        return info;
    }

    private void checkResponse(Response response) throws IOException {
        if (response.getStatus() != Response.Status.OK.getStatusCode()) {
            if (response.getStatus() == IOExceptionResponse.RESPONSE_CODE) {
                IOExceptionResponse ioError = response.readEntity(IOExceptionResponse.class);
                try {
                    Class<? extends IOException> exceptionClass = Class.forName(ioError.getExceptionClass()).asSubclass(IOException.class);
                    Constructor<? extends IOException> constructor = exceptionClass.getConstructor(String.class);
                    IOException io = constructor.newInstance(ioError.getMessage());
                    throw io;
                } catch (ReflectiveOperationException ex) {
                    throw new IOException("Remote Exception " + ioError.getExceptionClass() + " " + ioError.getMessage());
                }
            } else {
                throw new IOException("Response code " + response.getStatus() + " " + response.getStatusInfo());
            }
        }
    }

    BasicFileAttributeView getFileAttributeView() {
        return new BasicFileAttributeView() {
            @Override
            public String name() {
                return "basic";
            }

            @Override
            public BasicFileAttributes readAttributes() throws IOException {
                return getAttributes();
            }

            @Override
            public void setTimes(FileTime lastModifiedTime, FileTime lastAccessTime, FileTime createTime) throws IOException {
                throw new UnsupportedOperationException("Not supported yet.");
            }
        };
    }

    VersionedFileAttributeView getVersionedAttributeView() {
        return new VersionedFileAttributeView() {

            @Override
            public void setDefaultVersion(int version) throws IOException {
                Client client = ClientBuilder.newClient();
                URI uri = fileSystem.getRestURI("rest/version/set/", path);
                Response response = client.target(uri).request(MediaType.APPLICATION_JSON).put(Entity.entity(version, MediaType.APPLICATION_JSON));
                checkResponse(response);            
            }

            @Override
            public String name() {
                return "versioned";
            }

            @Override
            public VersionedFileAttributes readAttributes() throws IOException {
                return getVersionedAttributes();
            }

        };
    }

    void checkAccess(AccessMode... modes) throws IOException {
        Client client = ClientBuilder.newClient();
        Response response = client.target(fileSystem.getRestURI("rest/list/", path)).request(MediaType.APPLICATION_JSON).get();
        checkResponse(response);
        response.readEntity(RestFileInfo.class);
    }

    DirectoryStream<Path> newDirectoryStream(DirectoryStream.Filter<? super Path> filter) throws IOException {
        Client client = ClientBuilder.newClient();
        Response response = client.target(fileSystem.getRestURI("rest/list/", path)).request(MediaType.APPLICATION_JSON).get();
        checkResponse(response);
        RestFileInfo dirList = response.readEntity(RestFileInfo.class);
        List<Path> paths = dirList.getChildren().stream().map(fileInfo -> {
            List<String> newPath = new ArrayList<>(path);
            newPath.add(fileInfo.getName());
            return new RestPath(fileSystem, newPath, isReadOnly, true);
        }).collect(Collectors.toList());
        return new DirectoryStream<Path>() {
            @Override
            public Iterator<Path> iterator() {
                return paths.iterator();
            }

            @Override
            public void close() throws IOException {
            }
        };
    }

    void delete() throws IOException {
        Client client = ClientBuilder.newClient();
        String restPath = isVersionedFile() ? "rest/version/deleteFile/" : "rest/deleteFile/";
        Response response = client.target(fileSystem.getRestURI(restPath, path)).request(MediaType.APPLICATION_JSON).delete();
        checkResponse(response);
    }

    void createDirectory(FileAttribute<?>[] attrs) throws IOException {
        Client client = ClientBuilder.newClient();
        Response response = client.target(fileSystem.getRestURI("rest/createDirectory/", path)).request(MediaType.APPLICATION_JSON).get();
        checkResponse(response);
    }

    Map<String, Object> readAttributes(String attributes) throws IOException {
        return getRestFileInfo().getAsMap();
    }

    private boolean isVersionedFile() throws IOException {
        try {
            RestFileInfo info = getRestFileInfo();
            // TODO: Cache this?
            return info.isVersionedFile();
        } catch (NoSuchFileException | FileNotFoundException x) {
            return false;
        }
    }

    @Override
    public boolean startsWith(Path other) {
        if (!other.getFileSystem().equals(this.getFileSystem())) {
            return false;
        }

        RestPath otherPath = (RestPath) other;
        if (otherPath.path.size() > this.path.size()) {
            return false;
        }
        for (int i = 0; i < otherPath.path.size(); i++) {
            if (!otherPath.path.get(i).equals(this.path.get(i))) {
                return false;
            }
        }
        return true;
    }

    @Override
    public boolean endsWith(Path other) {
        if (!other.getFileSystem().equals(this.getFileSystem())) {
            return false;
        }

        RestPath otherPath = (RestPath) other;
        if (otherPath.path.size() > this.path.size()) {
            return false;
        }
        for (int i = otherPath.path.size() - 1; i > 0; i--) {
            if (!otherPath.path.get(i).equals(this.path.get(i))) {
                return false;
            }
        }
        return true;
    }

    @Override
    public Path resolve(Path other) {
        if (other.isAbsolute()) {
            return other;
        }
        if (other.getNameCount() == 0) {
            return this;
        }
        List<String> newPath = new LinkedList<>(this.path);
        for (int i = 0; i < other.getNameCount(); i++) {
            newPath.add(other.getName(i).toString());
        }
        return new RestPath(this.fileSystem, newPath, this.isReadOnly, this.isAbsolute);
    }

    @Override
    public int hashCode() {
        int hash = 5;
        hash = 89 * hash + Objects.hashCode(this.fileSystem);
        hash = 89 * hash + Objects.hashCode(this.path);
        hash = 89 * hash + (this.isAbsolute ? 1 : 0);
        return hash;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final RestPath other = (RestPath) obj;
        if (this.isAbsolute != other.isAbsolute) {
            return false;
        }
        if (!Objects.equals(this.fileSystem, other.fileSystem)) {
            return false;
        }
        return Objects.equals(this.path, other.path);
    }

    boolean isSameFile(RestPath other) {
        return this.toAbsolutePath().equals(other.toAbsolutePath());
    }

}
