package org.lsst.ccs.messaging.mock;

import org.lsst.ccs.bus.definition.Bus;
import org.lsst.ccs.messaging.DestinationsException;
import org.lsst.ccs.messaging.BusMessagingLayer;
import org.lsst.ccs.messaging.BusMessageForwarder;
import org.lsst.ccs.utilities.dispatch.CommandFor;
import org.lsst.ccs.utilities.dispatch.ParallelCommandDispatcher;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Set;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.CommandMessage;
import static org.lsst.ccs.messaging.BusMessagingLayer.parseDestination;
import org.lsst.ccs.messaging.DuplicateAgentNameException;
import org.lsst.ccs.messaging.MessagingAccessLayer;
import org.lsst.ccs.messaging.MessagingAccessLayer.BusAccess;

/**
 */
class MockBusMessagingLayer implements BusMessagingLayer {

    

    /**
     * for each bus keeps a list of registered "agents" (mostly subsystems)
     */
    private static class BusRegistry {
        ArrayList<String>[] registry;

        BusRegistry() {
            Bus[] buses = Bus.values();
            registry = new ArrayList[buses.length];
            for (int ix = 0; ix < registry.length; ix++) {
                registry[ix] = new ArrayList<String>();
            }
        }

    }

    private static class AgentBehaviour {
        ParallelCommandDispatcher[] dispatchers;

        AgentBehaviour() {
            Bus[] buses = Bus.values();
            dispatchers = new ParallelCommandDispatcher[buses.length];
            // NO only at registration!! change!
            /*
             for(int ix = 0 ; ix < dispatchers.length ; ix++) {
             dispatchers[ix] = new ParallelCommandDispatcher() ;
             }
             */
        }

    }

    private final HashMap<String, AgentBehaviour> agentMap = new HashMap<String, AgentBehaviour>();
    private final BusRegistry busRegistry = new BusRegistry();
    private volatile boolean isClosed = false;
    //TODO: in this implementation there is only one Listener per bus: this should be changed
    private final Object lock = new Object();

    private boolean single;

    MockBusMessagingLayer(boolean single) {
        this.single = single;
    }

    @Override
    public void register(String agentName, Bus... buses) throws IOException, DuplicateAgentNameException{
        //TODO: throw exception?
        // TODO: problem with multiple tests in same JVM
        //if (isClosed) return;
        if (isClosed) {
            isClosed = false;
    }
        // if buses is empty
        if (buses.length == 0) {
            buses = Bus.values();
        }
        synchronized (lock) {
            AgentBehaviour behaviour = agentMap.get(agentName);
            if (behaviour == null) {
                behaviour = new AgentBehaviour();
                agentMap.put(agentName, behaviour);
            } else { //TODO stupid addition
//                throw new DuplicateBusNameException(agentName, "Agent "+agentName+" is already registered with this Bus Messaging Layer");
            }
            for (Bus bus : buses) {
                int index = bus.ordinal();
                if (busRegistry.registry[index].contains(agentName)) {
                    throw new DuplicateAgentNameException(agentName, agentName);
                } else {
                    busRegistry.registry[index].add(agentName);
                    if (null == behaviour.dispatchers[index]) {
                        behaviour.dispatchers[index] = new ParallelCommandDispatcher(single);
                    }
                }
            }

        }
    }

    @Override
    public Set<String> getRegisteredLocalAgents(Bus... buses) {

        // If there are no provided buses, it means for all buses.
        if (buses == null || buses.length == 0) {
            buses = Bus.values();
        }
        Set<String> result = new HashSet<>();

        synchronized (lock) {
            for (String agent : agentMap.keySet()) {
                for (Bus bus : buses) {
                    int index = bus.ordinal();
                    if (busRegistry.registry[index].contains(agent)) {
                        if (!result.contains(agent)) {
                            result.add(agent);
                        }
                    }
                }
            }
        }
        return result;
    }

    @Override
    public void closeFor(String agentName, Bus... buses
    ) {
        //TODO: throw exception?
        if (isClosed) {
            return;
        }
        //TODO : stupid!
        //if (!isClosed) return;
        // if agent name null or ""
        // if buses is empty
        if (buses.length == 0) {
            buses = Bus.values();
        }
        synchronized (lock) {
            AgentBehaviour behaviour = agentMap.remove(agentName);
            for (Bus bus : buses) {
                int index = bus.ordinal();
                busRegistry.registry[index].remove(agentName);
                if (behaviour != null) {
                    behaviour.dispatchers[index] = null;
                }
            }
        }
    }

    @Override
    public void close() throws IOException {
        //TODO: change that!
        isClosed = true;
    }

    @Override
    public synchronized <T extends BusMessage> void sendMessage(String senderAgent, Bus<T> bus, final T message) throws IOException {
        if (bus == null) {
            throw new IllegalArgumentException("null bus");
        }
        if (message == null) {
            throw new IllegalArgumentException("null message");
        }
        synchronized (lock) {
            // is sendAgent registered for this bus?
            ArrayList<String> busAgents = busRegistry.registry[bus.ordinal()];
            if (!busAgents.contains(senderAgent)) {
                throw new IllegalArgumentException("agent " + senderAgent + "not registered on " + bus);
            }
            int index = bus.ordinal();
            
            CommandFor<BusAccess> cmdMessage = new CommandFor<BusAccess>() {
                @Override
                public void invokeOn(BusAccess instance) {
                    ArrayList<BusMessageForwarder> forwarderList = instance.getForwarderList();
                    for (BusMessageForwarder forwarder : forwarderList){
                        forwarder.update(message);
                    }
                    instance.processBusMessage(message);
                }
            };
            if (message instanceof CommandMessage){
                String destination = parseDestination(((CommandMessage)message).getDestination());
                // Does the destination exist ?
                AgentBehaviour behaviour = agentMap.get(destination);
                if (behaviour == null){
                    throw new DestinationsException(senderAgent, destination);
                }
                ParallelCommandDispatcher dispatcher = behaviour.dispatchers[index];
                if (dispatcher == null){
                    throw new DestinationsException(senderAgent, destination);
                }
            }
            // The message is broadcast
            for (String destination : busRegistry.registry[index]) {
                AgentBehaviour behaviour = agentMap.get(destination);
                if (behaviour == null) {
                    continue;
                }
                ParallelCommandDispatcher dispatcher = behaviour.dispatchers[index];
                if (dispatcher == null) {
                    continue;
                }
                dispatcher.dispatchCommand(cmdMessage);
            }
        }
    }

    @Override
    public void addMessageListener(String agentName, BusMessageForwarder forwarder, Bus... buses) {
        if (forwarder == null) {
            throw new IllegalArgumentException("null forwarder");
        }
        //TODO: throw exception?
        //TODO: change that!
        if (isClosed) {
            return;
        }
        // if buses is empty
        if (buses.length == 0) {
            buses = Bus.values();
        }
        synchronized (lock) {
            AgentBehaviour behaviour = agentMap.get(agentName);
            if (behaviour == null) {
                throw new IllegalArgumentException("agent with no registration");
            }

            for (Bus bus : buses) {
                int index = bus.ordinal();
                ParallelCommandDispatcher dispatcher = behaviour.dispatchers[index];
                if (dispatcher == null) {
                    throw new IllegalArgumentException(agentName + " not registered with " + bus);
                }
                dispatcher.addExecutant(forwarder);
            }
        }

    }

    @Override
    public void removeMessageListener(String agentName, BusMessageForwarder forwarder, Bus... buses) {
        if (forwarder == null) {
            throw new IllegalArgumentException("null forwarder");
        }
        //TODO: throw exception?
        //TODO: change that
        if (isClosed) {
            return;
        }
        // if buses is empty
        if (buses.length == 0) {
            buses = Bus.values();
        }
        synchronized (lock) {
            AgentBehaviour behaviour = agentMap.get(agentName);
            if (behaviour == null) {
                return;
            }

            for (Bus bus : buses) {
                int index = bus.ordinal();
                ParallelCommandDispatcher dispatcher = behaviour.dispatchers[index];
                if (dispatcher == null) {
                    return;
                }
                dispatcher.removeExecutant(forwarder);
            }
        }
    }
    
    @Override
    public void connect(MessagingAccessLayer layer) throws DuplicateAgentNameException, IOException {
        //TODO: throw exception?
        // TODO: problem with multiple tests in same JVM
        //if (isClosed) return;
        if (isClosed) {
            isClosed = false;
        }
        String agentName = layer.getName();
      
        synchronized (lock) {
            
            AgentBehaviour behaviour = agentMap.get(agentName);
            if (behaviour == null) {
                behaviour = new AgentBehaviour();
                agentMap.put(agentName, behaviour);
            } else { //TODO stupid addition
//                throw new DuplicateBusNameException(agentName, "Agent "+agentName+" is already registered with this Bus Messaging Layer");
            }
            for (BusAccess busAccess : layer.getBusAccesses()){
                int index = busAccess.getBus().ordinal();
                if (busRegistry.registry[index].contains(agentName)) {
                    throw new DuplicateAgentNameException(agentName, agentName);
                } else {
                    busRegistry.registry[index].add(agentName);
                    if (null == behaviour.dispatchers[index]) {
                        behaviour.dispatchers[index] = new ParallelCommandDispatcher(single);
                        behaviour.dispatchers[index].addExecutant(busAccess);
                    }
                }
            }
        }
    }
    
    @Override
    public void disconnect(MessagingAccessLayer accessLayer) {
        Bus[] buses = new Bus[accessLayer.getBusAccesses().size()];
        for (BusAccess busAccess : accessLayer.getBusAccesses()){
            buses[busAccess.getBus().ordinal()] = busAccess.getBus();
        }
        this.closeFor(accessLayer.getName(), buses);
    }
    
}
