package org.lsst.ccs.bus.mock;

import org.lsst.ccs.bus.*;
import org.lsst.ccs.utilities.dispatch.CommandFor;
import org.lsst.ccs.utilities.dispatch.ParallelCommandDispatcher;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;

/**
 */
public class MockBusMessagingLayer implements BusMessagingLayer, ProvidesDisconnectionInformation {
    /**
     * for each bus keeps a list of registered "agents" (mostly subsystems)
     */
    static class BusRegistry {
        ArrayList<String>[] registry;

        BusRegistry() {
            Bus[] buses = Bus.values();
            registry = new ArrayList[buses.length];
            for (int ix = 0; ix < registry.length; ix++) {
                registry[ix] = new ArrayList<String>();
            }
        }

    }

    static class AgentBehaviour {
        ParallelCommandDispatcher[] dispatchers;

        AgentBehaviour() {
            Bus[] buses = Bus.values();
            dispatchers = new ParallelCommandDispatcher[buses.length];
            // NO only at registration!! change!
            /*
            for(int ix = 0 ; ix < dispatchers.length ; ix++) {
                dispatchers[ix] = new ParallelCommandDispatcher() ;
            }
            */
        }

    }

    private static final HashMap<String, AgentBehaviour> agentMap = new HashMap<String, AgentBehaviour>();
    private static final BusRegistry busRegistry = new BusRegistry();
    private volatile boolean isClosed = false;
    //TODO: in this implementation there is only one Listener per bus: this should be changed
    private static final BusMembershipListener[] membershipListeners = new BusMembershipListener[Bus.values().length];
    private static final Object lock = new Object();

    private boolean single ;


    public MockBusMessagingLayer() {

    }

    public MockBusMessagingLayer(boolean single) {
        this.single = single ;
    }

    @Override
    public void register(String agentName, Bus... buses) {
        //TODO: throw exception?
        // TODO: problem with multiple tests in same JVM
        //if (isClosed) return;
        if(isClosed) {isClosed = false ;}
        // if agent name null or ""
        if ((agentName == null) || ("".equals(agentName))) {
            agentName = ANONYMOUS_AGENT;
        }
        // if buses is empty
        if (buses.length == 0) {
            buses = Bus.values();
        }
        synchronized (lock) {
            AgentBehaviour behaviour = agentMap.get(agentName);
            if (behaviour == null) {
                behaviour = new AgentBehaviour();
                agentMap.put(agentName, behaviour);
            } else { //TODO stupid addition
                //behaviour = new AgentBehaviour();
                //agentMap.put(agentName, behaviour);
            }
            for (Bus bus : buses) {
                int index = bus.ordinal();
                busRegistry.registry[index].add(agentName);
                //TODO: there might be numerous Listeners!
                BusMembershipListener listener = membershipListeners[index];
                if (listener != null) {
                    listener.connecting(agentName, "");
                }
                if (null == behaviour.dispatchers[index]) {
                    behaviour.dispatchers[index] = new ParallelCommandDispatcher(single);
                }
            }
        }
    }

    @Override
    public void closeFor(String agentName, Bus... buses) {
        //TODO: throw exception?
        if (isClosed) return;
        //TODO : stupid!
        //if (!isClosed) return;
        // if agent name null or ""
        if ((agentName == null) || ("".equals(agentName))) {
            agentName = ANONYMOUS_AGENT;
        }
        // if buses is empty
        if (buses.length == 0) {
            buses = Bus.values();
        }
        synchronized (lock) {
            AgentBehaviour behaviour = agentMap.get(agentName);
            for (Bus bus : buses) {
                int index = bus.ordinal();
                busRegistry.registry[index].remove(agentName);
                BusMembershipListener listener = membershipListeners[index];
                if (listener != null) {
                    listener.disconnecting(agentName, "");
                }
                if (behaviour != null) {
                    behaviour.dispatchers[index] = null;
                }
            }
        }
    }

    @Override
    public void close() throws IOException {
        //TODO: change that!
        isClosed = true;
    }


    @Override
    public synchronized <T extends BusPayload> void sendMessage(String senderAgent, Bus<T> bus, final T message, String... destinations) throws IOException {
        if (bus == null) throw new IllegalArgumentException("null bus");
        if (message == null) throw new IllegalArgumentException("null message");
        synchronized (lock) {
            // is sendAgent registered for this bus?
            ArrayList<String> busAgents = busRegistry.registry[bus.ordinal()];
            if (!busAgents.contains(senderAgent)) {
                throw new IllegalArgumentException("agent " + senderAgent + "not registered on " + bus);
            }
            if (destinations.length == 0 || "".equals(destinations[0]) || "*".equals(destinations[0])) {
                //send to all
                int index = bus.ordinal();
                destinations = (busRegistry.registry[index]).toArray(destinations);
            } else { // adds ANONYMOUS_AGENT to destinations
                int length = destinations.length;
                destinations = Arrays.copyOf(destinations, length + 1);
                destinations[length] = ANONYMOUS_AGENT;

            }
            CommandFor<BusMessageForwarder> cmdMessage = new CommandFor<BusMessageForwarder>() {
                @Override
                public void invokeOn(BusMessageForwarder instance) {
                    instance.update((BusMessage) message);
                }
            };

            List<String> failed = new ArrayList<String>();
            for (String destination : destinations) {
                AgentBehaviour behaviour = agentMap.get(destination);
                if (behaviour == null) {
                    if (!destination.equals(ANONYMOUS_AGENT)) {
                        failed.add(destination);
                    }
                    continue;
                    //throw new IllegalArgumentException(destination + " agent with no registration");
                    //return ;
                }
                int index = bus.ordinal();
                ParallelCommandDispatcher dispatcher = behaviour.dispatchers[index];
                if (dispatcher == null) {
                    // TODO change this!!!
                    if (!destination.equals(ANONYMOUS_AGENT)) {
                        failed.add(destination);
                    }
                    //throw new IllegalArgumentException(destination + " not registered with " + bus);
                    //return ;
                    continue;
                }
                dispatcher.dispatchCommand(cmdMessage);
            }
            if (failed.size() != 0) {
                throw new DestinationsException(failed.toArray());
            }
        }

    }

    @Override
    public void addMessageListener(String agentName, BusMessageForwarder forwarder, Bus... buses) {
        if (forwarder == null) throw new IllegalArgumentException("null forwarder");
        //TODO: throw exception?
        //TODO: change that!
        if (isClosed) return;
        // if agent name null or ""
        if ((agentName == null) || ("".equals(agentName))) {
            agentName = ANONYMOUS_AGENT;
        }
        // if buses is empty
        if (buses.length == 0) {
            buses = Bus.values();
        }
        synchronized (lock) {
            AgentBehaviour behaviour = agentMap.get(agentName);
            if (behaviour == null) {
                throw new IllegalArgumentException("agent with no registration");
            }

            for (Bus bus : buses) {
                int index = bus.ordinal();
                ParallelCommandDispatcher dispatcher = behaviour.dispatchers[index];
                if (dispatcher == null) {
                    throw new IllegalArgumentException(agentName + " not registered with " + bus);
                }
                dispatcher.addExecutant(forwarder);
            }
        }

    }

    @Override
    public void removeMessageListener(String agentName, BusMessageForwarder forwarder, Bus... buses) {
        if (forwarder == null) throw new IllegalArgumentException("null forwarder");
        //TODO: throw exception?
        //TODO: change that
        if (isClosed) return;
        // if agent name null or ""
        if ((agentName == null) || ("".equals(agentName))) {
            agentName = ANONYMOUS_AGENT;
        }
        // if buses is empty
        if (buses.length == 0) {
            buses = Bus.values();
        }
        synchronized (lock) {
            AgentBehaviour behaviour = agentMap.get(agentName);
            if (behaviour == null) {
                return;
            }

            for (Bus bus : buses) {
                int index = bus.ordinal();
                ParallelCommandDispatcher dispatcher = behaviour.dispatchers[index];
                if (dispatcher == null) {
                    return;
                }
                dispatcher.removeExecutant(forwarder);
            }
        }
    }

    @Override
    public void setMembershipListener(BusMembershipListener listener, Bus... buses) {
        if (buses == null || buses.length == 0) {
            buses = Bus.values();
        }
        synchronized (lock) {
            for (Bus bus : buses) {
                membershipListeners[bus.ordinal()] = listener;
            }
        }
    }
    
    @Override
    public BusMembershipListener getBusMembershipListener(Bus bus) {
        return membershipListeners[bus.ordinal()];
    }

    @Override
    public List<String> getConnectedNames(Bus bus) {
        int index = bus.ordinal();
        return busRegistry.registry[index];
    }
}
