package org.lsst.ccs.messaging.util;

import java.util.EnumMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;
import org.lsst.ccs.bus.definition.Bus;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.messaging.TransportStateException;
import org.lsst.ccs.utilities.scheduler.BasicThreadFactory;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * {@link Dispatcher} implementation that decouples message processing by CCS from message processing by JGroups.
 *
 * @author onoprien
 */
public class MultiQueueDispatcher extends AbstractDispatcher {

// -- Fields : -----------------------------------------------------------------
    
    private volatile boolean off = true;
    
    // Executors:
    
    private final EnumMap<Bus,Object> inExecutors;
    private final EnumMap<Bus,ExecutorService> outNormExecutors;
    private final ThreadPoolExecutor outOobCcExec;
    private final ThreadPoolExecutor oobExec;
    
    // Gates :
    
    private final AlertGate[] gatesDuration;
    private final AlertGate[] gatesQueueSize;
//    private final MessageGate[] gatesMessage;
    private final Throttle[] throttles;
    
    // Queue sizes :
    
    private final AtomicInteger[] queueSize; // keeps track of current que size
    
    
// -- Life cycle : -------------------------------------------------------------
    
    /**
     * Constructs an instance.
     * <p>
     * Recognized configuration parameters:
     * <dl>
     *   <dt>inThreads</dt>
     *   <dd>Number of processing threads for incoming messages: [LOG, COMMAND, STATUS]. Default: [1, 1, 8]</dd>
     *   <dt>duration/[in|out]/{BUS}/{STAGE}</dt>
     *   <dd>Logging and alert thresholds for message processing times (in milliseconds): [log, WARNING, ALARM]. Default: [] (no monitoring).</dd>
     *   <dt>queue/[in|out]/{BUS}</dt>
     *   <dd>Logging and alert thresholds for queue size: [log, WARNING, ALARM]. Default: [] (no monitoring).</dd>
     * </dl>
     * @param args Configuration string in {@code "key[=value]&...&key[=value]"} format, where value may be a comma-separated array.
     */
    public MultiQueueDispatcher(String args) {
        super(args);

        int[] inThreads = getIntArrayArg("inThreads", new int[] {1, 1, 8}); // LOG, COMMAND, STATUS
        
        // Gates :
        
        gatesDuration = new AlertGate[2 * Bus.values().length * Stage.values().length];
        int index = 0;
        String prefix = "duration/";
        for (int incoming=0; incoming < 2; incoming++) {
            String prefixIncoming = prefix + (incoming == 0 ? "in" : "out") +"/";
            for (Bus bus : Bus.values()) {
                String prefixIncomingBus = prefixIncoming + bus +"/";
                for (Stage stage : Stage.values()) {
                    String prefixIncomingBusStage = prefixIncomingBus + stage;
                    int[] duration = getIntArrayArg(prefixIncomingBusStage, new int[0]);
                    gatesDuration[index++] = new AlertGate(this, "dispatcher/"+ prefixIncomingBusStage, "High message processing time.", duration);
                }
            }
        }
        
        gatesQueueSize = new AlertGate[2 * Bus.values().length];
        index = 0;
        prefix = "queue/";
        for (int incoming=0; incoming < 2; incoming++) {
            String prefixIncoming = prefix + (incoming == 0 ? "in" : "out") +"/";
            for (Bus bus : Bus.values()) {
                String prefixIncomingBus = prefixIncoming + bus;
                int[] size = getIntArrayArg(prefixIncomingBus, new int[0]);
                gatesQueueSize[index++] = new AlertGate(this, "dispatcher/"+ prefixIncomingBus, "Long message queue.", size);
            }
        }
        
//        gatesMessage = new MessageGate[3 * Bus.values().length];
//        index = 0;
//        for (Bus bus : Bus.values()) {
//            prefix = "message/in/"+ bus;
//            index = bus.ordinal();
//            int[] n = getIntArrayArg(prefix +"/n", new int[0]);
//            int[] mbs = getIntArrayArg(prefix +"/mbs", new int[0]);
//            gatesMessage[index] = new MessageGate(this, "dispatcher/"+ prefix, n, mbs);
//            prefix = "message/outCCS/"+ bus;
//            index = bus.ordinal() + Bus.values().length;
//            n = getIntArrayArg(prefix +"/n", new int[0]);
//            mbs = getIntArrayArg(prefix +"/mbs", new int[0]);
//            gatesMessage[index] = new MessageGate(this, "dispatcher/"+ prefix, n, mbs);
//            prefix = "message/outJGroups/"+ bus;
//            index = bus.ordinal() + 2 * Bus.values().length;
//            n = getIntArrayArg(prefix +"/n", new int[0]);
//            mbs = getIntArrayArg(prefix +"/mbs", new int[0]);
//            gatesMessage[index] = new MessageGate(this, "dispatcher/"+ prefix, n, mbs);
//        }

        // Throttle:
        
        int size = getIntArg("throttle/size", 0);
        int rate = getIntArg("throttle/rate", 0);
        if (size > 0 && rate > 0) {
            throttles = new Throttle[Bus.values().length];
            for (Bus bus : Bus.values()) {
                throttles[bus.ordinal()] = new Throttle(this, "dispatcher/throttle/" + bus, size, rate);
            }
        } else {
            throttles = null;
        }
        
        // Executors:

        inExecutors = new EnumMap<>(Bus.class);
        for (Bus bus : Bus.values()) {
            int n = inThreads[bus.ordinal()];
            if (n == 1) {
                inExecutors.put(bus, Executors.newSingleThreadExecutor(new TFactory("MESSAGING_IN_"+ bus)));
            } else {
                inExecutors.put(bus, new KeyQueueExecutor("MESSAGING_IN_"+ bus, n));
            }
        }
        
        outNormExecutors = new EnumMap<>(Bus.class);
        for (Bus bus : Bus.values()) {
            outNormExecutors.put(bus, Executors.newSingleThreadExecutor(new TFactory("MESSAGING_OUT_"+ bus)));
        }
        
        outOobCcExec = new ThreadPoolExecutor(2, 2, 70L, TimeUnit.SECONDS, new LinkedBlockingQueue<>(), new TFactory("MESSAGING_OUT_OOB_CC"));
        outOobCcExec.allowCoreThreadTimeOut(true);
                
        oobExec = new ThreadPoolExecutor(0, Integer.MAX_VALUE, 70L, TimeUnit.SECONDS, new SynchronousQueue<>(), new TFactory("MESSAGING_OOB"));
        
        // Queue sizes
        
        int n = 2*Bus.values().length;
        queueSize = new AtomicInteger[n];
        for (int i=0; i<n; i++) {
            queueSize[i] = new AtomicInteger(0);
        }
        
        // Cleanup
        
        config = null;
    }

    /**
     * Initializes this dispatcher.
     * Empty implementation is provided by this class.
     */
    @Override
    public void initialize() {
        off = false;
    }

    /**
     * Orderly shuts down this dispatcher. Stops accepting tasks, executes previously
     * submitted tasks, delivers events to listeners, then shuts down.
     */
    @Override
    public void shutdown() {
        off = true;
        
        oobExec.shutdown();
        outOobCcExec.shutdown();
        outNormExecutors.values().forEach(exec -> exec.shutdown());
        inExecutors.values().forEach(exec -> {
            if (exec instanceof KeyQueueExecutor) {
                ((KeyQueueExecutor) exec).shutdown();
            } else {
                ((ExecutorService) exec).shutdown();
            }
        });
        
        try {
            oobExec.awaitTermination(1, TimeUnit.MINUTES);
            outOobCcExec.awaitTermination(1, TimeUnit.MINUTES);
            for (ExecutorService exec : outNormExecutors.values()) {
                exec.awaitTermination(1, TimeUnit.MINUTES);
            }
            for (Object exec : inExecutors.values()) {
                if (exec instanceof KeyQueueExecutor) {
                    ((KeyQueueExecutor) exec).awaitTermination(1, TimeUnit.MINUTES);
                } else {
                    ((ExecutorService) exec).awaitTermination(1, TimeUnit.MINUTES);
                }
            }
        } catch (InterruptedException x) {
            Thread.currentThread().interrupt();
        }
        
        super.shutdown();
    }

    
// -- Task submission : --------------------------------------------------------

    /**
     * Submits a task to process an incoming message or disconnection notification.
     * This method returns immediately, without waiting for the task to finish execution.
     * <p>
     * If one or more agent names are given, the task if guaranteed to be executed after any previously submitted
     * tasks for the same bus and agent. If no agents are specified, the task is independent of other tasks,
     * subject to capacity controls imposed by this service. If the {@code agents} argument
     * is {@code null}, the task is independent of other tasks, not subject to capacity controls.
     * 
     * @param run Task to be executed.
     * @param bus Bus (LOG, STATUS, or COMMAND).
     * @param agents Names of affected agents (source or disconnected).  
     * @throws TransportStateException If this Dispatcher is unable to accept the
     *         request for reasons other than being uninitialized or shut down.
     */
    @Override
    protected void in(DTask run, Bus bus, String... agents) {
        if (off) return;
//        if (gatesMessage != null) {
//            MessageGate gate = gatesMessage[bus.ordinal()];
//            if (gate != null) gate.check(run.getBusMessage());
//        }
        Order order = agents == null ? Order.OOB : (agents.length == 0 ? Order.OOB_CC : Order.NORM);
        TaskRunner runner = new TaskRunner(run, order);
        stageEnded(runner, Stage.START);
        try {
            switch (order) {
                case NORM:
                case OOB_CC:
                    Object exec = inExecutors.get(bus);
                    if (exec instanceof KeyQueueExecutor) {
                        ((KeyQueueExecutor)exec).execute(runner, agents);
                    } else {
                        ((ExecutorService)exec).execute(runner);
                    }
                    break;
                case OOB:
                    oobExec.execute(runner);
                    break;
            }
            int queueSizeIndex = 0 + bus.ordinal();
            int size = queueSize[queueSizeIndex].incrementAndGet();
            gatesQueueSize[queueSizeIndex].check(size);
        } catch (RejectedExecutionException x) {
            throw new TransportStateException(x);
        } finally {
            stageEnded(runner, Stage.SUBMIT);
        }
    }

    /**
     * Submits a task to process an outgoing message.
     * This method returns immediately, without waiting for the task to finish execution.
     * If the service has been shut down, calling this method does nothing.
     * <p>
     * Tasks submitted with {@code outOfBand} equal to {@code false} for the same bus are guaranteed to be
     * processed in the order of submission. If {@code outOfBand} equals {@code true}, the task is executed 
     * independently of others, subject to capacity controls imposed by this service.
     * If {@code outOfBand} equals {@code null}, the task is independent, not subject to capacity controls.
     * 
     * @param run Task to be executed.
     * @param bus Bus (LOG, STATUS, or COMMAND).
     * @param order Order of execution and capacity control policies.
     * @throws TransportStateException If this Dispatcher is unable to accept the
     *         request for reasons other than being uninitialized or shut down.
     */
    @Override
    protected void out(DTask run, Bus bus, Order order) {
        if (off) throw new TransportStateException();
//        if (gatesMessage != null) {
//            MessageGate gate = gatesMessage[Bus.values().length + bus.ordinal()];
//            if (gate != null) gate.check(run.getBusMessage());
//        }
        TaskRunner runner = new TaskRunner(run, order);
        stageEnded(runner, Stage.START);
        try {
            switch (order) {
                case NORM:
                    outNormExecutors.get(bus).execute(runner);
                    break;
                case OOB_CC:
                    outOobCcExec.execute(runner);
                    break;
                case OOB:
                    oobExec.execute(runner);
                    break;
            }
            int queueSizeIndex = 1 + bus.ordinal();
            int size = queueSize[queueSizeIndex].incrementAndGet();
            gatesQueueSize[queueSizeIndex].check(size);
        } catch (RejectedExecutionException x) {
            throw new TransportStateException();
        } finally {
            stageEnded(runner, Stage.SUBMIT);
        }
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    private void stageEnded(TaskRunner runner, Stage stage) {
        int duration = notifyTaskListeners(runner.getTask(), stage);
        int i = (runner.getTask().isOutgoing() ? 1 : 0) * (Bus.values().length * Stage.values().length) + runner.getTask().getBus().ordinal() * Stage.values().length + stage.ordinal();
        gatesDuration[i].check(duration);
    }


// -- Local classes : ----------------------------------------------------------
   
    private class TaskRunner implements Runnable {

        private final DTask task;
        private final Order order;

        TaskRunner(DTask runnable, Order order) {
            this.task = runnable;
            this.order = order;
        }

        @Override
        public void run() {
            boolean outgoing = task.isOutgoing();
            BusMessage busMessage = task.getBusMessage();
            Bus bus = task.getBus();
            int queueSizeIndex = (outgoing ? 1 : 0) + bus.ordinal();
            int size = queueSize[queueSizeIndex].decrementAndGet();
            gatesQueueSize[queueSizeIndex].check(size);
            stageEnded(this, Stage.WAIT);
            try {
                if (outgoing && throttles != null) {
                    throttles[bus.ordinal()].process(busMessage);
                }
                task.run();
            } catch (RuntimeException x) {
                String warningMessage = "Exception "+ (outgoing ? "sending" : "receiving")  +" message ";
                if ( busMessage != null ) {
                    warningMessage += busMessage.getClass().getSimpleName() +" ("+ busMessage.getClassName()+") ";
                    if (!outgoing) {
                        warningMessage += "from "+busMessage.getOriginAgentInfo().getName()+" ";
                    }
                } else {
                    warningMessage += "null ";
                }
                warningMessage += "on "+ bus +" bus, ("+ order +")";
                getLogger().log(Level.WARNING, warningMessage, x);
            } finally {
                stageEnded(this, Stage.RUN);
            }
        }
        
        DTask getTask() {
            return task;
        }

    }
    
    /** Customized thread factory for use by executors. */
    private class TFactory extends BasicThreadFactory {
        TFactory(String name) {
            super(name, null, true);
        }
        @Override
        public Thread newThread(Runnable r) {
            Thread thread = super.newThread(r);
            thread.setUncaughtExceptionHandler((t, x) -> getLogger().log(Level.WARNING, "Exception thrown from messaging executor: "+ thread.getName(), x));
            return thread;
        }
    }
        
}
