package org.lsst.ccs.messaging.util;

import java.util.*;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.Executors;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.definition.Bus;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.messaging.TransportStateException;
import org.lsst.ccs.utilities.scheduler.BasicThreadFactory;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * Base class to facilitate implementing {@link Dispatcher}.
 * <p>
 * Functionality provided by this class:
 * <ul>
 * <li>Handling of listeners. All events (including alerts) are delivered to listeners on a single dedicated thread.
 * <li>Registration and raising of alerts.
 * <li>Logging.
 * <li>Utility for parsing constructor parameters.
 * <li>Implementation of {@code Dispatcher.Task} that adds access to {@code BusMessage} and reference time, as well as 
 *     empty callback implementation.
 * </ul>
 *
 * @author onoprien
 */
abstract public class AbstractDispatcher implements Dispatcher {

// -- Fields : -----------------------------------------------------------------
    
    private final Logger LOGGER = Logger.getLogger(getClass().getName());
    
    /** Configuration obtained from constructor argument string. Used by {@code get...Arg(...) methods}. */
    protected HashMap<String,String> config;
    
    /** Executor for service tasks, including monitoring and listener notifications. */
    protected final ScheduledExecutorService serviceExec;
    
    /** Event listeners. */
    private final CopyOnWriteArrayList<Listener> listeners = new CopyOnWriteArrayList<>();
    /** Task listeners. */
    private final CopyOnWriteArrayList<TaskListener> taskListeners = new CopyOnWriteArrayList<>();
    /** Registered alerts. */
    protected final Map<String,Alert> alerts = Collections.synchronizedMap(new HashMap<>());

// -- Life cycle : -------------------------------------------------------------
    
    /**
     * Constructs {@code Dispatcher} instance in specified configuration.
     * 
     * @param args Configuration string in {@code key[=value]&...&key[=value]} format.
     */
    protected AbstractDispatcher(String args) {
        
        config = new HashMap<>();
        if (args != null && !args.isEmpty()) {
            for (String arg : args.split("\\&")) {
                String[] ss = arg.split("=");
                if (ss.length < 3) {
                    String key = ss[0].trim();
                    if (!key.isEmpty()) {
                        String value = ss.length == 2 ? ss[1].trim() : "";
                        config.put(key, value);
                    }
                }
            }
        }
        
        serviceExec = Executors.newScheduledThreadPool(1, new BasicThreadFactory("Dispatcher Service", null, true));
    }

    /**
     * Initializes this dispatcher.
     * Empty implementation is provided by this class.
     */
    @Override
    public void initialize() {
    }

    /**
     * Orderly shuts down this dispatcher. Stops accepting tasks, executes previously
     * submitted tasks, delivers events to listeners, then shuts down.
     * The implementation provided by this class drains and shuts down the executor for service tasks.
     */
    @Override
    public void shutdown() {
        try {
            serviceExec.shutdown();
            serviceExec.awaitTermination(10, TimeUnit.MINUTES);
        } catch (SecurityException | InterruptedException x) {
        }
    }
    

// -- Task submission : --------------------------------------------------------
    
    @Override
    public final Task in(BusMessage busMessage, long referenceTime, Runnable task, Bus bus, String... agents) {
        DTask t = new DTask(task, busMessage, bus, false, referenceTime);
        in(t, bus, agents);
        return t;
    }
    
    @Override
    public final Task out(BusMessage busMessage, long referenceTime, Runnable task, Bus bus, Order order) {
        DTask t = new DTask(task, busMessage, bus, true, referenceTime);
        out(t, bus, order);
        return t;
    }
    
    /**
     * To be implemented by concrete subclasses.
     * 
     * @param task Incoming task (message, deserialization error, or disconnection).
     * @param bus Bus.
     * @param agents Names of agents associated with this.
     */
    protected abstract void in(DTask task, Bus bus, String... agents);
    
    /**
     * To be implemented by concrete subclasses.
     * 
     * @param task Outgoing task (message).
     * @param bus Bus.
     * @param order Ordering flag.
     */
    protected abstract void out(DTask task, Bus bus, Order order);


// -- Events and listeners : ---------------------------------------------------
        
    /**
     * Registers a listener.
     * @param listener Listener to add.
     */
    @Override    
    public void addListener(Listener listener) {
        listeners.add(listener);
    }
    
    /**
     * Removes a listener.
     * @param listener Listener to remove.
     */
    @Override
    public void removeListener(Listener listener) {
        listeners.remove(listener);
    }

    /**
     * Notifies listeners of the given event.
     * This method is public to allow clients publish events through this {@code Dispatcher}.
     * @param event 
     */
    @Override
    public void fireEvent(Event event) {
        try {
            serviceExec.execute(() -> deliverEvent(event));
        } catch (RejectedExecutionException x) {
        }
    }
    
    protected void deliverEvent(Event event) {
        listeners.forEach(listener -> {
            try {
                listener.onEvent(event);
            } catch (RuntimeException x) {
                getLogger().log(Level.WARNING, "Error notifying " + listener + " of " + event.getClass().getSimpleName() + " event.", x);
            }
        });
    }

    @Override
    public void addTaskListener(TaskListener listener) {
        taskListeners.add(listener);
    }

    @Override
    public void removeTaskListener(TaskListener listener) {
        taskListeners.remove(listener);
    }

    @Override
    public void removeAllTaskListeners() {
        taskListeners.clear();
    }
    
    /**
     * Notifies task listeners.
     * 
     * @param task Task that completed the stage.
     * @param stage Stage that has been completed.
     * @return Time it took to complete the stage, in milliseconds.
     */
    protected int notifyTaskListeners(DTask task, Stage stage) {
        int duration = task.stageEnded(stage);
        taskListeners.forEach(listener -> {
            try {
                listener.stageEnded(task, stage);
            } catch (RuntimeException x) {
                LOGGER.log(Level.WARNING, "Error while calling dispatcher task listener.", x);
            }
        });
        return duration;
    }

    
// -- Alerts : -----------------------------------------------------------------
    
    /**
     * Notifies listeners by publishing an {@code AlertEvent}.
     * If an alert with matching {@code id} has not been registered, nothing is published.
     * 
     * @param id Alert id.
     * @param severity Alert severity.
     * @param cause Message.
     */
    @Override
    public void raiseAlert(String id, AlertState severity, String cause) {
        try {
            serviceExec.execute(() -> {
                Alert alert = alerts.get(id);
                if (alert == null) return;
                AlertEvent event = new AlertEvent(this, alert, severity, cause);
                deliverEvent(event);
            });
        } catch (RejectedExecutionException x) {
        }
    }
    
    /**
     * Notifies listeners by publishing an {@code AlertEvent}.
     * An event is published even if an alert with matching {@code id} has not been registered.
     * 
     * @param alert Alert instance.
     * @param severity Alert severity.
     * @param cause Message.
     */
    @Override
    public void raiseAlert(Alert alert, AlertState severity, String cause) {
        try {
            serviceExec.execute(() -> {
                AlertEvent event = new AlertEvent(this, alert, severity, cause);
                deliverEvent(event);
            });
        } catch (RejectedExecutionException x) {
        }
    }
    
    /**
     * Returns a set of alerts registered with this {@code Dispatcher}.
     * @return Registered alerts.
     */
    @Override
    public List<Alert> getRegisteredAlerts() {
        return new ArrayList<>(alerts.values());
    }
    
    /**
     * Registers an alert with this Dispatcher.
     * @param alert Alert to add.
     */
    @Override
    public void registerAlert(Alert alert) {
        alerts.put(alert.getAlertId(), alert);
    }


// -- Utility methods : --------------------------------------------------------
    
    /**
     * Returns a logger to be used for Dispatcher related messages.
     * @return Logger associated with this dispatcher.
     */
    @Override
    public Logger getLogger() {
        return LOGGER;
    }
    
    /**
     * Retrieves integer configuration parameter value from the constructor argument.
     * This method should only be called inside constructor.
     * 
     * @param key Parameter key. 
     * @param defaultValue Value to return if the argument strings do not provide value for the specified key.
     * @throws IllegalArgumentException If the specified key exists but the corresponding value is not integer.
     * @return Parameter value.
     */
    protected final int getIntArg(String key, int defaultValue) {
        try {
            String s = config.get(key);
            return s == null ? defaultValue : Integer.parseInt(s);
        } catch (NumberFormatException x) {
            throw new IllegalArgumentException("Illegal value for "+ key, x);
        }
    }
    
    /**
     * Retrieves boolean configuration parameter value from the constructor argument.
     * This method should only be called inside constructor.
     * 
     * @param key Parameter key. 
     * @return Parameter value.
     */
    protected final boolean getBooleanArg(String key) {
        return config.containsKey(key);
    }
    
    /**
     * Retrieves {@code int[]} configuration parameter value from the constructor argument.
     * If the configuration string passed to the constructor did not contain the specified key,
     * an array of the specified length with all elements equal to {@code defaultValue} is returned.
     * This method should only be called inside constructor.
     * 
     * @param key Parameter key. 
     * @param defaultValue Default value for all array elements.
     * @param length Array length.
     * @throws IllegalArgumentException If the given key exists but the corresponding value is not an integer array of the specified length.
     * @return Parameter value.
     */
    protected final int[] getIntArrayArg(String key, int defaultValue, int length) {
        try {
            int[] out = new int[length];
            String s = config.get(key);
            if (s == null) {
                Arrays.fill(out, defaultValue);
            } else {
                String[] ss = s.split(",");
                if(ss.length != length) {
                    throw new IllegalArgumentException("Illegal value for "+ key);
                }
                for (int i = 0; i < length; i++) {
                    out[i] = Integer.parseInt(ss[i].trim());
                }                
            }
            return out;
        } catch (NumberFormatException x) {
            throw new IllegalArgumentException("Illegal value for "+ key, x);
        }
    }
    
    /**
     * Retrieves {@code int[]} configuration parameter value from the constructor argument.
     * If the configuration string passed to the constructor did not contain the specified key, {@code defaultValue} is returned.
     * This method should only be called inside constructor.
     * 
     * @param key Parameter key. 
     * @param defaultValue Default value. If empty, no constraints on the length of the
     *                     array retrieved from the configuration string are enforced.
     * @throws IllegalArgumentException If the given key exists but the corresponding value is
     *                                  not an integer array of the same length as {@code defaultValue}.
     * @return Parameter value.
     */
    protected final int[] getIntArrayArg(String key, int[] defaultValue) {
        try {
            String s = config.get(key);
            if (s == null) {
                return defaultValue;
            } else {
                String[] ss = s.split(",");
                int length = ss.length;
                if(defaultValue.length != 0 && length != defaultValue.length) {
                    throw new IllegalArgumentException("Illegal value for "+ key);
                }
                int[] out = new int[length];
                for (int i = 0; i < length; i++) {
                    out[i] = Integer.parseInt(ss[i].trim());
                }                
                return out;
            }
        } catch (NumberFormatException x) {
            throw new IllegalArgumentException("Illegal value for "+ key, x);
        }
    }

    
// -- Instrumented task class : ------------------------------------------------
    
    /**
     * Handle of a task executed by {@code Dispatcher}.
     */
    static protected class DTask implements Dispatcher.Task, Runnable {
        
        protected final Runnable run;
        protected final BusMessage busMessage;
        protected final Bus bus;
        protected final boolean out;
        protected final long refTime;
        protected final EnumMap<Stage,Integer> times = new EnumMap<>(Stage.class);

        private DTask(Runnable run, BusMessage busMessage, Bus bus, boolean outgoing, long referenceTime) {
            this.run = run;
            this.busMessage = busMessage;
            this.bus = bus;
            this.out = outgoing;
            this.refTime = referenceTime > 0L ? referenceTime : System.currentTimeMillis();
        }

        @Override
        public BusMessage getBusMessage() {
            return busMessage;
        }

        @Override
        public Bus getBus() {
            return bus;
        }

        @Override
        public boolean isOutgoing() {
            return out;
        }

        @Override
        public long getRefTime() {
            return refTime;
        }

        @Override
        public int getDuration(Stage stage) {
            synchronized (times) {
                Integer time = times.get(stage);
                try {
                    switch (stage) {
                        case START:
                            return time;
                        case WAIT:
                            return time - times.get(Stage.START);
                        case RUN:
                            return time - times.get(Stage.WAIT);
                        case SUBMIT:
                            return time - times.get(Stage.START);
                    }
                } catch (NullPointerException x) {
                }
            }
            return -1;
        }

        @Override
        public int getDuration() {
            int max = 0;
            try {
                synchronized (times) {
                    for (Stage stage : Stage.values()) {
                        max = Math.max(max, times.get(stage));
                    }
                }
                return max;
            } catch (NullPointerException x) {
                return -1;
            }
        }

        @Override
        public Runnable getPayload() {
            return run;
        }

        @Override
        public void run() {
            run.run();
        }
        
        private int stageEnded(Stage stage) {
            int time = (int) (System.currentTimeMillis() - refTime);
            synchronized (times) {
                times.put(stage, time);
            }
            switch (stage) {
                case START:
                    if (busMessage != null) {
                        if (out) {
                            busMessage.setOutgoingQueueInTimeStamp(CCSTimeStamp.currentTime());
                        } else {
                            busMessage.setIncomingQueueInTimeStamp(CCSTimeStamp.currentTime());
                        }
                    }
                    break;
                case WAIT:
                    if (busMessage != null) {
                        if (out) {
                            busMessage.setOutgoingQueueOutTimeStamp(CCSTimeStamp.currentTime());
                        } else {
                            busMessage.setIncomingQueueOutTimeStamp(CCSTimeStamp.currentTime());
                        }
                    }
                    break;
            }
            return getDuration(stage);
        }

    }

}
