package org.lsst.ccs.messaging.util;

import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.logging.Level;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.states.AlertState;

/**
 * Raises an alert or logs a message when the rate of message processing crosses
 * thresholds on size or number, taking care not to do so too often for the same
 * reason. This class is thread-safe.
 *
 * @author onoprien
 */
public class MessageGate {

// -- Fields : -----------------------------------------------------------------
    
    private final Dispatcher dispatcher;
    private final Alert alert;
    
    private final int vetoTime;
    private final double vetoFactor;
    private final int[] thresholdsN, thresholdsMB;
    
    private int lastValueN, lastValueMB;
    private long lastTimeN, lastTimeMB;
    private volatile int lastLevel = -1;
    
    private final AtomicInteger nAccumulator = new AtomicInteger();
    private final AtomicLong sizeAccumulator = new AtomicLong();
    private long periodStartTime;

// -- Life cycle : -------------------------------------------------------------
    
    /**
     * Constructs an instance, registering new {@code Alert} with the {@code Dispatcher}.
     * 
     * @param dispatcher Dispatcher through which alerts should be raised and messages logged.
     * @param alertID ID for {@code Alert} raised through this gate.
     * @param vetoTime Veto time in milliseconds.
     * @param vetoFactor Veto factor.
     */
    public MessageGate(Dispatcher dispatcher, String alertID, int[] thresholdsN, int[] thresholdsMB, int vetoTime, double vetoFactor) {
        
        this.thresholdsN = thresholdsN == null || thresholdsN.length == 0 ? null : thresholdsN;
        this.thresholdsMB = thresholdsMB == null || thresholdsMB.length == 0 ? null : thresholdsMB;
        if (this.thresholdsN == null && this.thresholdsMB == null) {
            alert = null;
        } else {
            alert = new Alert(alertID, "Message rate alert.");
            dispatcher.registerAlert(alert);
        }
        this.dispatcher = dispatcher;
        this.vetoTime = vetoTime;
        this.vetoFactor = vetoFactor;
    }
    
    /**
     * Creates an instance with {@code vetoTime = 5000} and {@code vetoFactor = 1.5}.
     * The new {@code Alert} is registered with the {@code Dispatcher}.
     * 
     * @param dispatcher Dispatcher through which alerts should be raised and messages logged.
     * @param alertID ID for {@code Alert} raised through this gate.
     */
    public MessageGate(Dispatcher dispatcher, String alertID, int[] thresholdsN, int[] thresholdsMB) {
        this(dispatcher, alertID, thresholdsN, thresholdsMB, 5000, 1.5);
    }
    
// -----------------------------------------------------------------------------
    
    /**
     * Processes the message, taking an appropriate action if rate exceeds threshold.
     * @param message Message. If {@code null}, do nothing.
     */
//    public void check(BusMessage message) {
//        long size = thresholdsMB == null ? 0 : getSize(message);
//        int n = nAccumulator.incrementAndGet();
//        size = sizeAccumulator.addAndGet(size);
//        long delay0 = check(n, thresholdsN);
//        long delay1 = check((int)(size/1000000), thresholdsMB);
//    }
    
    /**
     * Checks the value and takes appropriate action.
     * 
     * @param value Current value of the watched quantity.
     */
//    public long check(int value, int[] thresholds) {
//        if (thresholds == null) return;
//        int level = -1;
//        for (int i = 0; i<thresholds.length; i++) {
//            if (value > thresholds[i]) {
//                level = i;
//            } else {
//                break;
//            }
//        }
//        if (level == -1 && lastLevel == -1) { // just another normal value
//            return;
//        }
//        synchronized (this) {
//            if (level == -1) { // back to normal
//                lastValue = Integer.MIN_VALUE;
//                lastTime = 0;
//                if (lastLevel > 0) {
//                    dispatcher.raiseAlert(alert, AlertState.NOMINAL, getMessage(value));
//                }
//                lastLevel = -1;
//            } else {
//                long currentTime = System.currentTimeMillis();
//                if (level != lastLevel || (currentTime - lastTime) > vetoTime || value > (lastValue * vetoFactor)) {
//                    lastValue = value;
//                    lastTime = currentTime;
//                    String message = getMessage(value);
//                    Level logLevel = level == 0 ? Level.INFO : (level == 1 ? Level.WARNING : Level.SEVERE);
//                    dispatcher.getLogger().log(logLevel, message);
//                    if (level != 0 || lastLevel != -1) {
//                        dispatcher.raiseAlert(alert, AlertState.values()[level], getMessage(value));
//                    }
//                    lastLevel = level;
//                }
//            }
//        }
//    }
    
    protected String getMessage(int value) {
        return "High value of "+ alert.getAlertId() +" : "+ value;
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
//    static public long getSize(BusMessage message) {
//        
//    }

}
