package org.lsst.ccs.messaging;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;

import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.AgentLock;
import org.lsst.ccs.bus.data.AgentLockInfo;
import org.lsst.ccs.bus.definition.Bus;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.CommandReply;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.messages.LogMessage;
import org.lsst.ccs.bus.messages.MessageFlag;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.utilities.scheduler.Scheduler;

/**
 * This interface provides messaging methods for a component (eg an agent) to be
 * able to communicate on the buses. Provided methods allow to add/remove
 * message listeners on each of the buses, to send message on each of the buses,
 * to specify a unique CommandExecutor associated to the agent and a list of
 * CommandOriginator.
 * 
 * @author LSST CCS Team
 */
public class AgentMessagingLayer implements AgentMessagingLayerMBean {
    
	private static final List<AgentMessagingLayer> msgAccesses = new CopyOnWriteArrayList<>();

	private Predicate<BusMessage<? extends Serializable, ?>> filterMessagesFromThisAgent;

	private final Logger curLogger = Logger.getLogger(AgentMessagingLayer.class.getName());

	private final BusApplicationLayer layer;

        public static enum ConnectionStatus {
            NOT_CONNECTED,
            CONNECTED,
            DISCONNECTED;
        }
        
        private volatile ConnectionStatus connectionStatus = ConnectionStatus.NOT_CONNECTED;
        private final Object connectionLock = new Object();
        
        private final AgentInfo agentInfo;
        private final LockLevelService lockLevelService;
        
        private final Scheduler scheduler;
        private boolean accumulateMsgStatistics = false;
        private final Object accumulationLock = new Object();
        
        private volatile MessageAccumulator msgAccumulator = null;
        
        private final Map<Bus,List<OutgoingMessageListener>> outgoingMessageListenersMap = new ConcurrentHashMap<>();
        
        /**
	 * Build an Instance of an AgentMessagingLayer for a given Agent by
	 * providing the AgentInfo object.
	 *
         * @param agentInfo
         * @param lockLevelService
	 * @return The corresponding AgentMessagingLayer.
	 */
	public static AgentMessagingLayer createInstance(AgentInfo agentInfo, LockLevelService lockLevelService) {
		AgentMessagingLayer agentMessagingLayer = new AgentMessagingLayer(agentInfo, lockLevelService);
		return agentMessagingLayer;
	}

	AgentMessagingLayer(AgentInfo agentInfo, LockLevelService lockLevelService) {
            this.layer = new BusApplicationLayer(agentInfo, this);
            filterMessagesFromThisAgent = BusMessageFilterFactory.messageOrigin(agentInfo.getName()).negate();            
            this.agentInfo = agentInfo;
            this.lockLevelService = lockLevelService;

            scheduler = new Scheduler("AgentMessagingLayer scheduler", 1);
            scheduler.setLogger(curLogger);
            
            accumulateMsgStatistics = BootstrapResourceUtils.getBootstrapSystemProperties().getProperty("org.lsst.ccs.messaging.accumulate.msg.statistics", "false").toLowerCase().equals("true");
            if ( accumulateMsgStatistics ) {
                msgAccumulator = new MessageAccumulator();
            }
	}
        
        /** For use in tests only. */
        AgentMessagingLayer(AgentInfo agentInfo) {
            layer = null;
            this.agentInfo = agentInfo;
            scheduler = new Scheduler("AgentMessagingLayer scheduler", 1);
            lockLevelService = null;
        }
                
        /**
         * This method is meant to be invoked from JMX
         * @param seconds The number of seconds to wait before restarting the messaging layer.
         * 
         */
        @Override
        public void restart(int seconds) {
            shutdownBusAccess();
            Thread restart = new Thread(
                () -> {
                    try {
                        Thread.sleep(1000*seconds);
                    } catch (InterruptedException ie) {
                        throw new RuntimeException(ie);
                    }
                    connectToBuses();                    
                }
            );
        }
        
        /**
         * Add a BusMessagePreProcessor the the AgentMessagingLayer.
         *
         * @param preProcessor The BusMessagePreProcessor to be added.
         */
        public final void addBusMessagePreProcessor(BusMessagePreProcessor preProcessor) {
            layer.addBusMessagePreProcessor(preProcessor);
        }

        public BusApplicationLayer getApplicationLayer() {
            return layer;
	}
        
        public MessagingLayer getMessagingLayer() {
            return layer.getBusMessagingLayer();
        }
        
        public AgentInfo getAgentInfo() {
            return agentInfo;
        }

	public void shutdownBusAccess() {
            layer.close();
            msgAccesses.remove(this);
            synchronized(connectionLock) {
                connectionStatus = ConnectionStatus.DISCONNECTED;
            }
	}

        public void connectToBuses() {
            if ( accumulateMsgStatistics ) {
                scheduler.scheduleAtFixedRate( () -> printAndResetAccumulation(), 1, 1, TimeUnit.MINUTES);
            }
            layer.connectToBuses();
            msgAccesses.add(this);
            synchronized(connectionLock) {
                connectionStatus = ConnectionStatus.CONNECTED;
                connectionLock.notify();
            }
        }
        
        private void printAndResetAccumulation() {
            MessageAccumulator oldMessageAccumulator = msgAccumulator;
            msgAccumulator = new MessageAccumulator();            
            curLogger.log(Level.FINE, oldMessageAccumulator.toString());            
        }
        
	public AgentPresenceManager getAgentPresenceManager() {
            return layer.getAgentPresenceManager();
	}
        
        /**
         * Does this messaging layer need a Heartbeat for failure detection?
         *
         * @return true if this feature is not provided and the Toolkit needs to
         * provide a heartbeat.
         */
        public boolean needsHeartbeat() {
            return ! layer.hasInternalHeartbeat();
        }
        
        /**
         * 
         * @return The public API
         */
        public LockLevelService getAgentLockService() {
            return lockLevelService;
        }

	/**
	 * Adds a listener on the Log bus. By default the listener is negate passed
	 * messages published from this AgentMessagingLayer.
	 * 
	 * @param listener
	 *            the listener to be added on the Log bus
	 */
	public void addLogMessageListener(LogMessageListener listener) {
            addLogMessageListener(listener, filterMessagesFromThisAgent);
	}

	/**
	 * Adds a listener on the Log bus with a filter. The listener is passed bus
	 * messages that pass the filter. If the filter is null, all messages are
	 * passed to the filter.
	 * 
	 * @param listener
	 *            the listener to be added on the Log bus
	 * @param filter
	 *            The BusMessageFilter to be applied to the incoming Bus
	 *            Messages
	 */
	public void addLogMessageListener(LogMessageListener listener,
			Predicate<BusMessage<? extends Serializable, ?>> filter) {
		layer.addLogListener(listener, filter);
	}

	/**
	 * Adds a listener on the Status bus. By default the listener is negate
	 * passed messages published from this AgentMessagingLayer.
	 * 
	 * @param listener
	 *            the listener to be added on the Status bus
	 */
	public void addStatusMessageListener(StatusMessageListener listener) {
            addStatusMessageListener(listener, filterMessagesFromThisAgent);
	}

	/**
	 * Adds a listener on the Status bus with a filter. The listener is passed
	 * bus messages that pass the filter. If the filter is null, all messages
	 * are passed to the filter.
	 * 
	 * @param listener
	 *            the listener to be added on the Status bus
	 * @param filter
	 *            The BusMessageFilter to be applied to the incoming Bus
	 *            Messages
	 */
	public void addStatusMessageListener(StatusMessageListener listener,
			Predicate<BusMessage<? extends Serializable, ?>> filter) {
		layer.addStatusListener(listener, filter);
	}

	/**
	 * Adds a listener on the Command bus. By default the listener is negate
	 * passed messages published from this AgentMessagingLayer.
	 * 
	 * @param listener
	 *            the listener to be added on the Command bus
	 */
	public void addCommandMessageListener(CommandMessageListener listener) {
            addCommandMessageListener(listener, filterMessagesFromThisAgent);
	}

	/**
	 * Adds a listener on the Command bus with a filter. The listener is passed
	 * bus messages that pass the filter. If the filter is null, all messages
	 * are passed to the filter.
	 * 
	 * @param listener
	 *            the listener to be added on the Command bus
	 * @param filter
	 *            The BusMessageFilter to be applied to the incoming Bus
	 *            Messages
	 */
	public void addCommandMessageListener(CommandMessageListener listener,
			Predicate<BusMessage<? extends Serializable, ?>> filter) {
            layer.addCommandListener(listener, filter);
	}

	/**
	 * Removes a listener on the Log bus
	 * 
	 * @param listener
	 *            the listener to be removed on the Log bus
	 */
	public void removeLogMessageListener(LogMessageListener listener) {
            layer.removeLogListener(listener);
	}

	/**
	 * Removes a listener on the Status bus
	 * 
	 * @param listener
	 *            the listener to be removed on the Status bus
	 */
	public void removeStatusMessageListener(StatusMessageListener listener) {
            layer.removeStatusListener(listener);
	}

	/**
	 * Removes a listener on the Command bus
	 * 
	 * @param listener
	 *            the listener to be removed on the Command bus
	 */
	public void removeCommandMessageListener(CommandMessageListener listener) {
            layer.removeCommandListener(listener);
	}

	/**
	 * Sends a Log Message on the Log Bus
	 * 
	 * @param msg The message to be sent on the Log bus
	 */
        public void sendLogMessage(LogMessage msg) {
            checkMessageLayerConnection();
            msg.setOriginAgentInfo(agentInfo);
            layer.sendLog(msg, MessageFlag.NO_RELIABILITY);
            if ( msgAccumulator != null ) {
                msgAccumulator.accumulateMessage(Bus.LOG, msg);
            }
            for ( OutgoingMessageListener l : getOutgoingMessageListenersForBus(Bus.LOG) ) {
                l.outgoingMessage(msg);
            }
	}

	/**
	 * Sends a Status Message on the Status Bus
	 * 
	 * @param msg The message to be sent on the Status bus
	 */
	public void sendStatusMessage(StatusMessage msg) {
            sendStatusMessage(msg, new MessageFlag[0]);
        }
	public void sendStatusMessage(StatusMessage msg, MessageFlag... flags) {
            checkMessageLayerConnection();
            msg.setOriginAgentInfo(agentInfo);
            curLogger.finest("sending status " + msg);
            layer.sendStatus(msg, flags);
            if ( msgAccumulator != null ) {
                msgAccumulator.accumulateMessage(Bus.STATUS, msg);
            }
            for ( OutgoingMessageListener l : getOutgoingMessageListenersForBus(Bus.STATUS) ) {
                l.outgoingMessage(msg);
            }
	}

	/**
	 * Sends a Command Request on the Command Bus. This message will be received
	 * by the CommandExecutor and the list of CommandMessageListener
	 * 
	 * @param cmd
	 *            The CommandRequest to be sent on the Command bus
	 * @param originator
	 *            The component that has requested the execution of the command
         * @throws DestinationsException if the command target is not present on 
         * the buses.
	 */
	public void sendCommandRequest(CommandRequest cmd, CommandOriginator originator) {
            sendCommandRequest(cmd, originator,true);
        }
	public void sendCommandRequest(CommandRequest cmd, CommandOriginator originator, boolean checkDestinationExists) {
            checkMessageLayerConnection();
            if ( lockLevelService != null ) {
                // Looking for a potential held lock
                String destination = BusMessagingLayer.parseDestination(cmd.getDestination());
                AgentLock lock = lockLevelService.getLockForAgent(destination);
                int desiredLevel = lockLevelService.getLevelForAgent(destination);
                if (lock == null && cmd.getBasicCommand().getOptions().hasOption("withLock")) {
                    lock = new AgentLockInfo(destination, lockLevelService.getUserId(), 0, AgentLockInfo.Status.INFO, "withLock", agentInfo); // dummy lock to suply userid for processing by target agent
                }
                cmd.setLockAndLevel(lock, desiredLevel);
            }
            cmd.setOriginAgentInfo(agentInfo);

            if ( checkDestinationExists ) {
                String destination = BusMessagingLayer.parseDestination(cmd.getDestination());
                if (!getAgentPresenceManager().agentExists(destination)) {
                    DestinationsException exc = new DestinationsException(agentInfo.getName(), destination);
                    curLogger.log(Level.FINE, "sending fail (closed){0} to destination {1}", new Object[]{agentInfo.getName(), destination});
                    curLogger.log(Level.WARNING, "destination problem", exc);
                    // toDO : wit until lock arbitrator
                    // The exception is thrown again to address
                    // https://jira.slac.stanford.edu/browse/LSSTCCS-181
                    // In the future we might want to have smarter logic to address
                    // any LockArbitrator issues.
                    throw exc;
                }
            }

            layer.sendCommand(cmd, originator);
            if ( msgAccumulator != null ) {
                msgAccumulator.accumulateMessage(Bus.COMMAND, cmd);
            }
            for ( OutgoingMessageListener l : getOutgoingMessageListenersForBus(Bus.COMMAND) ) {
                l.outgoingMessage(cmd);
            }
	}

	/**
	 * Sends a Command Reply on the Command Bus. This message will be received
	 * by the CommandOriginator and the list of CommandMessageListener The reply
	 * can be a Nack, Ack, Error or Result
	 * 
	 * @param reply
	 *            The CommandReply to be sent on the Command bus
	 */
	public void sendCommandReply(CommandReply reply) {
            checkMessageLayerConnection();
            reply.setOriginAgentInfo(agentInfo);
            layer.reply(reply);
            if ( msgAccumulator != null ) {
                msgAccumulator.accumulateMessage(Bus.COMMAND, reply);
            }
            for ( OutgoingMessageListener l : getOutgoingMessageListenersForBus(Bus.COMMAND) ) {
                l.outgoingMessage(reply);
            }
	}

        private void checkMessageLayerConnection() {
            switch (connectionStatus) {
                case NOT_CONNECTED:
                    throw new RuntimeException("The CCS Buses have not been connected yet.");
                 case DISCONNECTED:
                    throw new RuntimeException("The CCS Buses have been shutdown. It's no longer possible to send messages.");                                   
            }
        }
        
        public void waitForMessageLayerConnection() {
            synchronized(connectionLock) {
                switch(connectionStatus) {
                    case CONNECTED:
                        return;
                    case DISCONNECTED:
                        throw new RuntimeException("The connection to the messaging layer has been lost");
                    default:
                        try {
                            connectionLock.wait(1000);
                        } catch (InterruptedException ie) {
                            throw new RuntimeException("Failed to wait for bus connection", ie);
                        }
                }                
            }
        }
        
	/**
	 * Defines the component able to execute an incoming command
	 * 
	 * @param executor
	 */
	public void setCommandExecutor(CommandExecutor executor) {
		layer.setCommandExecutor(executor);
	}
        
        public void setClusterDeserializationErrorHandler(ClusterDeserializationErrorHandler h) {
            layer.setClusterDeserializationErrorHandler(h);
        }

        public void addOutgoingMessageListenerForBus(OutgoingMessageListener listener, Bus... bus) {
            for ( Bus b : bus) {
                getOutgoingMessageListenersForBus(b).add(listener);
            }
        }
        
        public void removeOutgoingMessageListenerForBus(OutgoingMessageListener listener) {
            for ( Bus b : Bus.values() ) {
                getOutgoingMessageListenersForBus(b).remove(listener);
            }
        }
        
        private List<OutgoingMessageListener> getOutgoingMessageListenersForBus(Bus bus) {
            return outgoingMessageListenersMap.computeIfAbsent(bus, (b) -> new CopyOnWriteArrayList<>());
        }
        
        
	static List<AgentMessagingLayer> getMessagingAccesses() {
		return msgAccesses;
	}

	static void printMessagingAccessInfo(AgentMessagingLayer msgAccess) {
		System.out.println("MessagingAccess " + msgAccess.agentInfo.getName());
		BusApplicationLayer layer = msgAccess.getApplicationLayer();
		System.out.println("BusApplicationLayer " + layer);
		BusMessagingLayer messagingLayer = layer.getBusMessagingLayer();
		System.out.println("BusMessagingLayer " + messagingLayer);
		Set<String> localAgents = messagingLayer.getRegisteredLocalAgents();
		System.out.println("Local Agents " + localAgents.size());
		for (String agent : localAgents) {
			System.out.println("\t" + agent);
		}
	}
        
        private class MessageAccumulator {
            
            private final Map<Bus,BusMessageAccumulator> accums = new ConcurrentHashMap<>();
            
            MessageAccumulator() {
                for ( Bus b : Bus.values() ) {
                    accums.put(b, new BusMessageAccumulator(b));
                }                
            }
            
            void accumulateMessage(Bus bus, BusMessage msg) {
                accums.get(bus).accumulateMessage(msg);
            }
            
            @Override
            public String toString() {
                StringBuilder sb = new StringBuilder();
                sb.append("\n");                        
                for ( Bus bus : Bus.values() ) {
                    sb.append(accums.get(bus).toString());
                }
                return sb.toString();
            }
        }
        
        private class BusMessageAccumulator {

            private final Bus bus;
            
            private volatile int totalCount = 0;
            private volatile double totalSize = 0;
            private final Map<Class, BusMessageAccumulator> classMap = new ConcurrentHashMap<>();
           
            
            BusMessageAccumulator(Bus bus) {
                this.bus = bus;
            }
            
            void accumulateMessage(BusMessage msg) {
                increaseCount();
                double size = sizeOf(msg);
                increaseSize(size);
                
                Class msgClass = msg.getClass();
                BusMessageAccumulator msgAccum = classMap.computeIfAbsent(msgClass, (c) -> { return new BusMessageAccumulator(bus);});
                msgAccum.increaseCount();
                msgAccum.increaseSize(size);
            }
            
            private void increaseCount() {
                totalCount++;
            }
            
            private void increaseSize(double size) {
                totalSize += size;
            }
            
            @Override            
            public String toString() {
                StringBuilder sb = new StringBuilder(bus+" messages in last minute: ").append(totalCount).append(" for total size: ").append(String.format("%.2f",totalSize)).append(" Kb \n");
                for ( Class c: classMap.keySet() ) {                    
                    BusMessageAccumulator bma = classMap.get(c);
                    sb.append("\t").append("Class: ").append(c.getSimpleName()).append(": ").append(bma.totalCount).append(" for total size: ").append(String.format("%.2f",bma.totalSize)).append(" Kb \n");
                }
                return sb.toString();                        
            }
            
        }
        
    static byte[] serialize(Serializable object) {
        ByteArrayOutputStream byteOutputStream = new ByteArrayOutputStream();
        try (ObjectOutputStream objectOutputStream = new ObjectOutputStream(byteOutputStream)) {
            objectOutputStream.writeObject(object);
            objectOutputStream.flush();
            return byteOutputStream.toByteArray();
        } catch (IOException ieo) {
            return new byte[0];
        }
    }
    
    
    static double sizeOf(Serializable object) {
        return (double)serialize(object).length / 1000.;
    }
       

}
