package org.lsst.ccs.messaging;

import java.io.IOException;
import org.lsst.ccs.bus.definition.Bus;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.MessageFlag;

/**
 * Interface providing methods to connect/disconnect MessagingAccessLayer objects to/from the CCS buses and to
 * send messages over the buses.
 * An instance of a MessagingLayer is created via the MessagingManager.
 * When connecting a MessagingAccessLayer, its name is used to uniquely identify it on the buses.
 * If such name is not unique, a DuplicateAgentNameException is thrown.
 * 
 * @author emarin
 */
public interface MessagingLayer {
    
    /**
     * Sends a message on a bus for a given MessagingAccessLayer.
     * The name of the MessagingAccessLayer determines the origin of the bus message.
     * note : that BusMessage should have sender and
     * destination information but it is not the role of the communication layer
     * to parse destination information such as "subsystem1, subsystem2" or
     * "subsystem3/module"a.
     * @param <T> the type of message sent on the bus
     * @param senderAgent provides the name as the origin of the message 
     * @param bus the bus on which to send the message
     * @param msg the message
     * @param flags MessageFlag array associated to this message.
     * @throws RuntimeException if the inner implementation fails sending the message
     */
    <T extends BusMessage> void sendMessage (String senderAgent, Bus bus, T msg, MessageFlag... flags);
    
    /**
     * Connects a MessagingAccessLayer object on the buses by using its name as a 
     * unique identifier.
     * Upon creation, if needed, all needed buses are created.
     * Depending on the underlying implementation, the MessagingLayer allows this method
     * to be called more than once, or throws a RuntimeException.
     * TBD : should all the buses be created or just the ones specified by the MessagingAccessLayer ?
     * @param accessLayer The layer to be connected on the buses
     * @throws DuplicateAgentNameException if the name of MessagingAccessLayer is not unique on the buses
     * @throws java.io.IOException
     */
    void connect(MessagingAccessLayer accessLayer) throws DuplicateAgentNameException, IOException;
    
    /**
     * Remove the entry point to the buses for the given MessagingAccessLayer.
     * At this point, the CCS buses might be closed.
     * @param accessLayer The layer to be connected on the buses
     */
    void disconnect(MessagingAccessLayer accessLayer);
}
