package org.lsst.ccs.messaging.util;

import java.util.*;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.Executors;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.utilities.scheduler.BasicThreadFactory;

/**
 * Base class to facilitate implementing {@link Dispatcher}.
 * <p>
 * Functionality provided by this class:
 * <ul>
 * <li>Handling of listeners. All events (including alerts) are delivered to listeners on a single dedicated thread.
 * <li>Registration and raising of alerts.
 * <li>Logging.
 * <li>Utility for parsing constructor parameters.
 * <li>Implementation of {@code Dispatcher.Task} that adds access to {@code BusMessage} and reference time, as well as 
 *     empty callback implementation.
 * </ul>
 *
 * @author onoprien
 */
abstract public class AbstractDispatcher implements Dispatcher {

// -- Fields : -----------------------------------------------------------------
    
    private final Logger LOGGER = Logger.getLogger(getClass().getName());
    
    /** Configuration obtained from constructor argument string. Used by {@code get...Arg(...) methods}. */
    protected HashMap<String,String> config;
    
    /** Executor for service tasks, including monitoring and listener notifications. */
    protected final ScheduledExecutorService serviceExec;
    
    /** Event listeners. */
    private final CopyOnWriteArrayList<Listener> listeners = new CopyOnWriteArrayList<>();
    /** Registered alerts. */
    protected final Map<String,Alert> alerts = Collections.synchronizedMap(new HashMap<>());

// -- Life cycle : -------------------------------------------------------------
    
    /**
     * Constructs {@code Dispatcher} instance in specified configuration.
     * 
     * @param args Configuration string in {@code key[=value]&...&key[=value]} format.
     */
    protected AbstractDispatcher(String args) {
        
        config = new HashMap<>();
        if (args != null && !args.isEmpty()) {
            for (String arg : args.split("\\&")) {
                String[] ss = arg.split("=");
                if (ss.length < 3) {
                    String key = ss[0].trim();
                    if (!key.isEmpty()) {
                        String value = ss.length == 2 ? ss[1].trim() : "";
                        config.put(key, value);
                    }
                }
            }
        }
        
        serviceExec = Executors.newScheduledThreadPool(1, new BasicThreadFactory("Dispatcher Service", null, true));
    }

    /**
     * Initializes this dispatcher.
     * Empty implementation is provided by this class.
     */
    @Override
    public void initialize() {
    }

    /**
     * Orderly shuts down this dispatcher. Stops accepting tasks, executes previously
     * submitted tasks, delivers events to listeners, then shuts down.
     * The implementation provided by this class drains and shuts down the executor for service tasks.
     */
    @Override
    public void shutdown() {
        try {
            serviceExec.shutdown();
            serviceExec.awaitTermination(10, TimeUnit.MINUTES);
        } catch (SecurityException | InterruptedException x) {
        }
    }
    

// -- Events and listeners : ---------------------------------------------------
        
    /**
     * Registers a listener.
     * @param listener Listener to add.
     */
    @Override    
    public void addListener(Listener listener) {
        listeners.add(listener);
    }
    
    /**
     * Removes a listener.
     * @param listener Listener to remove.
     */
    @Override
    public void removeListener(Listener listener) {
        listeners.remove(listener);
    }

    /**
     * Notifies listeners of the given event.
     * This method is public to allow clients publish events through this {@code Dispatcher}.
     * @param event 
     */
    @Override
    public void fireEvent(Event event) {
        try {
            serviceExec.execute(() -> deliverEvent(event));
        } catch (RejectedExecutionException x) {
        }
    }
    
    protected void deliverEvent(Event event) {
        listeners.forEach(listener -> {
            try {
                listener.onEvent(event);
            } catch (RuntimeException x) {
                getLogger().log(Level.WARNING, "Error notifying " + listener + " of " + event.getClass().getSimpleName() + " event.", x);
            }
        });
    }

    
// -- Alerts : -----------------------------------------------------------------
    
    /**
     * Notifies listeners by publishing an {@code AlertEvent}.
     * If an alert with matching {@code id} has not been registered, nothing is published.
     * 
     * @param id Alert id.
     * @param severity Alert severity.
     * @param cause Message.
     */
    @Override
    public void raiseAlert(String id, AlertState severity, String cause) {
        try {
            serviceExec.execute(() -> {
                Alert alert = alerts.get(id);
                if (alert == null) return;
                AlertEvent event = new AlertEvent(this, alert, severity, cause);
                deliverEvent(event);
            });
        } catch (RejectedExecutionException x) {
        }
    }
    
    /**
     * Notifies listeners by publishing an {@code AlertEvent}.
     * An event is published even if an alert with matching {@code id} has not been registered.
     * 
     * @param alert Alert instance.
     * @param severity Alert severity.
     * @param cause Message.
     */
    @Override
    public void raiseAlert(Alert alert, AlertState severity, String cause) {
        try {
            serviceExec.execute(() -> {
                AlertEvent event = new AlertEvent(this, alert, severity, cause);
                deliverEvent(event);
            });
        } catch (RejectedExecutionException x) {
        }
    }
    
    /**
     * Returns a set of alerts registered with this {@code Dispatcher}.
     * @return Registered alerts.
     */
    @Override
    public List<Alert> getRegisteredAlerts() {
        return new ArrayList<>(alerts.values());
    }
    
    /**
     * Registers an alert with this Dispatcher.
     * @param alert Alert to add.
     */
    @Override
    public void registerAlert(Alert alert) {
        alerts.put(alert.getAlertId(), alert);
    }


// -- Utility methods : --------------------------------------------------------
    
    /**
     * Returns a logger to be used for Dispatcher related messages.
     * @return Logger associated with this dispatcher.
     */
    @Override
    public Logger getLogger() {
        return LOGGER;
    }
    
    /**
     * Retrieves integer configuration parameter value from the constructor argument.
     * This method should only be called inside constructor.
     * 
     * @param key Parameter key. 
     * @param defaultValue Value to return if the argument strings do not provide value for the specified key.
     * @throws IllegalArgumentException If the specified key exists but the corresponding value is not integer.
     * @return Parameter value.
     */
    protected final int getIntArg(String key, int defaultValue) {
        try {
            String s = config.get(key);
            return s == null ? defaultValue : Integer.parseInt(s);
        } catch (NumberFormatException x) {
            throw new IllegalArgumentException("Illegal value for "+ key, x);
        }
    }
    
    /**
     * Retrieves boolean configuration parameter value from the constructor argument.
     * This method should only be called inside constructor.
     * 
     * @param key Parameter key. 
     * @return Parameter value.
     */
    protected final boolean getBooleanArg(String key) {
        return config.containsKey(key);
    }
    
    /**
     * Retrieves {@code int[]} configuration parameter value from the constructor argument.
     * If the configuration string passed to the constructor did not contain the specified key,
     * an array of the specified length with all elements equal to {@code defaultValue} is returned.
     * This method should only be called inside constructor.
     * 
     * @param key Parameter key. 
     * @param defaultValue Default value for all array elements.
     * @param length Array length.
     * @throws IllegalArgumentException If the given key exists but the corresponding value is not an integer array of the specified length.
     * @return Parameter value.
     */
    protected final int[] getIntArrayArg(String key, int defaultValue, int length) {
        try {
            int[] out = new int[length];
            String s = config.get(key);
            if (s == null) {
                Arrays.fill(out, defaultValue);
            } else {
                String[] ss = s.split(",");
                if(ss.length != length) {
                    throw new IllegalArgumentException("Illegal value for "+ key);
                }
                for (int i = 0; i < length; i++) {
                    out[i] = Integer.parseInt(ss[i].trim());
                }                
            }
            return out;
        } catch (NumberFormatException x) {
            throw new IllegalArgumentException("Illegal value for "+ key, x);
        }
    }
    
    /**
     * Retrieves {@code int[]} configuration parameter value from the constructor argument.
     * If the configuration string passed to the constructor did not contain the specified key, {@code defaultValue} is returned.
     * This method should only be called inside constructor.
     * 
     * @param key Parameter key. 
     * @param defaultValue Default value. If empty, no constraints on the length of the
     *                     array retrieved from the configuration string are enforced.
     * @throws IllegalArgumentException If the given key exists but the corresponding value is
     *                                  not an integer array of the same length as {@code defaultValue}.
     * @return Parameter value.
     */
    protected final int[] getIntArrayArg(String key, int[] defaultValue) {
        try {
            String s = config.get(key);
            if (s == null) {
                return defaultValue;
            } else {
                String[] ss = s.split(",");
                int length = ss.length;
                if(defaultValue.length != 0 && length != defaultValue.length) {
                    throw new IllegalArgumentException("Illegal value for "+ key);
                }
                int[] out = new int[length];
                for (int i = 0; i < length; i++) {
                    out[i] = Integer.parseInt(ss[i].trim());
                }                
                return out;
            }
        } catch (NumberFormatException x) {
            throw new IllegalArgumentException("Illegal value for "+ key, x);
        }
    }

    
// -- Instrumented task class : ------------------------------------------------
    
    /**
     * Instrumented {@code Runnable} for submission to {@code Dispatcher}.
     * Provides empty implementation for {@link Dispatcher.Task#stageEnded Dispatcher.Task.stageEnded(...)}.
     * Allows associating a {@code BusMessage} and a reference time with a task.
     */
    static abstract public class Task implements Dispatcher.Task {
        
        protected final BusMessage busMessage;
        protected final long refTime;
                
        public Task(BusMessage busMessage, long referenceTime) {
            this.busMessage = busMessage;
            this.refTime = referenceTime;
        }
        
        public Task(BusMessage busMessage) {
            this(busMessage, System.currentTimeMillis());
        }

        /**
         * Return the {@code BusMessage} sent or received by this task, if any.
         * 
         * @return Associated bus message, or {@code null} if this task is not sending or receiving a message.
         */
        public BusMessage getBusMessage() {
            return busMessage;
        }

        /**
         * Returns the reference time stamp (milliseconds since millennium) for  this task.
         * All  other times are reported in milliseconds after the reference time.
         * Typically, reference time is the time when the system starts processing a message or notification.
         * 
         * @return Reference timestamp of this task.
         */
        public long getRefTime() {
            return refTime;
        }

        /** Do nothing at the end of a stage. */
        @Override
        public void stageEnded(Stage stage) {}

        
    }
    

}
