package org.lsst.ccs.messaging;

import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.CommandAck;
import org.lsst.ccs.bus.messages.CommandReply;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.utilities.logging.Logger;

import java.io.IOException;
import java.io.Serializable;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.function.Predicate;
import org.lsst.ccs.bus.data.AgentGroup;

import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.definition.Bus;
import org.lsst.ccs.bus.messages.CommandMessage;
import org.lsst.ccs.bus.messages.CommandNack;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.messages.CommandResult;
import org.lsst.ccs.bus.messages.LogMessage;
import org.lsst.ccs.bus.messages.StatusHeartBeat;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.messaging.MessagingAccessLayer.BusAccess;

/**
 * Calls a BusMessagingLayer and adds all that is necessary to handle messages
 * following application layer concerns.
 * <p>
 * Each is bound to a subsystem.
 * <p>
 * This class is meant to be subclassed if there is a need to setup complex
 * correlations between messages (or more generally for "application layer"
 * concerns)
 * </P>
 * <B>beware</B> <TT>ThreadLocal</TT> objects may be changed in future releases:
 * they may be moved to <I>Context</I> objects carried through thread creations.
 */
public class BusApplicationLayer implements ClusterMembershipListener {
    
    private final AgentInfo agentInfo;
    private final String agentName;
    private final BusMessagingLayer busMessagingLayer;
    private static final Logger log = Logger.getLogger("org.lsst.ccs.bus.layer");
    private MessagingAccessLayer messagingAccessLayer;
    
    /** The Agent Presence Manager. */
    private final AgentPresenceManager agentPresenceManager;
    
    private final ForwarderToCommandExecutor commandExecutorForwarder = new ForwarderToCommandExecutor();
    private final ForwarderToCommandOriginator commandOriginatorForwarder = new ForwarderToCommandOriginator();
    
    private ClusterDeserializationErrorHandler cdeh = ClusterDeserializationErrorHandler.DEFAULT;
    
    private final HasClusterMembershipNotifications disconnectionNotificationProvider;
    

    private final BusAccess<LogMessage> logBusAccess;
    private final BusAccess<StatusMessage> statusBusAccess;
    private final BusAccess<CommandMessage> commandBusAccess;
    
    /**
     * creates an entry point to communication for the subsystem. One is negate
     * supposed to interact directly with the <TT>BusMessagingLayer</TT>
     * afterwards except for administrative purposes (example:
     * MembershipListeners).
     *
     * @param subsystemName
     *            rules of naming apply
     * @param busMessagingLayer
     *            transport layer
     * @throws NullPointerException
     *             if busMessagingLayer is null
     */
    BusApplicationLayer(AgentInfo agentInfo, AgentMessagingLayer agentMessagingLayer) {
        
        this.agentInfo = agentInfo;
        this.agentName = agentInfo.getName();
        
        agentPresenceManager = new AgentPresenceManager(agentInfo, agentMessagingLayer);
        
        String protocolProperty = BootstrapResourceUtils.getBootstrapSystemProperties()
                .getProperty("org.lsst.ccs.transport", "jgroups:udp_ccs:");
        String transportPropsProperty = BootstrapResourceUtils.getBootstrapSystemProperties()
                .getProperty("org.lsst.ccs.transport.properties", "");
        try {
            busMessagingLayer = TransportManager.getConnection(protocolProperty, transportPropsProperty);
            if ( busMessagingLayer instanceof HasClusterMembershipNotifications ) {
                disconnectionNotificationProvider = (HasClusterMembershipNotifications)busMessagingLayer;
            } else {
                disconnectionNotificationProvider = new DefaultClusterMembershipNotifier(agentName);
                
            }
        } catch (TransportException exc) {
            // todo: change this exception
            throw new RuntimeException(exc);
        }
        
        disconnectionNotificationProvider.addClusterMembershipListener(agentPresenceManager);        
        disconnectionNotificationProvider.addClusterMembershipListener(this);        
        
        
        logBusAccess = new BusAccess<>(Bus.LOG);
        
        statusBusAccess = new BusAccess<StatusMessage>(Bus.STATUS) {            
            @Override
            public void processClusterDeserializationError(String address, RuntimeException e) {
                cdeh.process(address, e);
            }
        };
        
        commandBusAccess = new BusAccess<CommandMessage>(Bus.COMMAND) {
            @Override
            public void processBusMessage(CommandMessage message) {
                super.processBusMessage(message);
                if (message instanceof CommandRequest) {
                    commandExecutorForwarder.update((CommandRequest) message);
                } else if (message instanceof CommandReply) {
                    commandOriginatorForwarder.update((CommandReply) message);
                }
            }
        };

        messagingAccessLayer = new MessagingAccessLayer(agentInfo, logBusAccess, statusBusAccess, commandBusAccess);
        
        addStatusListener(agentPresenceManager,
                (Predicate<BusMessage<? extends Serializable, ?>>) BusMessageFilterFactory.messageOrigin(agentName).negate());
                
        if ( disconnectionNotificationProvider instanceof StatusMessageListener ) {
            addStatusListener((StatusMessageListener)disconnectionNotificationProvider, (Predicate<BusMessage<? extends Serializable, ?>>) BusMessageFilterFactory
                        .messageOrigin(agentName).negate());
        }
        
    }

    private final List<String> membersInCluster = new CopyOnWriteArrayList<>();
    
    @Override
    public void membersLeft(List<String> left) {
        synchronized(membersInCluster) {
            membersInCluster.removeAll(left);
        }
    }

    @Override
    public void membersJoined(List<String> joined) {
        synchronized(membersInCluster) {
            membersInCluster.addAll(joined);
        }
    }


    
    final boolean hasInternalHeartbeat() {
        return disconnectionNotificationProvider.hasInternalHeartbeat();
    }
    
    /**
     * Add a BusMessagePreProcessor the the BusApplicationLayer.
     * 
     * @param preProcessor The BusMessagePreProcessor to be added.
     */
    final void addBusMessagePreProcessor(BusMessagePreProcessor preProcessor) {
        messagingAccessLayer.addBusMessagePreProcessor(preProcessor);
    }
    
    final BusMessagingLayer getBusMessagingLayer() {
        return busMessagingLayer;
    }
    
    public final MessagingAccessLayer getMessagingAccessLayer() {
        return messagingAccessLayer;
    }
    
    final void connectToBuses() {
        //Add a group filter if the agent defines a AgentGroup.AGENT_GROUP_PROPERTY property.
        //See https://jira.slac.stanford.edu/browse/LSSTCCS-1079
        final String groupNameFilter = agentInfo.getAgentProperty(AgentGroup.AGENT_GROUP_PROPERTY);
        if ( groupNameFilter != null ) {
            Predicate<BusMessage<? extends Serializable, ?>> groupFilter = (BusMessage<? extends Serializable, ?>  bm) -> {
                String group = bm.getOriginAgentInfo().getAgentProperty(AgentGroup.AGENT_GROUP_PROPERTY);
                if ( group == null ) {
                    return true;
                }
                boolean result = group.equals(groupNameFilter);                
                if ( !result ) {
                    synchronized(membersInCluster) {
                        if ( membersInCluster.remove(bm.getOriginAgentInfo().getName()) ) {
                            //Notify the AgentPresenceManager that this agent will
                            //not join the cluster due to its group definition
                            //https://jira.slac.stanford.edu/browse/LSSTCCS-2797
                            agentPresenceManager.processDisconnectionForDelayedNotifications(AgentPresenceManager.AgentPresenceState.DISCONNECTING, bm.getOriginAgentInfo());
                            StatusHeartBeat hb = new StatusHeartBeat(1000);
                            hb.setOriginAgentInfo(agentInfo);                
                            sendStatus(hb);
                        }
                    }
                }
                return result;
            };
            logBusAccess.addBusMessageFilter(groupFilter);
            statusBusAccess.addBusMessageFilter(groupFilter);
            commandBusAccess.addBusMessageFilter(groupFilter);            
        }
                
        try {
            busMessagingLayer.connect(messagingAccessLayer);
        } catch (DuplicateAgentNameException | IOException ex) {
            throw new RuntimeException(ex);
        }        
    }
    
    /**
     * sends a command message to all destinations. If origin of message is
     * negate set then sets it using the current subsystem name. If a
     * Correlation Id is negate set then creates one.
     *
     * @param cmd
     * @throws DestinationsException if the command target is not present on
     * the buses.
     * @see org.lsst.ccs.bus.BusMembershipListener#anormalEvent(Exception) for
     *      another way to signal destinations exceptions
     */
    void sendCommand(CommandRequest cmd, CommandOriginator originator) {
        String destination = BusMessagingLayer.parseDestination(cmd.getDestination());
        if (destination == null || destination.isEmpty()) {
            throw new RuntimeException("Invalid destination for command " + cmd);
        }
        commandOriginatorForwarder.addCommandOriginator(cmd.getCorrelationId(), originator);
        
        if (cmd.getOriginAgentInfo().getName() == null) {
            throw new RuntimeException("Agent name must not be null");
        }
                
        busMessagingLayer.sendMessage(agentName, Bus.COMMAND, cmd);
    }
    
    /**
     * broadcasts a status message. If origin is negate set then sets it with
     * the current subsystem name.
     *
     * @param status
     * @throws IOException
     */
    void sendStatus(StatusMessage status) {
        String origin = status.getOriginAgentInfo().getName();
        if (origin == null) {
            throw new RuntimeException("Agent name must not be null");
        }
        busMessagingLayer.sendMessage(agentName, Bus.STATUS, status);
    }
    
    /**
     * broadcasts a log message. If origin is negate set then sets it with the
     * current subsystem name.
     *
     * @param evt
     * @throws IOException
     */
    void sendLog(LogMessage evt) {
        String origin = evt.getOriginAgentInfo().getName();
        if (origin == null) {
            throw new RuntimeException("Agent name must not be null");
        }
        busMessagingLayer.sendMessage(agentName, Bus.LOG, evt);
        
    }
    
    /**
     * sends a reply or an ack responding to a command. <B>Beware</B> : if
     * <TT>originalCommand</TT> or <TT>CorrelID</TT> or <TT>destination</TT> are
     * negate set in the parameter this code will try to populate these fields
     * with informations coming from <TT>ThreadLocal</TT> data so if the initial
     * Thread that received the command spawn children (or more generally if
     * replies or ack are generated in different Threads) then this facility
     * will be defeated.
     *
     * @param cmd
     */
    void reply(CommandReply cmd) {
        String origin = cmd.getOriginAgentInfo().getName();
        if (origin == null) {
            throw new RuntimeException("Agent name must not be null");
        }
        // String destination = cmd.getDestination();
        busMessagingLayer.sendMessage(agentName, Bus.COMMAND, cmd);
    }
    
    class ForwarderToCommandExecutor implements BusMessageForwarder<CommandRequest> {
        
        private volatile CommandExecutor commandExecutor;
        
        void setCommandExecutor(CommandExecutor commandExecutor) {
            if (this.commandExecutor != null) {
                throw new RuntimeException(
                        "A CommandExecutor is already registered for this Agent. There can be only one!");
            } else {
                this.commandExecutor = commandExecutor;
            }
        }
        
        @Override
        public void update(CommandRequest message) {
            
            // Check that this message was really intended for this
            // CommandExecutor
            String destination = message.getDestination();
            if (destination.contains("/")) {
                destination = destination.substring(0, destination.indexOf("/"));
            }
            if (!destination.equals(agentName)) {
                return;
            }
            
            if (commandExecutor == null) {
                throw new RuntimeException("A CommandExecutor has not been registered for this Agent.");
            }
            try {
                commandExecutor.executeCommandRequest(message);
            } catch (Exception throwable) {
                throwable.printStackTrace();
                log.error("on command :" + throwable);
            }
            
        }
    }
    
    /**
     * Instances of this class forward commands coming from the transport layer
     * to registered {@code CommandOriginator}s. If a {@code CommandResult} is
     * received before {@code CommandAck}, its processing by the
     * {@code CommandOriginator} is postponed. Note, however, that registered
     * command listeners will still be notified of all messages in the order the
     * messages are received.
     */
    class ForwarderToCommandOriginator implements BusMessageForwarder<CommandReply> {
        
        private class RequestData {
            RequestData(CommandOriginator originator) {
                this.originator = originator;
            }
            
            CommandOriginator originator;
            CommandResult result;
            boolean ackReceived;
        }
        
        private final ConcurrentHashMap<UUID, RequestData> currentRequests = new ConcurrentHashMap<>();
        
        ForwarderToCommandOriginator() {
        }
        
        void addCommandOriginator(UUID originatorId, CommandOriginator originator) {
            currentRequests.put(originatorId, new RequestData(originator));
        }
        
        @Override
        public void update(CommandReply message) {
            
            String destination = message.getDestination();
            if (!destination.equals(agentName)) {
                return;
            }
            UUID originatorId = message.getCorrelationId();
            
            synchronized (this) {
                RequestData reqData = currentRequests.get(originatorId);
                if (reqData == null) {
                    throw new RuntimeException(
                            "Could not find the originator of the CommandRequest " + message.getEncodedData());
                }
                if (message instanceof CommandAck) {
                    reqData.originator.processAck((CommandAck) message);
                    if (reqData.result == null) {
                        reqData.ackReceived = true;
                    } else {
                        reqData.originator.processResult(reqData.result);
                        currentRequests.remove(originatorId);
                    }
                } else {
                    if (message instanceof CommandNack) {
                        reqData.originator.processNack((CommandNack) message);
                        currentRequests.remove(originatorId);
                    } else if (message instanceof CommandResult) {
                        if (reqData.ackReceived) {
                            reqData.originator.processResult((CommandResult) message);
                            currentRequests.remove(originatorId);
                        } else {
                            reqData.result = (CommandResult) message;
                        }
                    }
                }
            }
            
        }
    }
    
    /**
     * instances of this class will forward commands coming from the transport
     * layer to the registered <TT>CommandMessageListener</TT>. When the
     * incoming message is a command to be executed the <TT>TreadLocal</TT> data
     * that links commands with their corresponding relpis or ack is populated
     * (see <TT>reply</TT> method documentation).
     */
    private class ForwarderToCommandListeners implements BusMessageForwarder<CommandMessage> {
        
        private final CommandMessageListener listener;
        private final Predicate<BusMessage<? extends Serializable, ?>> filter;
        
        ForwarderToCommandListeners(CommandMessageListener listener,
                Predicate<BusMessage<? extends Serializable, ?>> filter) {
            this.listener = listener;
            this.filter = filter;
        }
        
        @Override
        public void update(CommandMessage message) {
            if (filter == null || filter.test(message)) {
                listener.onCommandMessage(message);
            }
        }
    }
    
    /**
     * instances of this class will forward directly status messages to
     * statuslistener
     */
    private class ForwarderToStatusListeners implements BusMessageForwarder<StatusMessage> {
        
        private final StatusMessageListener listener;
        private final Predicate<BusMessage<? extends Serializable, ?>> filter;
        
        ForwarderToStatusListeners(StatusMessageListener listener,
                Predicate<BusMessage<? extends Serializable, ?>> filter) {
            this.listener = listener;
            this.filter = filter;
        }
        
        @Override
        public void update(StatusMessage message) {
            if (filter == null || filter.test(message)) {
                listener.onStatusMessage(message);
            }
        }
    }
    
    /**
     * instances of this class will forward directly log messages to a log
     * listener
     */
    private class ForwarderToLogListeners implements BusMessageForwarder<LogMessage> {
        
        private final LogMessageListener listener;
        private final Predicate<BusMessage<? extends Serializable, ?>> filter;
        
        ForwarderToLogListeners(LogMessageListener listener, Predicate<BusMessage<? extends Serializable, ?>> filter) {
            this.listener = listener;
            this.filter = filter;
        }
        
        @Override
        public void update(LogMessage message) {
            if (filter == null || filter.test(message)) {
                listener.onLogMessage(message);
            }
        }
    }
    
    /**
     * Set the CommandExecutor for this Agent.
     */
    void setCommandExecutor(CommandExecutor executor) {
        commandExecutorForwarder.setCommandExecutor(executor);
    }
    
    /**
     * registers a CommandMessageListener (in fact a ForwarderToCommandListeners
     * to the underlying transport)
     *
     * @param l
     */
    void addCommandListener(CommandMessageListener l, Predicate<BusMessage<? extends Serializable, ?>> filter) {
        ForwarderToCommandListeners forwarder = new ForwarderToCommandListeners(l, filter);
        messagingAccessLayer.getBusAccess(Bus.COMMAND).addForwarder(l, forwarder);
    }
    
    /**
     * removes a CommandMessageListener: since this command is based on strict
     * identity the listener should be exactly the same as the one registered.
     *
     * @param l
     */
    void removeCommandListener(CommandMessageListener l) {
        messagingAccessLayer.getBusAccess(Bus.COMMAND).removeForwarder(l);
    }
    
    /**
     * registers a StatusMessageListener (in fact a ForwarderToStatusListeners
     * to the underlying transport)
     *
     * @param l
     */
    void addStatusListener(StatusMessageListener l, Predicate<BusMessage<? extends Serializable, ?>> filter) {
        ForwarderToStatusListeners forwarder = new ForwarderToStatusListeners(l, filter);
        messagingAccessLayer.getBusAccess(Bus.STATUS).addForwarder(l, forwarder);
    }
    
    void removeStatusListener(StatusMessageListener l) {
        messagingAccessLayer.getBusAccess(Bus.STATUS).removeForwarder(l);
    }
    
    /**
     * registers a LogMessageListener (in fact a ForwarderToLogListeners to the
     * underlying transport)
     *
     * @param l
     */
    void addLogListener(LogMessageListener l, Predicate<BusMessage<? extends Serializable, ?>> filter) {
        ForwarderToLogListeners forwarder = new ForwarderToLogListeners(l, filter);
        messagingAccessLayer.getBusAccess(Bus.LOG).addForwarder(l, forwarder);
    }
    
    void removeLogListener(LogMessageListener l) {
        messagingAccessLayer.getBusAccess(Bus.LOG).removeForwarder(l);
    }
    
    void setClusterDeserializationErrorHandler(ClusterDeserializationErrorHandler h) {
        this.cdeh = h;
    }
    
    /**
     * closes the underlying transport layer, stops the listening threads, after
     * this call all other sending calls will fail.
     */
    void close() {

        if ( disconnectionNotificationProvider instanceof StatusMessageListener ) {
            removeStatusListener((StatusMessageListener)disconnectionNotificationProvider);
        }
        disconnectionNotificationProvider.removeClusterMembershipListener(agentPresenceManager);
        disconnectionNotificationProvider.clear();
        
        removeStatusListener(agentPresenceManager);        
        agentPresenceManager.disconnect();
        busMessagingLayer.disconnect(messagingAccessLayer);
    }
    
    public AgentPresenceManager getAgentPresenceManager() {
        return agentPresenceManager;
    }
    
}
