package org.lsst.ccs.messaging.util;

import java.util.EnumMap;
import java.util.concurrent.atomic.AtomicLong;
import org.lsst.ccs.bus.definition.Bus;
import org.lsst.ccs.messaging.TransportStateException;

/**
 * {@link Dispatcher} implementation that mimics previously existing system.
 * All processing is done on caller threads.
 *
 * @author onoprien
 */
public class LegacyDispatcher extends AbstractDispatcher {

// -- Fields : -----------------------------------------------------------------
    
    private final EnumMap<Bus,Object> busLocks; // held while sending messages
    private volatile boolean off = true;

// -- Life cycle : -------------------------------------------------------------
    
    public LegacyDispatcher(String args) {
        super(args);
        busLocks = new EnumMap<>(Bus.class);
        for (Bus bus : Bus.values()) {
            busLocks.put(bus, new Object());
        }
        config = null;
    }

    /**
     * Initializes this dispatcher.
     * Empty implementation is provided by this class.
     */
    @Override
    public void initialize() {
        off = false;
    }

    /**
     * Orderly shuts down this dispatcher. Stops accepting tasks, executes previously
     * submitted tasks, delivers events to listeners, then shuts down.
     */
    @Override
    public void shutdown() {
        off = true;
        super.shutdown();
    }
    
// -- Task submission : --------------------------------------------------------

    /**
     * Submits a task to process an incoming message or disconnection notification.
     * The task is executed on the caller thread.
     * <p>
     * Calling this method has no effect if this Dispatcher has not been initialized or it has been shut down.
     * 
     * @param runnable Task to be executed.
     * @param bus Bus (LOG, STATUS, or COMMAND).
     * @param agents Names of affected agents (source or disconnected).
     */
    @Override
    public void in(Runnable runnable, Bus bus, String... agents) {
        if (off) return;
        Dispatcher.Task task = runnable instanceof Dispatcher.Task ? (Dispatcher.Task)runnable : null;
        if (task != null) {
            task.stageEnded(Stage.START);
            task.stageEnded(Stage.WAIT);
        }
        runnable.run();
        if (task != null) {
            task.stageEnded(Stage.RUN);
            task.stageEnded(Stage.SUBMIT);
        }
    }

    /**
     * Submits a task to process an outgoing message.
     * The task is executed on the caller thread.
     * 
     * @param runnable Task to be executed.
     * @param bus Bus (LOG, STATUS, or COMMAND).
     * @param order Order of execution and capacity control policies.
     * @throws TransportStateException If this Dispatcher is unable to accept the request.
     */
    @Override
    public void out(Runnable runnable, Bus bus, Order order) {
        if (off) throw new TransportStateException();
        Dispatcher.Task task = runnable instanceof Dispatcher.Task ? (Dispatcher.Task)runnable : null;
        if (task != null) {
            task.stageEnded(Stage.START);
            task.stageEnded(Stage.WAIT);
        }
        synchronized (busLocks.get(bus)) {
            runnable.run();
        }
        if (task != null) {
            task.stageEnded(Stage.RUN);
            task.stageEnded(Stage.SUBMIT);
        }
    }

}
