package org.lsst.ccs.messaging;

import org.lsst.ccs.bus.messages.LogMessage;
import org.lsst.ccs.utilities.logging.LogPropertiesLoader;
import org.lsst.ccs.utilities.logging.TextFormatter;

import java.io.Closeable;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.logging.*;

/**
 * A java util log Handler that forwards messages to the Log bus.
 * As it may be used in the context of "multi-logging" of the <TT>org.lsst.ccs.utilities.logging</TT>
 * package it checks for duplicate messages.
 * <BR/>
 * Should be a singleton with delegation to a unique static instance (but how to pass the communication layer?)
 * If we share two subsystems on the same JVM the Singleton will not work
 * so it is utmost importance to get the subsystemName directly and create two instances!!!
 *
 * @author bamade
 */
// Date: 30/05/13

public class LogBusHandler extends Handler implements Closeable {
    protected AgentMessagingLayer messagingAccess;
    protected boolean closed = false;
    protected String subSystemName;
    protected ExecutorService executor = Executors.newSingleThreadExecutor(new ThreadFactory() {
        @Override
        public Thread newThread(Runnable r) {
            Thread res = new Thread(r, "logbusHandlerExecutor");
//            res.setDaemon(true);
            return res;
        }
    });
    protected volatile boolean logBusInitialized = false;

    public LogBusHandler(AgentMessagingLayer messagingAccess) {
        this.messagingAccess = messagingAccess;
        init();
    }

    private void init() {
        //default value
        this.setLevel(Level.WARNING);
        String className = this.getClass().getCanonicalName();
        Formatter formatter = LogPropertiesLoader.loaderGetFormatterProperty(className + ".formatter", new TextFormatter());
        this.setFormatter(formatter);
        Level level = LogPropertiesLoader.loaderGetLevelProperty(className + ".level", Level.WARNING);
        this.setLevel(level);
    }


    /**
     * should send a LogRecord over the Log Bus.
     *
     * @param record
     */
    @Override
    public synchronized void publish(LogRecord record) {
       // assert Tracer.trace(" busHandler publish request : [" + getLevel() + "] " + getFormatter().format(record));
        if (closed) return;
        Level curLevel = record.getLevel();
        if (curLevel == null) {
            curLevel = Level.SEVERE;
            record.setLevel(curLevel);
        }
        if (curLevel.equals(Level.SEVERE)) {
            Object[] objects = record.getParameters();
            if (objects != null && objects.length > 0) {
                Object object = objects[0];
                if (object instanceof LogMessage) {
                    Throwable th = record.getThrown();
                    this.getErrorManager().error("suspected panic for: " + object, th instanceof Exception ? (Exception) th : null, ErrorManager.WRITE_FAILURE);
                    return;
                }
            }
        }
        
        if (this.isLoggable(record)) {
            try {
              //  assert Tracer.trace(" busHandler publishing: [" + getLevel() + "] " + record);
                final LogMessage event = generateLogEvent(record) ;
                //DO it in a different Thread
                executor.submit(() -> {
                    sendLogEvent(event);
                    return null;
                });
            } catch (Exception th) {
               this.getErrorManager().error("suspected panic for: " + th.toString(), th instanceof Exception ? (Exception) th : null, ErrorManager.WRITE_FAILURE);
            }
        }
    }

    @Override
    public void flush() {
    }

    @Override
    public synchronized void close() throws SecurityException {
        closed = true;
        //TODO: severe ... may be print the remaining tasks
        executor.shutdown();
    }

    @Override
    public void setLevel(Level level) {
        if (!logBusInitialized) {
            if (level.equals(Level.OFF)) {
                super.setLevel(level);
                return;
            }
            LogRecord record = new LogRecord(Level.FINE, "starting log bus");
            sendLogEvent(generateLogEvent(record));
            logBusInitialized = true;
        }
        super.setLevel(level);
    }

    void sendLogEvent(LogMessage event) {
        if (messagingAccess != null) {
            messagingAccess.sendLogMessage(event);
        }
    }

    LogMessage generateLogEvent(LogRecord record) {
        String threadName = Thread.currentThread().getName();
        String loggerName = record.getLoggerName() ;
        String sourceName = record.getSourceClassName() + "#" + record.getSourceMethodName() ;
        Formatter formatter = getFormatter() ;
        String formattedDetail ;
        if(formatter != null) {
            formattedDetail = getFormatter().format(record);
        } else {
            formattedDetail = String.valueOf(record) ;
        }

        String levelString = String.valueOf(record.getLevel()) ;
        return new LogMessage(threadName,loggerName,sourceName,formattedDetail, levelString);
    }

}