package org.lsst.ccs.messaging.util;

import java.util.EnumMap;
import java.util.concurrent.atomic.AtomicLong;
import org.lsst.ccs.bus.definition.Bus;
import org.lsst.ccs.messaging.TransportStateException;

/**
 * {@link Dispatcher} implementation that mimics previously existing system.
 * All processing is done on caller threads.
 *
 * @author onoprien
 */
public class LegacyDispatcher extends AbstractDispatcher {

// -- Fields : -----------------------------------------------------------------
    
    private final AtomicLong taskID = new AtomicLong();
    private final EnumMap<Bus,Object> busLocks; // held while sending messages
    private volatile boolean off = true;

// -- Life cycle : -------------------------------------------------------------
    
    public LegacyDispatcher(String... args) {
        super(args);
        busLocks = new EnumMap<>(Bus.class);
        for (Bus bus : Bus.values()) {
            busLocks.put(bus, new Object());
        }
    }

    @Override
    public void initialize() {
        super.initialize();
        off = false;
    }

    @Override
    public void shutdown() {
        off = true;
        super.shutdown();
    }
    
// -- Task submission : --------------------------------------------------------

    @Override
    public void in(Runnable task, Bus bus, String... agents) {
        if (off) return;
        long time = System.currentTimeMillis();
        Order order = agents == null ? Order.OOB : (agents.length == 0 ? Order.OOB_CC : Order.NORM);
        long id = taskID.getAndIncrement();
        report(task, id, true, bus, order, time, Stage.START);
        task.run();
        report(task, id, true, bus, order, System.currentTimeMillis() - time, Stage.RUN, Stage.SUBMIT);
    }

    @Override
    public void out(Runnable task, Bus bus, Order order) {
        if (off) throw new TransportStateException();
        long submitTime = System.currentTimeMillis();
        long id = taskID.getAndIncrement();
        report(task, id, false, bus, order, submitTime, Stage.START);
        long runTime;
        synchronized (busLocks.get(bus)) {
            runTime = System.currentTimeMillis();
            report(task, id, false, bus, order, runTime - submitTime, Stage.WAIT);
            task.run();
        }
        report(task, id, false, bus, order, System.currentTimeMillis() - runTime, Stage.RUN, Stage.SUBMIT);
    }

}
