package org.lsst.ccs.messaging.util;

import org.lsst.ccs.bus.definition.Bus;
import org.lsst.ccs.messaging.TransportStateException;

/**
 * Task dispatcher that decouples MessagingLayer implementation (currently, JGroups) from CCS messaging framework.
 * {@code MessagingLayer} submits executable tasks (either message processing or disconnection notifications) through
 * the {@link #in in(...)} method. CCS framework submits message sending tasks through the {@link #out out(...)} method.
 * <p>
 * All implementations should provide public constructor that takes {@code String[]} parameter.
 * This is a way to supply configuration parameters through the protocol string passed to {@code TransportManager}.
 * Implementations should be thread-safe, methods of this interface can be called on any thread.
 * <p>
 * Dispatcher collects statistics on task processing that can be retrieved thorough its getters. Some implementations
 * might not provide all types of statistics. In that case, corresponding getters will return 0.
 *
 * @author onoprien
 */
public interface Dispatcher {
    
    /** Enumeration of order of execution and capacity control policies. */
    public enum Order {
        /** Normal. */ NORM,
        /** Out-of-band, capacity-controlled. */ OOB_CC,
        /** Out-of-band, no capacity-controls. */ OOB}
    
    /** Enumeration of statistics. */
    public enum Stat {MAX, AVERAGE}
    
    /** Enumeration of task execution stages for which statistical data can be reported. */
    public enum Stage {START, WAIT, RUN, SUBMIT}

// -- Life cycle : -------------------------------------------------------------
    /**
     * Initializes this dispatcher.
     */
    default void initialize() {
    }
    
    /**
     * Orderly shuts down this dispatcher.
     * The {@code MessagingLayer} will not be closed until this method returns.
     */
    default void shutdown() {
    }
    
// -- Task submission : --------------------------------------------------------
    
    /**
     * Submits a task to process an incoming message or disconnection notification.
     * Depending on the implementation, this method may return immediately or after the task has been executed.
     * <p>
     * If one or more agent names are given, the task if guaranteed to be executed after any previously submitted
     * tasks for the same bus and agent. If no agents are specified, the task is independent of other tasks,
     * subject to capacity controls imposed by the implementation of this service. If the {@code agents} argument
     * is {@code null}, the task is independent of other tasks, not subject to capacity controls.
     * <p>
     * Calling this method has no effect if this Dispatcher has not been initialized or it has been shut down.
     * No exceptions are thrown in this case.
     * 
     * @param task Task to be executed.
     * @param bus Bus (LOG, STATUS, or COMMAND).
     * @param agents Names of affected agents (source or disconnected).
     * @throws TransportStateException If this Dispatcher is unable to accept the
     *         request for reasons other than being uninitialized or shut down.
     *               
     */
    void in(Runnable task, Bus bus, String... agents);
    
    /**
     * Submits a task to process an outgoing message.
     * Depending on the implementation, this method may return immediately or after the task has been executed.
     * <p>
     * Tasks submitted with {@code outOfBand} equal to {@code false} for the same bus are guaranteed to be
     * processed in the order of submission. If {@code outOfBand} equals {@code true}, the task is executed 
     * independently of others, subject to capacity controls imposed by the implementation of this service.
     * If {@code outOfBand} equals {@code null}, the task is independent, not subject to capacity controls.
     * 
     * @param task Task to be executed.
     * @param bus Bus (LOG, STATUS, or COMMAND).
     * @param order Order of execution and capacity control policies.
     * @throws TransportStateException If this Dispatcher is unable to accept the request, typically
     *         either because it has not been initialized or because it has been shut down.
     */
    void out(Runnable task, Bus bus, Order order);
    
// -- Statistics : -------------------------------------------------------------
    
    public interface Status {

        /**
         * Returns time spent by tasks in the specified stage of processing.
         * Tasks that completed that stage during the specified period are
         * included.
         *
         * @param incoming {@code true} for incoming, {@code false} for outgoing
         * messages.
         * @param bus Bus (LOG, STATUS, or COMMAND).
         * @param ordering Order of execution and capacity control policies.
         * @param stage Processing stage.
         * @param stat Type of statistics.
         * @param period ID of a time period over which statistics should be
         * computed.
         * @return Time in in milliseconds.
         */
        long getTime(boolean incoming, Bus bus, Order ordering, Stage stage, Stat stat, int period);

        /**
         * Returns the number of tasks that completed the specified stage of
         * processing during the specified period.
         *
         * @param incoming {@code true} for incoming, {@code false} for outgoing
         * messages.
         * @param bus Bus (LOG, STATUS, or COMMAND).
         * @param ordering Order of execution and capacity control policies.
         * @param stage Processing stage.
         * @param periodID ID of a time period over which statistics should be
         * computed.
         * @return Number of tasks.
         */
        long getCompletedTasks(boolean incoming, Bus bus, Order ordering, Stage stage, int periodID);

    }
    
    /**
     * Interface to be implemented by classes that should by notified of status changes.
     */
    public interface StatusListener {
        
        /**
         * Called when status of a {@code Dispatcher} changes, typically when
         * statistics for another monitoring period becomes available.
         * Implementations of this method should return promptly, any time-consuming
         * or blocking operations should be off-loaded to other threads.
         *
         * @param status Current status. Should not be modified.
         */
        void statusChanged(Status status);
        
    }
    
    /**
     * Returns statistics and diagnostic data on the current state of this {@code Dispatcher}.
     * Implementations of {@code Status} returned by specific implementations of {@code Dispatcher} may provide additional data.
     * The object returned by this method should be treated as immutable.
     * It reflects the latest recorded state and will not be updated.
     * 
     * @return Dispatcher status, or {@code null} if not available.
     */
    Status getStatus();
    
    /**
     * Registers status listener.
     * The listener will be notified whenever an updated status becomes available.
     * 
     * @param listener Listener to add.
     */
    void addStatusListener(StatusListener listener);
    
    /**
     * Removes status listener.
     * @param listener Listener to remove.
     */
    void removeStatusListener(StatusListener listener);
    
    
// -- Instrumented task class : ------------------------------------------------
    
    public interface Task extends Runnable {
        
        void stageEnded(Stage... stages);
        
    }

}
