package org.lsst.ccs.messaging;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import org.lsst.ccs.bus.data.AgentInfo;

import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusStateChangeNotification;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.utilities.logging.Logger;

public class StateBundleAggregator implements StatusMessageListener, AgentPresenceListener {

    static Logger log = Logger.getLogger("lsst.ccs.bus");
    private final AgentMessagingLayer messagingLayer;

    public StateBundleAggregator(AgentMessagingLayer messagingLayer) {
        this.messagingLayer = messagingLayer;
        this.messagingLayer.getAgentPresenceManager().addAgentPresenceListener(this);
    }

    Set<String> origins = new HashSet<>();
    Set<String> isAvailable = new HashSet<>();

    private final Object updateLock = new Object();

    Map<String, StateBundle> states = new ConcurrentHashMap<String, StateBundle>();

    public static interface Observer {

        public void stateChanged(String sysName, StateBundle out, StateBundle changed);
    }

    // we could delegate the filter to the messaging layer through a filter
    // we do it internally to do it in a more dynamical way, and let
    // clients register us to the bus without knowing our internals
    public void addOrigin(String origin) {
        synchronized(updateLock) {
            origins.add(origin);
            if ( messagingLayer.getAgentPresenceManager().agentExists(origin) ) {
                isAvailable.add(origin);
            }
        }
    }

    @Override
    public void onStatusMessage(StatusMessage msg) {
        String origin = msg.getOriginAgentInfo().getName();

        synchronized (updateLock) {
            //This test should not be needed
            if (!isAvailable.contains(origin)) {
                return;
            }

            if (!origins.contains(origin)) {
                return;
            }

            if (observers.isEmpty()) {
                return;
            }

            StateBundle oldBundle = states.get(origin);
            StateBundle newBundle = null;
            if (oldBundle == null) {
                newBundle = msg.getState();
                oldBundle = new StateBundle();
            } else if (msg instanceof StatusStateChangeNotification) {
                newBundle = msg.getState();
            }

            if (newBundle != null) {
                StateBundle diff = newBundle.diffState(oldBundle);
                StateBundle outDiff = oldBundle.diffState(newBundle);
                log.debug("state of " + origin + " changed : " + diff);
                notifyObservers(origin, outDiff, diff);
                states.put(origin, newBundle);
            }
        }

    }

    ArrayList<Observer> observers = new ArrayList<Observer>();

    public synchronized void addObserver(Observer o) {
        observers.add(o);
    }

    public synchronized void deleteObserver(Observer o) {
        observers.remove(o);
    }

    public void notifyObservers(String origin, StateBundle old, StateBundle change) {
        ArrayList<Observer> obs;
        synchronized (this) {
            obs = new ArrayList<Observer>(observers);
        }
        for (Observer o : obs) {
            o.stateChanged(origin, old, change);
        }

    }

    public StateBundle getState(String origin) {
        return states.get(origin);
    }

    public Map<String, StateBundle> getStates() {
        return Collections.unmodifiableMap(states);
    }

    @Override
    public void connecting(AgentInfo... agents) {
        for ( AgentInfo agent : agents ) {
            String name = agent.getName();
            synchronized (updateLock) {
                if (origins.contains(name)) {
                    isAvailable.add(name);
                }
            }
        }
    }

    @Override
    public void disconnecting(AgentInfo agent) {
        String name = agent.getName();
        synchronized (updateLock) {
            boolean wasThere = isAvailable.remove(name);
            if (wasThere) {
                states.remove(name);
            }
        }
    }

}
