package org.lsst.ccs.messaging;

import java.util.*;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.AgentPropertyPredicate;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.states.PhaseState;

/**
 * Tracks agent connection and disconnection on the buses.
 *
 * @author emarin
 */
public class AgentPresenceManager implements StatusMessageListener, ClusterDisconnectionsListener {

    private final Object agentsLock = new Object();
    final CopyOnWriteArrayList<AgentPresenceListener> listAPL = new CopyOnWriteArrayList<>();
    private final Map<AgentInfo, AgentPresenceState> mapAgents = new ConcurrentHashMap<>();
    private final List<AgentInfo> fullyConnectedAgents = new CopyOnWriteArrayList<>();
    private final List<AgentInfo> pendingConnectedAgents = new CopyOnWriteArrayList<>();
    private static final Logger LOG = Logger.getLogger(AgentPresenceManager.class.getName());
    
    /** Lock used by clients waiting for a particular Agent to be fully connected. */
    private final ReentrantLock agentConnectionWaitLock = new ReentrantLock(true);
    /** List of clients waiting for a particular Agent to be fully connected. */
    private volatile LinkedHashMap<Condition,AgentPropertyPredicate> agentConnectionWaitList;

    /** Lock used by clients waiting for a particular Agent to be disconnected. */
    private final ReentrantLock agentDisconnectionWaitLock = new ReentrantLock(true);
    /** List of clients waiting for a particular Agent to be disconnected. */
    private volatile LinkedHashMap<Condition,String> agentDisconnectionWaitList;

    private volatile TimerTask delayedNotificationTask;
    private final Timer timer = new Timer(true);
    
    private final AgentInfo agentInfo;

    //Queue for processing listener's notifications.
    private final BlockingQueue<Runnable> notifications = new ArrayBlockingQueue<>(100);
    
    public enum AgentPresenceState {
        CONNECTING, CONNECTED, DISCONNECTING;
    }
            
    public AgentPresenceManager(AgentInfo agentInfo) {		
        ExecutorService queueExecutor = Executors.newSingleThreadExecutor(r -> {
            Thread t = new Thread(r, "Connection/Disconnection queue");
            t.setDaemon(true);
            return t;
        });		
        queueExecutor.submit(()->checkQueue());		
        this.agentInfo = agentInfo;
    }		
    
    /**
     * This method is invoked when the agent shuts down 
     */
    void disconnect() {
        synchronized (agentsLock) {
            mapAgents.clear();
            fullyConnectedAgents.clear();
        }        
    }
    		   
    private void checkQueue() {
        try {
            while (true) {
                notifications.take().run();
            }
        } catch (InterruptedException e) {
            throw new RuntimeException("Failed when draining Connection/Disconnection queue.", e);
        }
    }
    
    
    @Override
    public void onStatusMessage(StatusMessage s) {

        AgentInfo a = (s.getOriginAgentInfo());
        if (a == null) return;

        if ( a.getName().startsWith("image-handling") ) {
            LOG.log(Level.FINEST,"Message from {0} in state {1}", new Object[]{a.getName(),s.getState().getState(PhaseState.class)});
        }
        if (s.getState().isInState(PhaseState.OFF_LINE)) {
            disconnectedAgent(a);
        } else if (s.getState().isInState(PhaseState.INITIALIZING) ){
            updateAgent(a, AgentPresenceState.CONNECTING);
        } else {
            updateAgent(a, AgentPresenceState.CONNECTED);            
        }
    }

    /** Called on non-OFF_LINE agent status message. */
    private void updateAgent(AgentInfo a, AgentPresenceState state) {
        ConnectionDisconnectionNotification notification = null;     

        LOG.log(Level.FINEST,"Updating {0} for AP State {1}: exists? {2} ({3})", new Object[]{a.getName(),state,mapAgents.containsKey(a), mapAgents.get(a)});
        synchronized (agentsLock) {
            AgentPresenceState oldState = mapAgents.get(a);
            if (oldState != null) { // Agent already known to the map as it came through the CONNECTING step
                if (oldState != state) {
                    if ( state != AgentPresenceState.CONNECTED ) {
                        throw new RuntimeException("Something went really wrong with this agent's state: "+a.getName()+" "+state);
                    }
                    mapAgents.put(a, AgentPresenceState.CONNECTED);
                    //This is an update to the current state to trigger a "connected" notification
                    notification = new ConnectionDisconnectionNotification(AgentPresenceState.CONNECTED, a);
                }
            } else { // New agent
                mapAgents.put(a, state);
                //This is a brand new notification and the Agent can be either
                //CONNECTED or CONNECTING
                //If it is in CONNECTING state then we process it immediately
                //otherwise if it's in CONNECTED we delay the notification.
                //The reason is that we detect CONNECTED agents without seeing
                //the CONNECTING state when we either join the buses and many
                //agents are already present or we are merging after a clster split
                if ( state == AgentPresenceState.CONNECTING ) {
                    notification = new ConnectionDisconnectionNotification(AgentPresenceState.CONNECTING, a);
                } else if ( state == AgentPresenceState.CONNECTED ) {
                    notification = new ConnectionDisconnectionNotification(AgentPresenceState.CONNECTING, a);
                    pendingConnectedAgents.add(a);
                    submitDelayedNotificationIfNeeded();                    
                }
            }
            if ( notification != null ) {
                try { 
                    notifications.put(notification);                
                } catch (InterruptedException ie) {
                    throw new RuntimeException("Problem submitting AgentPresenceManager notifications ",ie);
                }
            }
        }    
    }
    
    class DelayedConnectedNotificationTask extends TimerTask {
        

        @Override
        public void run() {
            processDelayedConnectionNotification();
        }
        
    }
    
    
    private void submitDelayedNotificationIfNeeded() {
        LOG.log(Level.FINEST,"Submitting delayed notification {0} ", new Object[]{delayedNotificationTask});
        synchronized(agentsLock) {
            if ( delayedNotificationTask == null ) {
                delayedNotificationTask = new DelayedConnectedNotificationTask();
                timer.schedule(delayedNotificationTask, agentInfo.getType().compareTo(AgentInfo.AgentType.CONSOLE) >= 0 && ! agentInfo.isScriptingConsole() ? 1200L : 0L);
            }
        }
        
    }
    
    private void processDelayedConnectionNotification() {
        LOG.log(Level.FINEST,"Processing delayed notification {0} ", new Object[]{pendingConnectedAgents});
        synchronized(agentsLock) {
            if ( !pendingConnectedAgents.isEmpty() ) {
                AgentInfo[] connectedAgents = pendingConnectedAgents.toArray(new AgentInfo[pendingConnectedAgents.size()]);
                notifications.offer(new ConnectionDisconnectionNotification(AgentPresenceState.CONNECTED, connectedAgents));
                pendingConnectedAgents.clear();
                delayedNotificationTask = null;
            }
        }
    }

    /** Called to remove agent. */
    private void disconnectedAgent(AgentInfo... agents) {
        synchronized (agentsLock) {
            List<AgentInfo> disconnectedAgents = new ArrayList<>();
            for ( AgentInfo agent : agents ) {
                AgentPresenceState state = mapAgents.remove(agent);
                if (state != null) {
                    disconnectedAgents.add(agent);
                    pendingConnectedAgents.remove(agent);
                }
            }
            
            if ( ! disconnectedAgents.isEmpty() ) {
                LOG.log(Level.FINER,"disconnecting agents {0}",disconnectedAgents);
                ConnectionDisconnectionNotification notification = new ConnectionDisconnectionNotification(AgentPresenceState.DISCONNECTING, disconnectedAgents.toArray(new AgentInfo[disconnectedAgents.size()]));
                notifications.offer(notification);
            }
            
        }
            
    }
    
    /**
     *
     * @return The list with the currently connected Agents
     */
    public List<AgentInfo> listConnectedAgents() {
        return new ArrayList<>(fullyConnectedAgents);
    }

    @Override
    public void membersLeft(List<String> left) {
        List<AgentInfo> agentsToRemove = new ArrayList<>();
        synchronized (agentsLock) {
            for (AgentInfo a : mapAgents.keySet()) {
                if ( left.contains(a.getName()) ) {
                    agentsToRemove.add(a);
                }
            }
        }
        
        if (!agentsToRemove.isEmpty()) {
            disconnectedAgent(agentsToRemove.toArray(new AgentInfo[agentsToRemove.size()]));
        }
    }

    public void addAgentPresenceListener(AgentPresenceListener l) {
        synchronized (agentsLock) {
            l.connecting(mapAgents.keySet().toArray(new AgentInfo[0]));
            l.connected(fullyConnectedAgents.toArray(new AgentInfo[0]));
            listAPL.add(l);
        }
    }

    public void removeAgentPresenceListener(AgentPresenceListener l) {
        listAPL.remove(l);
    }

    /**
     * This method returns true as soon as the AgentPresenceManager is aware
     * of the existence of the given agent by name.
     * At this point we don't guarantee that the corresponding agent is ready
     * to receive command requests as it might still be INITIALIZING.
     * To be sure that an agent is fully connected, OPERATIONAL and ready to
     * receive message use method {@code #isAgentConnected(String)} instead.
     * 
     * @param agentName The name of the agent
     * @return   true if the AgentPresenceManager is aware of the existence of this agent.
     */
    public boolean agentExists(String agentName) {
        List<AgentInfo> existingAgents = new ArrayList<>(mapAgents.keySet());
        for (AgentInfo a : existingAgents) {
            if (a.getName().equals(agentName)) {
                return true;
            }
        }
        return false;
    }

    /**
     * This method returns true when the given agent is fully connected, OPERATIONAL
     * and ready to receive commands.
     * 
     * @param agentName The name of the Agent.
     * @return     true when the given agent is fully connected.
     */
    public boolean isAgentConnected(String agentName) {
        synchronized(fullyConnectedAgents) {
            for (AgentInfo a : fullyConnectedAgents) {
                if (a.getName().equals(agentName)) {
                    return true;
                }
            }
            return false;
        }
    }
    /**
     * This method returns true when the given agent is fully connected, OPERATIONAL
     * and ready to receive commands.
     * 
     * @param agentPredicate The predicate to select the Agent.
     * @return     true when the given agent is fully connected.
     */
    public boolean isAgentConnected(AgentPropertyPredicate agentPredicate) {
        synchronized(fullyConnectedAgents) {
            for (AgentInfo a : fullyConnectedAgents) {
                if (agentPredicate.test(a)) {
                    return true;
                }
            }
            return false;
        }
    }

    /**
     * Wait for an Agent to be fully connected: all the listeners have been
     * notified on the connected method.
     * @param name The name of the Agent
     * @param timeout The time to wait for the agent to become available.
     * @param unit The TimeUnit for the timeout.
     * @return True if the Agent is fully connected within the provided timeout, false otherwise.
     * @throws InterruptedException 
     */
    public final boolean waitForAgent(String name, long timeout, TimeUnit unit) throws InterruptedException {
        Map<String,String> properties = new HashMap<>();
        properties.put("agentName",name);
        AgentPropertyPredicate innerPredicate = new AgentPropertyPredicate(properties);
        return waitForAgent(innerPredicate,timeout,unit);
    }

    public final boolean waitForAgent(AgentPropertyPredicate predicate, long timeout, TimeUnit unit) throws InterruptedException {
        long deadline = TimeUnit.MILLISECONDS.convert(timeout, unit) + System.currentTimeMillis();
        if (!agentConnectionWaitLock.tryLock(timeout, unit)) return false;
        try {
            if ( isAgentConnected(predicate) ) {
                return true;
            }
            Condition condition = agentConnectionWaitLock.newCondition();
            if (agentConnectionWaitList == null) agentConnectionWaitList = new LinkedHashMap<>(4);
            agentConnectionWaitList.put(condition, predicate);
            while (agentConnectionWaitList != null && agentConnectionWaitList.containsKey(condition)) {
                timeout = deadline - System.currentTimeMillis();
                if (!condition.await(timeout, TimeUnit.MILLISECONDS)) {
                    if (agentConnectionWaitList != null) agentConnectionWaitList.remove(condition);
                    if (agentConnectionWaitList.isEmpty()) agentConnectionWaitList = null;
                    return false;
                }
            }
        } finally {
            agentConnectionWaitLock.unlock();
        }
        return true;
    }

    
    private void agentConnectionWaitNotify(final AgentInfo agentInfo) {
        try {
            agentConnectionWaitLock.lockInterruptibly();
            try {
                if (agentConnectionWaitList == null) {
                    return;
                }
                Iterator<Map.Entry<Condition, AgentPropertyPredicate>> it = agentConnectionWaitList.entrySet().iterator();
                while (it.hasNext()) {
                    Map.Entry<Condition, AgentPropertyPredicate> entry = it.next();
                    AgentPropertyPredicate agentWaitPredicate = entry.getValue();
                    if (agentWaitPredicate.test(agentInfo)) {
                        entry.getKey().signal();
                        it.remove();
                    }
                }
                if (agentConnectionWaitList.isEmpty()) {
                    agentConnectionWaitList = null;
                }
            } finally {
                agentConnectionWaitLock.unlock();
            }
        } catch (InterruptedException x) {
            LOG.log(Level.SEVERE,"Exception when notifying for disconnection ", x);
        }
    }

    /**
     * Wait for an Agent to be fully disconnected: all the listeners have been
     * notified on the disconnected method.
     * @param name The name of the Agent
     * @param timeout The time to wait for the agent to become available.
     * @param unit The TimeUnit for the timeout.
     * @return True if the Agent is disconnected from the buses within the provided timeout, false otherwise.
     * @throws InterruptedException 
     */
    public final boolean waitForAgentDisconnection(String name, long timeout, TimeUnit unit) throws InterruptedException {
        long deadline = TimeUnit.MILLISECONDS.convert(timeout, unit) + System.currentTimeMillis();
        if (!agentDisconnectionWaitLock.tryLock(timeout, unit)) return false;
        try {
            if ( ! isAgentConnected(name) ) {
                return true;
            }
            Condition condition = agentDisconnectionWaitLock.newCondition();
            if (agentDisconnectionWaitList == null) agentDisconnectionWaitList = new LinkedHashMap<>(4);
            agentDisconnectionWaitList.put(condition, name);
            while (agentDisconnectionWaitList != null && agentDisconnectionWaitList.containsKey(condition)) {
                timeout = deadline - System.currentTimeMillis();
                if (!condition.await(timeout, TimeUnit.MILLISECONDS)) {
                    if (agentDisconnectionWaitList != null) agentDisconnectionWaitList.remove(condition);
                    if (agentDisconnectionWaitList.isEmpty()) agentDisconnectionWaitList = null;                    
                    return false;
                }
            }
        } finally {
            agentDisconnectionWaitLock.unlock();
        }
        return true;
    }

    private void agentDisconnectionWaitNotify(final String name) {
        try {
            agentDisconnectionWaitLock.lockInterruptibly();
            try {
                if (agentDisconnectionWaitList == null) {
                    return;
                }
                Iterator<Map.Entry<Condition, String>> it = agentDisconnectionWaitList.entrySet().iterator();
                while (it.hasNext()) {
                    Map.Entry<Condition, String> entry = it.next();
                    String agentWaitName = entry.getValue();
                    if (name.equals(agentWaitName)) {
                        entry.getKey().signal();
                        it.remove();
                    }
                }
                if (agentDisconnectionWaitList.isEmpty()) {
                    agentDisconnectionWaitList = null;
                }
            } finally {
                agentDisconnectionWaitLock.unlock();
            }
        } catch (InterruptedException ie) {
            LOG.log(Level.SEVERE,"Exception when notifying for disconnection ", ie);
        }
    }

    private class ConnectionDisconnectionNotification implements Runnable {
        private final AgentInfo[] agentInfos;
        private final AgentPresenceState state;
        
        ConnectionDisconnectionNotification(AgentPresenceState state, AgentInfo... agentInfos ) {
            this.agentInfos = agentInfos;
            this.state = state;
        }
        
        @Override
        public void run() {
            LOG.log(Level.FINEST, "Processing APL notifications for state {0} and agents {1}", new Object[]{state, Arrays.asList(agentInfos)});
            
            switch (state) {
                
                case DISCONNECTING:
                    listAPL.forEach((l) -> {
                        try {
                            l.disconnected(agentInfos);
                        } catch (RuntimeException x) {
                            warn(l, x, state);
                        }
                    });
                    //Agents must be removed before the notification is issued
                    //or a deadlock potential will occurr
                    for (AgentInfo agentInfo : agentInfos) {
                        fullyConnectedAgents.remove(agentInfo);
                        agentDisconnectionWaitNotify(agentInfo.getName());
                    }
                    break;
                    
                case CONNECTING:
                    listAPL.forEach((l) -> {
                        try {
                            l.connecting(agentInfos);
                        } catch (RuntimeException x) {
                            warn(l, x, state);
                        }
                    });
                    break;
                    
                case CONNECTED:
                    listAPL.forEach((l) -> {
                        try {
                            l.connected(agentInfos);
                        } catch (RuntimeException x) {
                            warn(l, x, state);
                        }
                    });
                    //Agents must be added before the notification is issued
                    //or a deadlock potential will occurr
                    for (AgentInfo agentInfo : agentInfos) {
                        fullyConnectedAgents.add(agentInfo);
                        agentConnectionWaitNotify(agentInfo);
                    }
                    break;
            }
        }
        
        private void warn(AgentPresenceListener listener, RuntimeException x, AgentPresenceState state) {
            String agents = agentInfos.length == 1 ? agentInfos[0].getName() : Arrays.asList(agentInfos).stream().map(a -> a.getName()).collect(Collectors.joining(", ", "[", "]"));
            LOG.log(Level.WARNING, "Exception while notifying "+ listener.getClass().getSimpleName() +" of "+ state +" of "+ agents, x);
        }
        
    }
    
}
