package org.lsst.ccs.messaging;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.messages.StatusHeartBeat;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.states.PhaseState;

/**
 * A class that provides Cluster Membership notifications when the 
 * transport layer does not provide such notifications.
 * 
 * This is a message based implementation.
 * 
 */
class DefaultClusterMembershipNotifier implements StatusMessageListener, HasClusterDisconnectionNotifications {
    
    private final AgentPresenceListenerDelayedNotification delayedNotification = new AgentPresenceListenerDelayedNotification();
    private final Map<AgentInfo, TimeoutTask> mapAgents = new ConcurrentHashMap<>();
    private final String agentName;
    private final Timer timer = new Timer(true);
    private final Object agentsLock = new Object();
    private static final int SUSPICION_LENGTH = 10;
    private static final Logger LOG = Logger.getLogger(DefaultClusterMembershipNotifier.class.getName());
    private final List<ClusterDisconnectionsListener> disconnectionListeners = new CopyOnWriteArrayList<>();

    
    
    public DefaultClusterMembershipNotifier(String agentName) {
        this.agentName = agentName;
        
        
    }

    public void clear() {
        synchronized (agentsLock) {
            for (TimerTask task : mapAgents.values()) {
                if (task != null) {
                    task.cancel();
                }
            }
            delayedNotification.stopNotifications();
            mapAgents.clear();
        }        
    } 
    
    
    
    @Override
    public void onStatusMessage(StatusMessage s) {
        AgentInfo a = (s.getOriginAgentInfo());
        if (a == null) return;

        int broadCastPeriod = -1;
        if (s instanceof StatusHeartBeat) {
            StatusHeartBeat hb = (StatusHeartBeat) s;
            broadCastPeriod = hb.getStatusBroadcastPeriod();
        }
        
        if (!s.getState().isInState(PhaseState.OFF_LINE)) {
            //Normal disconnection
            removeAgent(a, false);
        } else {
            updateAgent(a, broadCastPeriod);
        } 
    }
    
    
    class TimeoutTask extends TimerTask {
        
        private final AgentInfo agent;
        private final int broadcastPeriod;
        
        TimeoutTask(AgentInfo agent, int broadcastPeriod) {
            this.agent = agent;
            this.broadcastPeriod = broadcastPeriod;
        }

        int getBroadcastPeriod() {
            return broadcastPeriod;
        }

        @Override
        public void run() {
            removeAgent(agent, true);
        }
        
    }
    
    /** Called to remove agent. */
    private void removeAgent(AgentInfo agent, boolean sendNotification) {
        LOG.log(Level.FINE, "Removing agent {0}", agent.getName());
        TimeoutTask t = null;
        synchronized (agentsLock) {
            t = mapAgents.remove(agent);
            if (t != null) {
                t.cancel();
                if ( sendNotification) {
                    delayedNotification.removeAgent(agent);
                }
            }
        }
        if (t == null) {
            LOG.log(Level.FINEST,"removing agent with null timer");
        }
    }
    
    /** Called on non-OFF_LINE agent status message. */
    private void updateAgent(AgentInfo a, int broadcastPeriod) {
        LOG.log(Level.FINE, "Upating {0}", a.getName());
        synchronized (agentsLock) {
            
            TimeoutTask task = mapAgents.get(a);
            
            if (task != null) { // Agent already known to the map : its timeouttask is updated
                task.cancel();
                if (broadcastPeriod == -1) {
                    broadcastPeriod = task.getBroadcastPeriod();
                }
                task = new TimeoutTask(a, broadcastPeriod);
            } else { // New agent or returning after disappearing
                if (broadcastPeriod == -1) {
                    broadcastPeriod = 500;
                }
                task = new TimeoutTask(a, broadcastPeriod);
            }
            
            mapAgents.put(a, task);
            timer.schedule(task, SUSPICION_LENGTH * 1000 * broadcastPeriod);
        }    
        LOG.log(Level.FINER, "reset timer for agent {0} to {1}", new Object[]{a.getName(), broadcastPeriod});
    }
    
    
    /**
     * The purpose of this class is to group together notifications with
     * the same AgentPresenceState when they happen in rapid succession.
     * When a notification is added to this class if there are existing notifications
     * of in a different state, then we send out the existing notification and
     * submit the new one with a delay.
     * 
     */    
    class AgentPresenceListenerDelayedNotification {

        private final List<AgentInfo> leavingAgents = new CopyOnWriteArrayList<>();
        private final static long DELAYED_WAIT_PERIOD = 1000L;
        private final Object notificationLock = new Object();
        private DelayedNotificationTask notificationTask;
        private volatile boolean isStopped = false;

        
        private class DelayedNotificationTask extends TimerTask { 
            @Override
            public void run() {
                synchronized (notificationLock) {
                    if ( isStopped ) {
                        return;
                    }
                    LOG.log(Level.FINE, "{0} Submitting notification for agents leaving: {1}", new Object[]{agentName, leavingAgents});
                    ArrayList<String> toSubmit = new ArrayList<>();
                    
                    if ( !leavingAgents.isEmpty() ) {
                        leavingAgents.forEach((a) -> {
                            toSubmit.add(a.getName());
                        });
                        for (ClusterDisconnectionsListener l : disconnectionListeners) {
                            l.membersLeft(toSubmit);
                        }
                    }                     
                    leavingAgents.clear();
                }
            }
        }
        
        public void removeAgent(AgentInfo agent) {
            synchronized(notificationLock) {       
                if ( isStopped ) {
                    return;
                }
                if (leavingAgents.contains(agent)) {
                    return;
                }
                LOG.log(Level.FINE, "{0} Removing agent {1} to list of notifications", new Object[]{agentName, agent});
                leavingAgents.add(agent);

                if (notificationTask != null) {
                    notificationTask.cancel();
                }
                notificationTask = new DelayedNotificationTask();
                timer.schedule(notificationTask, DELAYED_WAIT_PERIOD);
            }
        }

        
        void stopNotifications() {
            synchronized(notificationLock) {       
                isStopped = true;
                if (notificationTask != null) {
                    notificationTask.cancel();
                }
                leavingAgents.clear();
            }
        }
    }
    
    @Override
    public void addClusterMembershipListener(ClusterDisconnectionsListener listener) {
        disconnectionListeners.add(listener);
    }

    @Override
    public void removeClusterMembershipListener(ClusterDisconnectionsListener listener) {
        disconnectionListeners.remove(listener);
    }
    
    
}
