package org.lsst.ccs.messaging;

import org.lsst.ccs.bus.definition.Bus;
import org.lsst.ccs.bus.messages.BusMessage;

/**
 * Interface to be implemented by Agent Services that wish to modify the
 * content of an incoming BusMessage.
 *
 * Objects implementing this interface are automatically registered with the
 * MessagingAccessLayer after the HasLifecycle::build step with the appropriate
 * bus.
 * 
 * All the BusMessagePreProcessor interfaces are called to pre-process incoming
 * BusMessages before any MessageListeners are called.
 * 
 * This interface allows Agent Services to introduce a layer of separation 
 * between the messages sent on the CCS buses and the messages offered to 
 * MessageListeners.
 * 
 * @author The LSST CCS Team
 */
public interface BusMessagePreProcessor {

    /**
     * Pre-process the incoming BusMessage and return the message to be 
     * forwarded to the bus message listeners.
     * 
     * The returned message can either be a modified version of the original 
     * message or a brand new message.
     * 
     * The Class of the returned message can be different than the incoming message.
     * 
     * If null is returned the message will not be sent to the BusMessage listeners,
     * but it will still be passed to other BusMessagePreProcessors
     * 
     * @param msg The incoming BusMessage
     * @return    The modified BusMessage to be forwarded to the BusMessageListeners
     * 
     **/
    public BusMessage preProcessMessage(BusMessage msg);

    /**
     * The Bus this BusMessagePreProcessor applies to.
     * 
     * @return The Bus.
     */
    public Bus getBus();
}
