package org.lsst.ccs.messaging;

import org.lsst.ccs.utilities.logging.Logger;

import java.io.Serializable;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.function.Predicate;

import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.AgentLock;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.CommandReply;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.messages.LogMessage;
import org.lsst.ccs.bus.messages.StatusMessage;

/**
 * This interface provides messaging methods for a component (eg an agent) to be
 * able to communicate on the buses. Provided methods allow to add/remove
 * message listeners on each of the buses, to send message on each of the buses,
 * to specify a unique CommandExecutor associated to the agent and a list of
 * CommandOriginator.
 * 
 * @author LSST CCS Team
 */
public class AgentMessagingLayer implements AgentMessagingLayerMBean {
    
	private static final List<AgentMessagingLayer> msgAccesses = new CopyOnWriteArrayList<>();

	private Predicate<BusMessage<? extends Serializable, ?>> filterMessagesFromThisAgent;

	private final Logger curLogger = Logger.getLogger("org.lsst.ccs.bus");

	private final BusApplicationLayer layer;

        private volatile boolean isConnected = false;
        
        private final AgentInfo agentInfo;
        private final LockLevelService lockLevelService;
        
	/**
	 * Build an Instance of an AgentMessagingLayer for a given Agent by
	 * providing the AgentInfo object.
	 *
         * @param agentInfo
         * @param lockLevelService
	 * @return The corresponding AgentMessagingLayer.
	 */
	public static AgentMessagingLayer createInstance(AgentInfo agentInfo, LockLevelService lockLevelService) {
		AgentMessagingLayer agentMessagingLayer = new AgentMessagingLayer(agentInfo, lockLevelService);
		return agentMessagingLayer;
	}

	AgentMessagingLayer(AgentInfo agentInfo, LockLevelService lockLevelService) {
            this.layer = new BusApplicationLayer(agentInfo);
            filterMessagesFromThisAgent = BusMessageFilterFactory.messageOrigin(agentInfo.getName()).negate();            
            this.agentInfo = agentInfo;
            this.lockLevelService = lockLevelService;
	}
                
        /**
         * This method is meant to be invoked from JMX
         * @param seconds The number of seconds to wait before restarting the messaging layer.
         * 
         */
        @Override
        public void restart(int seconds) {
            shutdownBusAccess();
            Thread restart = new Thread(
                () -> {
                    try {
                        Thread.sleep(1000*seconds);
                    } catch (InterruptedException ie) {
                        throw new RuntimeException(ie);
                    }
                    connectToBuses();                    
                }
            );
        }
       
        BusApplicationLayer getApplicationLayer() {
            return layer;
	}
        
        public AgentInfo getAgentInfo() {
            return agentInfo;
        }

	public void shutdownBusAccess() {
            isConnected = false;
            layer.close();
            msgAccesses.remove(this);
	}

        public void connectToBuses() {
            layer.connectToBuses();
            msgAccesses.add(this);
            isConnected = true;
        }
        
	public AgentPresenceManager getAgentPresenceManager() {
            return layer.getAgentPresenceManager();
	}
        
        /**
         * 
         * @return The public API
         */
        public LockLevelService getAgentLockService() {
            return lockLevelService;
        }

	/**
	 * Adds a listener on the Log bus. By default the listener is negate passed
	 * messages published from this AgentMessagingLayer.
	 * 
	 * @param listener
	 *            the listener to be added on the Log bus
	 */
	public void addLogMessageListener(LogMessageListener listener) {
            addLogMessageListener(listener, filterMessagesFromThisAgent);
	}

	/**
	 * Adds a listener on the Log bus with a filter. The listener is passed bus
	 * messages that pass the filter. If the filter is null, all messages are
	 * passed to the filter.
	 * 
	 * @param listener
	 *            the listener to be added on the Log bus
	 * @param filter
	 *            The BusMessageFilter to be applied to the incoming Bus
	 *            Messages
	 */
	public void addLogMessageListener(LogMessageListener listener,
			Predicate<BusMessage<? extends Serializable, ?>> filter) {
		layer.addLogListener(listener, filter);
	}

	/**
	 * Adds a listener on the Status bus. By default the listener is negate
	 * passed messages published from this AgentMessagingLayer.
	 * 
	 * @param listener
	 *            the listener to be added on the Status bus
	 */
	public void addStatusMessageListener(StatusMessageListener listener) {
            addStatusMessageListener(listener, filterMessagesFromThisAgent);
	}

	/**
	 * Adds a listener on the Status bus with a filter. The listener is passed
	 * bus messages that pass the filter. If the filter is null, all messages
	 * are passed to the filter.
	 * 
	 * @param listener
	 *            the listener to be added on the Status bus
	 * @param filter
	 *            The BusMessageFilter to be applied to the incoming Bus
	 *            Messages
	 */
	public void addStatusMessageListener(StatusMessageListener listener,
			Predicate<BusMessage<? extends Serializable, ?>> filter) {
		layer.addStatusListener(listener, filter);
	}

	/**
	 * Adds a listener on the Command bus. By default the listener is negate
	 * passed messages published from this AgentMessagingLayer.
	 * 
	 * @param listener
	 *            the listener to be added on the Command bus
	 */
	public void addCommandMessageListener(CommandMessageListener listener) {
            addCommandMessageListener(listener, filterMessagesFromThisAgent);
	}

	/**
	 * Adds a listener on the Command bus with a filter. The listener is passed
	 * bus messages that pass the filter. If the filter is null, all messages
	 * are passed to the filter.
	 * 
	 * @param listener
	 *            the listener to be added on the Command bus
	 * @param filter
	 *            The BusMessageFilter to be applied to the incoming Bus
	 *            Messages
	 */
	public void addCommandMessageListener(CommandMessageListener listener,
			Predicate<BusMessage<? extends Serializable, ?>> filter) {
            layer.addCommandListener(listener, filter);
	}

	/**
	 * Removes a listener on the Log bus
	 * 
	 * @param listener
	 *            the listener to be removed on the Log bus
	 */
	public void removeLogMessageListener(LogMessageListener listener) {
            layer.removeLogListener(listener);
	}

	/**
	 * Removes a listener on the Status bus
	 * 
	 * @param listener
	 *            the listener to be removed on the Status bus
	 */
	public void removeStatusMessageListener(StatusMessageListener listener) {
            layer.removeStatusListener(listener);
	}

	/**
	 * Removes a listener on the Command bus
	 * 
	 * @param listener
	 *            the listener to be removed on the Command bus
	 */
	public void removeCommandMessageListener(CommandMessageListener listener) {
            layer.removeCommandListener(listener);
	}

	/**
	 * Sends a Log Message on the Log Bus
	 * 
	 * @param msg The message to be sent on the Log bus
	 */
        public void sendLogMessage(LogMessage msg) {
            if ( !isConnected ) {
                throw new RuntimeException("A connection to the buses has not been established yet.");
            }
            msg.setOriginAgentInfo(agentInfo);
            layer.sendLog(msg);
	}

	/**
	 * Sends a Status Message on the Status Bus
	 * 
	 * @param msg The message to be sent on the Status bus
	 */
	public void sendStatusMessage(StatusMessage msg) {
            if ( !isConnected ) {
                throw new RuntimeException("A connection to the buses has not been established yet.");
            }
		msg.setOriginAgentInfo(agentInfo);
		curLogger.finest("sending status " + msg);
                layer.sendStatus(msg);
	}

	/**
	 * Sends a Command Request on the Command Bus. This message will be received
	 * by the CommandExecutor and the list of CommandMessageListener
	 * 
	 * @param cmd
	 *            The CommandRequest to be sent on the Command bus
	 * @param originator
	 *            The component that has requested the execution of the command
         * @throws DestinationsException if the command target is not present on 
         * the buses.
	 */
	public void sendCommandRequest(CommandRequest cmd, CommandOriginator originator) {
            if ( !isConnected ) {
                throw new RuntimeException("A connection to the buses has not been established yet.");
            }
            
            if ( lockLevelService != null ) {
                // Looking for a potential held lock
                String destination = BusMessagingLayer.parseDestination(cmd.getDestination());
                AgentLock lock = lockLevelService.getLockForAgent(destination);
                int desiredLevel = lockLevelService.getLevelForAgent(destination);
                cmd.setLockAndLevel(lock, desiredLevel);
            }
            
            cmd.setOriginAgentInfo(agentInfo);
            layer.sendCommand(cmd, originator);
	}

	/**
	 * Sends a Command Reply on the Command Bus. This message will be received
	 * by the CommandOriginator and the list of CommandMessageListener The reply
	 * can be a Nack, Ack, Error or Result
	 * 
	 * @param reply
	 *            The CommandReply to be sent on the Command bus
	 */
	public void sendCommandReply(CommandReply reply) {
            if ( !isConnected ) {
                throw new RuntimeException("A connection to the buses has not been established yet.");
            }
		reply.setOriginAgentInfo(agentInfo);
                layer.reply(reply);
	}

	/**
	 * Defines the component able to execute an incoming command
	 * 
	 * @param executor
	 */
	public void setCommandExecutor(CommandExecutor executor) {
		layer.setCommandExecutor(executor);
	}
        
        public void setClusterDeserializationErrorHandler(ClusterDeserializationErrorHandler h) {
            layer.setClusterDeserializationErrorHandler(h);
        }

	static List<AgentMessagingLayer> getMessagingAccesses() {
		return msgAccesses;
	}

	static void printMessagingAccessInfo(AgentMessagingLayer msgAccess) {
		System.out.println("MessagingAccess " + msgAccess.agentInfo.getName());
		BusApplicationLayer layer = msgAccess.getApplicationLayer();
		System.out.println("BusApplicationLayer " + layer);
		BusMessagingLayer messagingLayer = layer.getBusMessagingLayer();
		System.out.println("BusMessagingLayer " + messagingLayer);
		Set<String> localAgents = messagingLayer.getRegisteredLocalAgents();
		System.out.println("Local Agents " + localAgents.size());
		for (String agent : localAgents) {
			System.out.println("\t" + agent);
		}
	}

}
