package org.lsst.ccs.messaging;

import java.io.Closeable;
import java.io.IOException;
import java.util.Set;
import org.lsst.ccs.bus.definition.Bus;

/**
  * Interface defining the access to the Transport Layer of Buses
  * @author bamade
 * TODO: create an accept URI
 *
  */
public interface BusMessagingLayer extends Closeable, MessagingLayer {

    /**
     * utility method: parse the destination string in Commands.
     * extracts the agent name from a complete destination
     * @param destination the complete destination (e.g : "sft/carousel")
     * @return the agent name (e.g : "sft")
     */
    static String parseDestination(String destination) {
            if (destination.contains("/")) {
                destination = destination.substring(0, destination.indexOf("/"));
            }
        return destination;
    }

    /**
     * Creates  low level communication entry points for a subsystem.
     * Abstract vision is that there is a different entry point for each bus/topic.
     *
     * @param agentName name of the sending/receiving point as will be known by transport
     * null or empty string  means that the local agent will receive all messages
     * (whatever the destination of the message is  : "anonymous agent")
     * if an agent with same name is already locally registered for these buses nothing happens (the call is
     * idempotent), but if another agent has the same name on the network a <TT>DuplicateBusNameException</TT>
     * may be fired (this is an optional behaviour) but this exception is reported only to the corresponding
     * <TT>BusMembershipListener</TT>
     *
     * @param buses list of  buses we want to connect to, if empty connects to all buses
     * @throws IOException if connection impossible,
     */
    public void register(String agentName, Bus... buses) throws IOException, DuplicateAgentNameException;

    /**
     * close  entry points for  a bus for an agent.
     * calls should be idempotent.
     * @param agentName (if empty voids the "anonymous" agent capabilities
     * @param buses if empty all registered buses for the agent will be closed
     * @throws IllegalArgumentException if you do not "own" the agent corresponding to the name.
     */
    public void closeFor(String agentName, Bus... buses);

    /**
     * Get the list of local agents currently connected to the buses.
     * @param buses The buses for which we are seeking the connected agents.
     *              If empty it will report agents connected to all buses.
     * @return      The List of registered buses.
     * @throws IOException 
     */
    public Set<String> getRegisteredLocalAgents(Bus... buses);

    /**
     * Get the list of local agents currently connected to all buses.
     * @return      The List of registered buses.
     * @throws IOException 
     */
    public default Set<String> getRegisteredLocalAgents() {
        return getRegisteredLocalAgents(Bus.values());
    }
    
    /**
     * Close the Transport Layer. Subsequent method calls will have no effect.
     * @throws IOException
     */
    @Override
    public void close() throws IOException ;

    /**
     * Sets up callback configuration for a topic and subsystem.
     * There could be multiple <TT>Forwarder</TT> for a given bus.
     * <B>All callbacks are supposed to be  multithreaded</B>
     * @param agentName if empty adds a forwarder to "anonymous" agent that receives all messages
     * @param forwarder code that handles the incoming messages
     * @param buses if empty the forwarder listens to all buses.
     * @throws IllegalArgumentException if the subsystem is not registered to one of the buses
     * or if forwarder is null
     */
    public void addMessageListener(String agentName, BusMessageForwarder forwarder, Bus... buses) ;

    /**
     * Removes a message listener from the forwarder list.
     * @param agentName
     * @param forwarder
     * @param buses if empty forwarder is removed from all buses
     * @throws IllegalArgumentException   if forwarder is null
     * (but no exception if the subsystem is not registered, or forwarder not present)
     */
    public void removeMessageListener(String agentName, BusMessageForwarder forwarder, Bus... buses) ;
    
}
