package org.lsst.ccs.messaging;

import org.lsst.ccs.bus.data.AgentInfo;

/**
 * Agents that implement this interface are notified when an agent is connecting
 * or has disconnected on the status bus
 * 
 * @author emarin
 */
public interface AgentPresenceListener {
    
    /**
     * Indicates that the agent represented by {@code agent} is present on the
     * buses. 
     * 
     * This method is called when the first messages for a given agent is received.
     * In the case of an agent starting up the fist messages is sent when the
     * agent is in PhaseState::INITIALIZING state. At this point not all services
     * are guaranteed to have completely started. This method should be used only
     * to receive a notification that an agent is connecting.
     * For a guarantee that an agent information is fully present, please use the
     * {@code #connected} methods. 
     * 
     * The implementation should return immediately : any blocking operations
     * should be scheduled on a separate thread.
     * 
     * @deprecated user {@code connecting(AgentInfo...)} instead.
     * @param agent 
     */
    @Deprecated
    public default void connecting(AgentInfo agent) {
        
    }
    
    /**
     * Indicates that the list of agents represented by {@code agents} is present on the
     * buses.
     * 
     * This method is invoked when the listener is added with all the agents that
     * are already connected.
     * 
     * At this point not all services on the connecting agents 
     * are guaranteed to have completely started. This method should be used only
     * to receive a notification that an agent is connecting.
     * For a guarantee that an agent information is fully present, please use the
     * {@code #connected} methods. 
     * 
     * The implementation should return immediately : any blocking operations
     * should be scheduled on a separate thread.
     * 
     * @param agents
     */
    public default void connecting(AgentInfo... agents) {
        for ( AgentInfo a : agents) {
            connecting(a);
        }
    }

    /**
     * Indicates that the agent represented by {@code agent} has left the buses.
     * 
     * The implementation should return immediately : any blocking operations
     * should be scheduled on a separate thread.
     * @param agent 
     */
    public default void disconnecting(AgentInfo agent) {
        
    }
    
    
    /**
     * Indicates that the list of agents represented by {@code agents} is present on the
     * buses.
     * 
     * This method is invoked when the listener is added with all the agents that
     * are already connected.
     * 
     * When this method is invoked the agent is in PhaseState::OPERATIONAL state
     * and all its services are guaranteed to have been properly initialized.
     * 
     * The implementation should return immediately : any blocking operations
     * should be scheduled on a separate thread.
     * 
     * @param agents
     */
    public default void connected(AgentInfo... agents) {
    }
    
    
}
