package org.lsst.ccs.messaging;

import java.util.*;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.messages.StatusHeartBeat;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.states.PhaseState;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Tracks agent connection and disconnection on the buses.
 *
 * @author emarin
 */
public class AgentPresenceManager implements StatusMessageListener {

    private final Object agentsLock = new Object();
    final CopyOnWriteArrayList<AgentPresenceListener> listAPL = new CopyOnWriteArrayList<>();
    private final Map<AgentInfo, AgentPresenceManager.TimeoutTask> mapAgents = new ConcurrentHashMap<>();
    private final List<AgentInfo> fullyConnectedAgents = new CopyOnWriteArrayList<>();
    private static final int SUSPICION_LENGTH = 10;
    private final Timer timer = new Timer(true);
    protected static Logger log = Logger.getLogger("org.lsst.ccs.messaging.agentpresencemanager");
    /** Lock used by clients waiting for a particular Agent to be fully connected. */
    private final ReentrantLock agentConnectionWaitLock = new ReentrantLock(true);
    /** List of clients waiting for a particular Agent to be fully connected. */
    private volatile LinkedHashMap<Condition,String> agentConnectionWaitList;

    /** Lock used by clients waiting for a particular Agent to be disconnected. */
    private final ReentrantLock agentDisconnectionWaitLock = new ReentrantLock(true);
    /** List of clients waiting for a particular Agent to be disconnected. */
    private volatile LinkedHashMap<Condition,String> agentDisconnectionWaitList;


    //Queue for processing listener's notifications.
    private final BlockingQueue<Runnable> notifications = new ArrayBlockingQueue<>(10);
    
    public enum AgentPresenceState {
        CONNECTING, CONNECTED, DISCONNECTING;
    }
        
    private static boolean hasStateChanged(AgentPresenceState oldState, AgentPresenceState newState) {
        if (oldState == newState) {
            return false;
        }
        if ( oldState.ordinal() < newState.ordinal() ) {
            return true;
        }
        throw new RuntimeException("Illegal AgentPresenceState Transition : "+oldState+" -> "+newState);
    }
    
    public AgentPresenceManager() {		
        ExecutorService queueExecutor = Executors.newSingleThreadExecutor(r -> {
            Thread t = new Thread(r, "Connection/Disconnection queue");
            t.setDaemon(true);
            return t;
        });		
        queueExecutor.submit(()->checkQueue());		
    }		
    
    /**
     * This method is invoked when the agent shuts down 
     */
    void disconnect() {
        synchronized (agentsLock) {
            for ( TimerTask task : mapAgents.values() ) {
                if ( task != null ) {
                    task.cancel();
                }
            }
            mapAgents.clear();
            fullyConnectedAgents.clear();
        }        
    }
    		   
    class TimeoutTask extends TimerTask {
        
        private final AgentInfo agent;
        private final int broadcastPeriod;
        private final AgentPresenceState state;
        
        TimeoutTask(AgentInfo agent, int broadcastPeriod, AgentPresenceState state) {
            this.agent = agent;
            this.broadcastPeriod = broadcastPeriod;
            this.state = state;
        }

        int getBroadcastPeriod() {
            return broadcastPeriod;
        }

        @Override
        public void run() {
            removeAgent(agent);
        }
        
        AgentPresenceState getAgentPresenceState() {
            return state;
        }
    }

    private void checkQueue() {
        try {
            while (true) {
                notifications.take().run();
            }
        } catch (InterruptedException e) {
            throw new RuntimeException("Failed when draining Connection/Disconnection queue.", e);
        }
    }
    
    
    @Override
    public void onStatusMessage(StatusMessage s) {

        AgentInfo a = (s.getOriginAgentInfo());
        if (a == null) return;

        int broadCastPeriod = -1;
        if (s instanceof StatusHeartBeat) {
            StatusHeartBeat hb = (StatusHeartBeat) s;
            broadCastPeriod = hb.getStatusBroadcastPeriod();
        }
        if (s.getState().isInState(PhaseState.OFF_LINE)) {
            log.debug("remove agent on status end");
            removeAgent(a);
        } else if (s.getState().isInState(PhaseState.INITIALIZING) ){
            updateAgent(a, broadCastPeriod, AgentPresenceState.CONNECTING);
        } else {
            updateAgent(a, broadCastPeriod, AgentPresenceState.CONNECTED);            
        }
    }

    /** Called on non-OFF_LINE agent status message. */
    private void updateAgent(AgentInfo a, int broadcastPeriod, AgentPresenceState state) {
        ConnectionDisconnectionNotification notification = null;     
        synchronized (agentsLock) {
            AgentPresenceManager.TimeoutTask task = mapAgents.get(a);
            if (task != null) { // Agent already known to the map : its timeouttask is updated
                task.cancel();
                if (broadcastPeriod == -1) {
                    broadcastPeriod = task.getBroadcastPeriod();
                }
                AgentPresenceState oldState = task.getAgentPresenceState();
                boolean hasStateChanged = false;
                try {
                    hasStateChanged = hasStateChanged(oldState, state);
                } catch (RuntimeException e) {
                    log.warning(e.getMessage());
                }
                AgentPresenceState newState = hasStateChanged ? state : oldState;
                task = new AgentPresenceManager.TimeoutTask(a, broadcastPeriod, newState);
                mapAgents.put(a, task);
                if (hasStateChanged) {
                    //This is an update to the current state to trigger a "connected" notification
                    notification = new ConnectionDisconnectionNotification(a, newState);
                }
            } else { // New agent
                if (broadcastPeriod == -1) {
                    broadcastPeriod = 10;
                }
                task = new AgentPresenceManager.TimeoutTask(a, broadcastPeriod, state);
                mapAgents.put(a, task);
                //This is a brand new notification and the Agent can be either
                //CONNECTED or CONNECTING
                notification = new ConnectionDisconnectionNotification(a, state, true);
            }
            if ( notification != null ) {
                notifications.offer(notification);                
            }
            timer.schedule(task, SUSPICION_LENGTH * 1000 * broadcastPeriod);
        }    
        log.debug("reset timer for agent " + a.getName() + " to " + broadcastPeriod);
    }
    

    /** Called to remove agent. */
    private void removeAgent(AgentInfo agent) {
        ConnectionDisconnectionNotification notification = null;     
        log.debug("removing agent " + agent.getName());
        AgentPresenceManager.TimeoutTask t = null;
        synchronized (agentsLock) {
            t = mapAgents.remove(agent);
            if (t != null) {
                t.cancel();
                notification = new ConnectionDisconnectionNotification(agent, AgentPresenceState.DISCONNECTING);
                notifications.offer(notification);
            }
        }
        if (t == null) {
            log.debug("removing agent with null timer");
        }
    }
    
    /**
     *
     * @return The list with the currently connected Agents
     */
    public List<AgentInfo> listConnectedAgents() {
        return new ArrayList<>(fullyConnectedAgents);
    }

    /**
     * This method is called by transport layers that provide disconnection
     * suspicion information
     *
     * @param agentName
     */
    void disconnecting(String agentName) {
        AgentInfo agentToRemove = null;
        synchronized (agentsLock) {
            for (AgentInfo a : mapAgents.keySet()) {
                if (a.getName().equals(agentName)) {
                    agentToRemove = a;
                    break;
                }
            }
        }
        if (agentToRemove != null) {
            removeAgent(agentToRemove);
        }
    }

    public void addAgentPresenceListener(AgentPresenceListener l) {
        synchronized (agentsLock) {
            l.connecting(mapAgents.keySet().toArray(new AgentInfo[0]));
            l.connected(fullyConnectedAgents.toArray(new AgentInfo[0]));
            listAPL.add(l);
        }
    }

    public void removeAgentPresenceListener(AgentPresenceListener l) {
        listAPL.remove(l);
    }

    /**
     * This method returns true as soon as the AgentPresenceManager is aware
     * of the existence of the given agent by name.
     * At this point we don't guarantee that the corresponding agent is ready
     * to receive command requests as it might still be INITIALIZING.
     * To be sure that an agent is fully connected, OPERATIONAL and ready to
     * receive message use method {@code #isAgentConnected(String)} instead.
     * 
     * @param agentName The name of the agent
     * @return   true if the AgentPresenceManager is aware of the existence of this agent.
     */
    public boolean agentExists(String agentName) {
        List<AgentInfo> existingAgents = new ArrayList<>(mapAgents.keySet());
        for (AgentInfo a : existingAgents) {
            if (a.getName().equals(agentName)) {
                return true;
            }
        }
        return false;
    }

    /**
     * This method returns true when the given agent is fully connected, OPERATIONAL
     * and ready to receive commands.
     * 
     * @param agentName The name of the Agent.
     * @return     true when the given agent is fully connected.
     */
    public boolean isAgentConnected(String agentName) {
        synchronized(fullyConnectedAgents) {
            for (AgentInfo a : fullyConnectedAgents) {
                if (a.getName().equals(agentName)) {
                    return true;
                }
            }
            return false;
        }
    }

    /**
     * Wait for an Agent to be fully connected: all the listeners have been
     * notified on the connected method.
     * @param name The name of the Agent
     * @param timeout The time to wait for the agent to become available.
     * @param unit The TimeUnit for the timeout.
     * @return True if the Agent is fully connected within the provided timeout, false otherwise.
     * @throws InterruptedException 
     */
    public final boolean waitForAgent(String name, long timeout, TimeUnit unit) throws InterruptedException {
        long deadline = TimeUnit.MILLISECONDS.convert(timeout, unit) + System.currentTimeMillis();
        if (!agentConnectionWaitLock.tryLock(timeout, unit)) return false;
        try {
            if ( isAgentConnected(name) ) {
                return true;
            }
            Condition condition = agentConnectionWaitLock.newCondition();
            if (agentConnectionWaitList == null) agentConnectionWaitList = new LinkedHashMap<>(4);
            agentConnectionWaitList.put(condition, name);
            while (agentConnectionWaitList != null && agentConnectionWaitList.containsKey(condition)) {
                timeout = deadline - System.currentTimeMillis();
                if (!condition.await(timeout, TimeUnit.MILLISECONDS)) return false;
            }
        } finally {
            agentConnectionWaitLock.unlock();
        }
        return true;
    }
    
    /**
     * Wait for an Agent to be fully disconnected: all the listeners have been
     * notified on the disconnected method.
     * @param name The name of the Agent
     * @param timeout The time to wait for the agent to become available.
     * @param unit The TimeUnit for the timeout.
     * @return True if the Agent is disconnected from the buses within the provided timeout, false otherwise.
     * @throws InterruptedException 
     */
    public final boolean waitForAgentDisconnection(String name, long timeout, TimeUnit unit) throws InterruptedException {
        long deadline = TimeUnit.MILLISECONDS.convert(timeout, unit) + System.currentTimeMillis();
        if (!agentDisconnectionWaitLock.tryLock(timeout, unit)) return false;
        try {
            if ( ! isAgentConnected(name) ) {
                return true;
            }
            Condition condition = agentDisconnectionWaitLock.newCondition();
            if (agentDisconnectionWaitList == null) agentDisconnectionWaitList = new LinkedHashMap<>(4);
            agentDisconnectionWaitList.put(condition, name);
            while (agentDisconnectionWaitList != null && agentDisconnectionWaitList.containsKey(condition)) {
                timeout = deadline - System.currentTimeMillis();
                if (!condition.await(timeout, TimeUnit.MILLISECONDS)) return false;
            }
        } finally {
            agentDisconnectionWaitLock.unlock();
        }
        return true;
    }



    private void agentConnectionWaitNotify(final String name) {
        if (agentConnectionWaitList != null) {
            try {
                agentConnectionWaitLock.lockInterruptibly();
                if (agentConnectionWaitList == null) {
                    return;
                }
                Iterator<Map.Entry<Condition, String>> it = agentConnectionWaitList.entrySet().iterator();
                while (it.hasNext()) {
                    Map.Entry<Condition, String> entry = it.next();
                    String agentWaitName = entry.getValue();
                    if (name.equals(agentWaitName)) {
                        entry.getKey().signal();
                        it.remove();
                    }
                }
                if (agentConnectionWaitList.isEmpty()) {
                    agentConnectionWaitList = null;
                }
            } catch (InterruptedException x) {
            } finally {
                agentConnectionWaitLock.unlock();
            }
        }
    }
 
    private void agentDisconnectionWaitNotify(final String name) {
        if (agentDisconnectionWaitList != null) {
            try {
                agentDisconnectionWaitLock.lockInterruptibly();
                if (agentDisconnectionWaitList == null) {
                    return;
                }
                Iterator<Map.Entry<Condition, String>> it = agentDisconnectionWaitList.entrySet().iterator();
                while (it.hasNext()) {
                    Map.Entry<Condition, String> entry = it.next();
                    String agentWaitName = entry.getValue();
                    if (name.equals(agentWaitName)) {
                        entry.getKey().signal();
                        it.remove();
                    }
                }
                if (agentDisconnectionWaitList.isEmpty()) {
                    agentDisconnectionWaitList = null;
                }
            } catch (InterruptedException x) {
            } finally {
                agentDisconnectionWaitLock.unlock();
            }
        }
    }

    private class ConnectionDisconnectionNotification implements Runnable {
        private final AgentInfo agentInfo;
        private final AgentPresenceState state;
        private final boolean isFirstNotification;
        
        ConnectionDisconnectionNotification(AgentInfo agentInfo, AgentPresenceState state) {
            this(agentInfo, state, false);
        }

        ConnectionDisconnectionNotification(AgentInfo agentInfo, AgentPresenceState state, boolean isFirstNotification) {
            this.agentInfo = agentInfo;
            this.state = state;
            this.isFirstNotification = isFirstNotification;
        }
        
        
        @Override
        public void run() {
            //Notification for DISCONNECTING agents
            if (state == AgentPresenceState.DISCONNECTING) {
                fullyConnectedAgents.remove(agentInfo);
                for (AgentPresenceListener l : listAPL) {
                    l.disconnecting(agentInfo);
                }
                agentDisconnectionWaitNotify(agentInfo.getName());
            }
            //Notification for CONNECTING agents
            if (isFirstNotification) {
                AgentInfo[] agents = new AgentInfo[]{agentInfo};
                for (AgentPresenceListener l : listAPL) {
                    l.connecting(agents);
                }
            }
            //Notification for CONNECTED agents
            if (state == AgentPresenceState.CONNECTED) {
                for (AgentPresenceListener l : listAPL) {
                    l.connected(agentInfo);
                }
                fullyConnectedAgents.add(agentInfo);
                agentConnectionWaitNotify(agentInfo.getName());
            }
        }
        
    }
    
}
