package org.lsst.ccs.messaging;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.NoSuchElementException;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.messages.StatusMessage;

import org.lsst.ccs.utilities.logging.Logger;

//This service will listen to the status bus and keep track in memory of selected status data. 
//
//It should be configurable: 
//- list of status data to record 
//- for some, keep some history for a defined duration 
//- for some, keep track of min, max, average, and other relevant aggregate data, for a defined window 
//
//It could be used 
//- by the FITS image writer to include data in header/tables 
//- by the MCM 
// TODO configure from regular expression.
// TODO keep some values for a number of samples and not a duration
// TODO get a key/value map of all "last" values, or all "average" values.
/**
 * A <TT>StatusAggregator</TT> listens to the status bus and keeps track in
 * memory of selected status data.
 *
 * <p>
 * It will keep at least the last value of each status data it is configured to
 * monitor.
 * </p>
 * <p>
 * It can also keep a history during a given time, and accumulate statistics
 * over a given time.
 * </p>
 * <p/>
 * <p>
 * The StatusAggregator should be registered on the bus, with something like <br/>
 * <tt>MessagingAccess.getInstance().forSubsystem("archon-sl").addStatusListener(sa);</tt>
 * <br/>
 * note the specific name so that the listener receives status messages also
 * from its host subsystem, if it is useful.
 * </p>
 * <p>
 * It should also be configured to monitor and store status data using the
 * <tt>setAggregate</tt> or <tt>setAggregatePattern</tt> methods for each
 * relevant data name.
 * </p>
 *
 * @author aubourg
 *
 */
public class StatusAggregator implements StatusMessageListener {

    Map<String, StatusAggregateConfig> config = new ConcurrentHashMap<String, StatusAggregateConfig>();
    Map<String, StatusAggregateConfig> configViaVars = new ConcurrentHashMap<String, StatusAggregateConfig>();

    Map<String, StatusAggregateConfig> patternConfig = new ConcurrentHashMap<String, StatusAggregateConfig>();

    Map<String, ConcurrentLinkedQueue<TimedValue>> history = new ConcurrentHashMap<String, ConcurrentLinkedQueue<TimedValue>>();
    Map<String, TimedValue> last = new ConcurrentHashMap<String, TimedValue>();
    Map<String, TimedValueStats> stats = new ConcurrentHashMap<String, TimedValueStats>();

    Map<String, String> variables = new ConcurrentHashMap<String, String>();

    // List<Pattern> patterns = new ArrayList<Pattern>();
    private final Logger log = Logger.getLogger("org.lsst.ccs.bus");

    public StatusAggregator() {
    }

    /**
     * Configures the <tt>StatusAgregator</tt> to monitor a given data.
     *
     * @param key
     * the name ("subsystem/key") of the data to be monitored
     *
     * @param historyDuration
     * duration (in ms) for which to keep history. -1 not to keep
     * history.
     * @param aggregateWindow
     * duration (in ms) for which to compute aggregates and stats. -1
     * not to compute.
     */
    public void setAggregate(String key, int historyDuration,
            int aggregateWindow) {
        config.put(key, new StatusAggregateConfig(key, historyDuration,
                aggregateWindow));
        if (historyDuration > 0 || aggregateWindow > 0) {
            history.put(key,
                    new ConcurrentLinkedQueue<StatusAggregator.TimedValue>());
        }

        if (aggregateWindow > 0) {
            stats.put(key, new TimedValueStats());
        }

    }

    /**
     * Configures the <tt>StatusAgregator</tt> to monitor all the data which
     * name matches a regex pattern.
     *
     * @param pattern
     * the pattern that the data ("subsystem/key") to be monitored
     * must match.
     *
     * @param historyDuration
     * duration (in ms) for which to keep history. -1 not to keep
     * history.
     * @param aggregateWindow
     * duration (in ms) for which to compute aggregates and stats. -1
     * not to compute.
     */
    public void setAggregatePattern(String pattern, int historyDuration,
            int aggregateWindow) {
        setAggregatePattern(Pattern.compile(pattern), historyDuration,
                aggregateWindow);
    }

    /**
     * Configures the <tt>StatusAgregator</tt> to monitor all the data which
     * name matches a regex pattern.
     *
     * @param pattern
     * the pattern that the data ("subsystem/key") to be monitored
     * must match.
     *
     * @param historyDuration
     * duration (in ms) for which to keep history. -1 not to keep
     * history.
     * @param aggregateWindow
     * duration (in ms) for which to compute aggregates and stats. -1
     * not to compute.
     */
    public void setAggregatePattern(Pattern pattern, int historyDuration,
            int aggregateWindow) {
        StatusAggregateConfig c = new StatusAggregateConfig(pattern,
                historyDuration, aggregateWindow);
        patternConfig.put(c.name, c);
    }

    /**
     * stop monitoring a given data.
     *
     * @param key
     * the name ("subsystem/key") of the data to be cleared
     *
     */
    public void clearAggregate(String key) {
        config.remove(key);
        history.remove(key);
        stats.remove(key);
        last.remove(key);
    }

    /**
     * set a variable value. The variable can be used as $var in configuration sources
     *
     * @param variable
     * the variable name
     * @param value
     * the variable value
     *
     */
    public void setVariable(String variable, String value) {
        variables.put(variable, value);
        updVars();
    }

     /**
      * set variable values. The variables can be used as $var in configuration
      * sources
      * 
      * @param values
      *            a name-value map
      * 
      */
    public void setVariables(Map<String, String> values) {
	variables.putAll(values);
	updVars();
    }
    
    
    Pattern varRx = Pattern.compile("\\$([a-zA-Z0-9]+)");

    public void updVars() {
        configViaVars.clear();
        config.entrySet()
                .stream()
                .filter(p -> p.getKey().startsWith("$"))
                .forEach(
                        p -> {
                            Matcher m = varRx.matcher(p.getKey());
                            m.find();
                            String vn = m.group(1);
                            String vv = variables.get(vn);
                            if (vv != null) {
                                configViaVars.put(
                                        p.getKey().replace("$" + vn, vv),
                                        p.getValue());
                            }
                        });
    }

    protected StatusAggregateConfig getConfig(String key) {
        StatusAggregateConfig c = config.get(key);
        if (c != null) {
            return c;
        }
        c = configViaVars.get(key);
        if (c != null) {
            return c;
        }

        for (Entry<String, StatusAggregateConfig> e : patternConfig.entrySet()) {
            c = e.getValue();
            if (!c.pattern.matcher(key).matches()) {
                continue;
            }
            c = new StatusAggregateConfig(key, c.historyDuration,
                    c.aggregateWindow);
            config.put(key, c);
            if (c.historyDuration > 0 || c.aggregateWindow > 0) {
                history.put(
                        key,
                        new ConcurrentLinkedQueue<StatusAggregator.TimedValue>());
            }

            if (c.aggregateWindow > 0) {
                stats.put(key, new TimedValueStats());
            }

            return c;
        }

        return null;
    }

    @Override
    public void onStatusMessage(StatusMessage msg) {

        Object encodedData = msg.getEncodedData();
        if (encodedData instanceof KeyValueDataList) {

            KeyValueDataList list = (KeyValueDataList) encodedData;
            String source = msg.getOriginAgentInfo().getName();

            for (KeyValueData data : list.getListOfKeyValueData()) {

                String key = data.getKey();
                Object value = data.getValue();
                long timeStamp = data.getTimestamp();

                log.debug("SA from " + source + " : " + key + "  > " + value);

                String k = source + '/' + key;

                StatusAggregateConfig c = getConfig(k);
                if (c == null) {
                    continue;
                }

                log.debug("SA get " + k + " " + value);

                last.put(k, new TimedValue(k, timeStamp, value));

                int kept = c.historyDuration > c.aggregateWindow ? c.historyDuration
                        : c.aggregateWindow;
                if (kept < 0) {
                    continue;
                }

                log.debug("keeping " + kept + " for " + k);

        // remove old samples from aggregate and history
                long aggregateBound = c.aggregateWindow < 0 ? Long.MAX_VALUE
                        : timeStamp - c.aggregateWindow;
        // long historyBound = c.historyDuration < 0 ? Long.MAX_VALUE :
                // timeStamp - c.historyDuration;
                long stopBound = timeStamp - kept; // assuming timeStamp is upper bound
                // : use current time ??

                ConcurrentLinkedQueue<TimedValue> q = history.get(k);

                if (q == null) {
                    q = new ConcurrentLinkedQueue<TimedValue>();
                    history.put(k, q);
                }

                synchronized (q) {
                    q.add(new TimedValue(k, timeStamp, value));
                    if (c.aggregateWindow > 0) {
                        TimedValueStats st = stats.get(k);
                        if (st == null) {
                            st = new TimedValueStats();
                            stats.put(k, st);
                        }
                        if (value instanceof Number) {
                            st.add(((Number) value).doubleValue());
                        }

                        // iterate on queue and remove from aggregate
                        if (aggregateBound > st.firstSample) {
                            long firstRemaining = Long.MAX_VALUE;
                            for (TimedValue v : q) {
                                if (v.tStamp < aggregateBound
                                        && v.tStamp >= st.firstSample
                                        && v.value instanceof Number) {
                                    st.remove(((Number) v.value).doubleValue(), q,
                                            aggregateBound);
                                } else {
                                    if (v.tStamp < firstRemaining
                                            && v.tStamp >= aggregateBound) {
                                        firstRemaining = v.tStamp;
                                    }
                                }
                            }
                            st.firstSample = firstRemaining;
                        }
                    }
                    // poll from history
                    while (true) {
                        TimedValue v = q.peek();
                        if (v != null && v.tStamp < stopBound) {
                    // System.out.println("removing from history " + v.name +
                            // " "
                            // + v.tStamp + " added " + timeStamp);
                            q.poll();
                        } else {
                            break;
                        }

                    }
                }
            }

        }
    }

    /**
     * returns the last value seen for a given key (which has to be monitored)
     *
     * @param key
     * @return the last value seen on the status bus.
     */
    public Object getLast(String key) {
        TimedValue v = last.get(key);
        return (v == null) ? null : v.value;
    }

    /**
     * returns the last value seen for a given key (which has to be monitored)
     *
     * @param key
     * @return the last value seen on the status bus, as a TimedValue.
     */
    public TimedValue getLastTV(String key) {
        return last.get(key);
    }

    /**
     * returns the average (over the configured duration) for a given key.
     *
     * The data has to be a number.
     *
     * @param key
     * @return the average, as a double
     */
    public double getAverage(String key) {
        TimedValueStats st = stats.get(key);
        if (st == null) {
            throw new NoSuchElementException();
        }
        if (st.n == 0) {
            return 0;
        }
        // throw new NoSuchElementException();

        return st.average();
    }

    // TODO distinguish between "not enough data" and
    // "not configured to compute this" ?
    /**
     * returns the stddev (over the configured duration) for a given key.
     *
     * The data has to be a number.
     *
     * @param key
     * @return the stddev, as a double
     */
    public double getStdDev(String key) {
        TimedValueStats st = stats.get(key);
        if (st == null) {
            throw new NoSuchElementException();
        }
        if (st.n == 0) {
            return 0;
        }
        // throw new NoSuchElementException();
        return st.stddev();
    }

    /**
     * returns the min (over the configured duration) for a given key.
     *
     * The data has to be a number.
     *
     * @param key
     * @return the min, as a double
     */
    public double getMin(String key) {
        TimedValueStats st = stats.get(key);
        if (st == null) {
            throw new NoSuchElementException();
        }
        if (st.n == 0) {
            return 0;
        }
        // throw new NoSuchElementException();
        return st.min;
    }

    /**
     * returns the max (over the configured duration) for a given key.
     *
     * The data has to be a number.
     *
     * @param key
     * @return the max, as a double
     */
    public double getMax(String key) {
        TimedValueStats st = stats.get(key);
        if (st == null) {
            throw new NoSuchElementException();
        }
        if (st.n == 0) {
            return 0;
        }
        // throw new NoSuchElementException();
        return st.max;
    }

    /**
     * returns the history (over the configured duration) for a given key.
     *
     *
     * @param key
     * @return the history as a list of TimedValue
     */
    public List<TimedValue> getHistory(String key) {

        // TODO : accept both variable-based value and effective value ?
        ConcurrentLinkedQueue<TimedValue> hh = history.get(key);
        if (hh == null) {
            return null;
        }

        ArrayList<TimedValue> l = new ArrayList<TimedValue>();
        StatusAggregateConfig cf = getConfig(key);
        long lastSample = last.get(key).tStamp;
        long firstSample = lastSample - cf.historyDuration;

        for (TimedValue tv : hh) {
            if (tv.tStamp >= firstSample) {
                l.add(tv);
            }
        }

        return l.size() == 0 ? null : l;
    }

    /**
     * returns statistics (over the configured duration) for a given key.
     *
     * Statistics contain min, max, average and stddev.
     *
     *
     * @param key
     * @return the Statistics object
     */
    public Statistics getStatistics(String key) {
        ConcurrentLinkedQueue<TimedValue> q = history.get(key);
        Statistics s;
        synchronized (q) {
            s = new Statistics(getMin(key), getMax(key), getAverage(key),
                    getStdDev(key));
        }
        return s;
    }

    /**
     * returns the last values of all parameters monitored
     *
     * @return a map of the last values, the key is the data name.
     */
    public Map<String, Object> getAllLast() {
        HashMap<String, Object> m = new HashMap<String, Object>();

        for (String k : last.keySet()) {
            m.put(k, getLast(k));
        }

        return m;
    }

    /**
     * returns the last values of all parameters monitored, with their
     * timestamp.
     *
     * @return a map of the last values as TimedValue, the key is the data name.
     */
    public Map<String, TimedValue> getAllLastTV() {
        HashMap<String, TimedValue> m = new HashMap<String, TimedValue>();

        for (String k : last.keySet()) {
            m.put(k, getLastTV(k));
        }

        return m;
    }

    /**
     * returns the statistics of all parameters monitored.
     *
     * @return a map of the Statistics, the key is the data name.
     */
    public Map<String, Statistics> getAllStatistics() {
        HashMap<String, Statistics> m = new HashMap<String, Statistics>();

        for (String k : stats.keySet()) {
            m.put(k, getStatistics(k));
        }

        return m;
    }

    /**
     * Internal class, unmutable data structure with min, max, average and
     * stddev.
     *
     * @author aubourg
     *
     */
    public static class Statistics {

        public Statistics(double min, double max, double average, double stddev) {
            super();
            this.min = min;
            this.max = max;
            this.average = average;
            this.stddev = stddev;
        }

        public double getMin() {
            return min;
        }

        public double getMax() {
            return max;
        }

        public double getAverage() {
            return average;
        }

        public double getStddev() {
            return stddev;
        }

        private double min, max, average, stddev;
    }

    private static class StatusAggregateConfig {

        public StatusAggregateConfig(String name, int historyDuration,
                int aggregateWindow) {
            this.name = name;
            this.historyDuration = historyDuration;
            this.aggregateWindow = aggregateWindow;
        }

        public StatusAggregateConfig(Pattern pattern, int historyDuration,
                int aggregateWindow) {
            this.pattern = pattern;
            this.name = pattern.toString();
            this.historyDuration = historyDuration;
            this.aggregateWindow = aggregateWindow;
        }

        public String name; // "subsystem/name", key in map
        public Pattern pattern; // if this is a regex config
        public int historyDuration; // ms, <0 : do not keep
        public int aggregateWindow; // ms, <0 : do not keep
    }

    /**
     * Internal class, unmutable data structure with name, timestamp and value.
     *
     * @author aubourg
     *
     */
    public static class TimedValue {

        public TimedValue(String name, long tStamp, Object value) {
            super();
            this.name = name;
            this.tStamp = tStamp;
            this.value = value;
        }

        public String getName() {
            return name;
        }

        public long gettStamp() {
            return tStamp;
        }

        public Object getValue() {
            return value;
        }

        private String name;
        private long tStamp;
        private Object value;
    }

    private static class TimedValueStats {

        public String name;
        public int n = 0;
        public double sum = 0;
        public double sum2 = 0;
        public double min = Double.MAX_VALUE;
        public double max = Double.MIN_NORMAL;

        public long firstSample;

        public double average() {
            return (n > 0) ? (sum / n) : 0;
        }

        public double stddev() {
            if (n == 0) {
                return 0;
            }
            return Math.sqrt(sum2 / n - sum * sum / (n * n));
        }

        public void add(double value) {
            sum += value;
            sum2 += value * value;
            n++;
            updateMinMaxIn(value);
        }

        public void remove(double value, ConcurrentLinkedQueue<TimedValue> h,
                long bound) {
            sum -= value;
            sum2 -= value * value;
            n--;
            updateMinMaxOut(value, h, bound);
        }

        private void updateMinMaxIn(double newValue) {
            if (newValue < min) {
                min = newValue;
            }
            if (newValue > max) {
                max = newValue;
            }
        }

        private void updateMinMaxOut(double removedValue,
                ConcurrentLinkedQueue<TimedValue> h, long bound) {
            if (removedValue <= min || removedValue >= max) {
                min = Double.MAX_VALUE;
                max = Double.MIN_NORMAL;
                for (TimedValue v : h) {
                    if (v.tStamp >= bound && v.value instanceof Number) {
                        double x = ((Number) v.value).doubleValue();
                        if (x > max) {
                            max = x;
                        }
                        if (x < min) {
                            min = x;
                        }
                    }
                }
            }
        }

    }

}
